//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <Candera/TextEngine/Async/TextRenderHandleController.h>
#include <Candera/TextEngine/Async/TextRenderDispatcher.h>

#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <FeatStd/Platform/CriticalSectionLocker.h>
#include <FeatStd/Platform/CriticalSection.h>
#endif
namespace Candera {
    namespace TextRendering {
        namespace Internal {
            static const UInt8 s_invalidIndex = 2;

            TextRenderHandleController::TextRenderHandleController(ITextValidationUser * owner) :m_validationHelper(0), m_owner(owner)
            {}



            TextRenderHandleController::~TextRenderHandleController()
            {
                AbortAll();
            }

            void TextRenderHandleController::SetHandle(UInt8 const idx, THandleBase handle)
            {
#ifdef FEATSTD_THREADSAFETY_ENABLED
                FeatStd::Internal::CriticalSectionLocker lock(TextRenderDispatcher::GetCsLock());
#endif
                FEATSTD_DEBUG_ASSERT(m_data[idx].m_state == TextRenderState::Preparing);
                m_data[idx].m_handle = handle;
                m_data[idx].m_state = TextRenderState::Waiting;
            }

            void TextRenderHandleController::SetFinishedHandle(UInt8 const idx)
            {
#ifdef FEATSTD_THREADSAFETY_ENABLED
                FeatStd::Internal::CriticalSectionLocker lock(TextRenderDispatcher::GetCsLock());
#endif
                FEATSTD_DEBUG_ASSERT(m_data[idx].m_state == TextRenderState::Preparing);
                FEATSTD_DEBUG_ASSERT(m_data[idx].m_handle.PointsToNull());
                m_data[idx].m_state = TextRenderState::Completed;
            }

            Candera::UInt8 TextRenderHandleController::GetRenderCompletedIndex(TArgsSharedPointer& result, bool& renderResult)
            {
#ifdef FEATSTD_THREADSAFETY_ENABLED
                FeatStd::Internal::CriticalSectionLocker lock(TextRenderDispatcher::GetCsLock());
#endif
                ThrowAwayMarkedSlots();
                UInt8 resultIndex = InvalidIndex();
                UInt8 idx = 0;
                UpdateAllRenderStates();
                while (idx < InvalidIndex()) {
                    if (m_data[idx].m_state == TextRenderState::Completed) // found a completed task
                    {
                        resultIndex = idx;
                        ++idx;
                        break;
                    }
                    ++idx;
                }
                while (idx < InvalidIndex())//throw away all the other completed render processes (too old)
                {
                    if (m_data[idx].m_state == TextRenderState::Completed) {
                        FreeIndex(idx);
                    }
                    ++idx;
                }
                if (resultIndex != InvalidIndex()) {
                    FEATSTD_DEBUG_ASSERT(m_data[resultIndex].m_state == TextRenderState::Completed);
                    result = m_data[resultIndex].m_args;
                    // handle == 0 --> sync call --> renderResult already set
                    if (!m_data[resultIndex].m_handle.PointsToNull()) {
                        renderResult = m_data[resultIndex].m_handle->GetResult();
                    }
                    m_data[resultIndex].m_state = TextRenderState::PostAnalysis;
                }
                return resultIndex;
            }

            void TextRenderHandleController::FreeIndex(UInt8 idx)
            {
#ifdef FEATSTD_THREADSAFETY_ENABLED
                FeatStd::Internal::CriticalSectionLocker lock(TextRenderDispatcher::GetCsLock());
#endif
                ResetIndex(idx);
            }

            Candera::UInt8 TextRenderHandleController::InvalidIndex() const
            {
                return s_invalidIndex;
            }

            Candera::UInt8 TextRenderHandleController::GetQueuedCount() const
            {
                UInt8 cnt = 0;
                for (UInt8 i = 0; i < InvalidIndex(); i++) {
                    if (m_data[i].m_state > TextRenderState::Free) {
                        cnt++;
                    }
                }
                return cnt;
            }

            Candera::UInt8 TextRenderHandleController::GetFreeArgs(TextRenderArgs::SharedPointer& resArgs)
            {
#ifdef FEATSTD_THREADSAFETY_ENABLED
                FeatStd::Internal::CriticalSectionLocker lock(TextRenderDispatcher::GetCsLock());
#endif
                ThrowAwayMarkedSlots();
                UInt8 freeIdx = GetFreeArgsIndex();
                if (freeIdx == InvalidIndex()) {
                    return freeIdx;
                }
                if (m_data[freeIdx].m_args.PointsToNull()) {
                    m_data[freeIdx].m_args = TextRenderArgs::Create();
                }
                resArgs = m_data[freeIdx].m_args;
                resArgs->SetValidationHelper(GetValidationHelper());
                return freeIdx;
            }

            UInt8 TextRenderHandleController::CheckForFinished()
            {
                UInt8 finished = 0;
                UInt8 i = 0;
                while (i < InvalidIndex()) {
                    if (CheckRenderState(i) == TextRenderState::Completed) {
                        finished++;
                    }
                    i++;
                }
                return finished;
            }

            void TextRenderHandleController::AbortAll()
            {
                for (UInt8 idx = 0; idx < InvalidIndex(); idx++) {
                    static_cast<void>(AbortTask(idx));
                }
            }


            void TextRenderHandleController::ThrowAwayMarkedSlots()
            {
                for (UInt8 idx = 0; idx < InvalidIndex(); idx++) {
                    if ((m_data[idx].m_throwAway) && (CheckRenderState(idx) == TextRenderState::Completed)) {
                        if (!m_data[idx].m_handle.PointsToNull()) {
                            m_data[idx].m_handle.Release();
                        }
                        m_data[idx].m_state = TextRenderState::Free;
                        m_data[idx].m_throwAway = false;
                    }
                }
            }

            bool TextRenderHandleController::AbortTask(UInt8 idx)
            {
                if (m_data[idx].m_handle.PointsToNull()) {
                    FEATSTD_DEBUG_ASSERT((m_data[idx].m_state == TextRenderState::Free) || (m_data[idx].m_state == TextRenderState::Preparing));
                    return true;
                }
                bool returnValue = m_data[idx].m_handle->Abort();
                if (returnValue)//abort were successful
                {
                    // clean up data
                    if (!m_data[idx].m_args.PointsToNull()) {
                        if (m_data[idx].m_args->GetValidationHelper() != 0) {
                            //set valid again --> cannot be called by render anymore
                            m_data[idx].m_args->GetValidationHelper()->SetValid();
                            // the valid one is also handled (because it is ignored)
                            m_data[idx].m_args->GetValidationHelper()->ConfirmValidHandled();
                        }
                    }
                    //free the index
                    ResetIndex(idx);
                }
                else {
                    if (CheckRenderState(idx) == TextRenderState::Completed) {
                        if (m_data[idx].m_args->GetValidationHelper() != 0) {
                            //set valid again --> cannot be called by render anymore
                            m_data[idx].m_args->GetValidationHelper()->SetValid();
                            // the valid one is also handled (because it is ignored)
                            m_data[idx].m_args->GetValidationHelper()->ConfirmValidHandled();
                        }
                        //free the index
                        ResetIndex(idx);
                    }
                    else {
                        m_data[idx].m_throwAway = true;
                    }

                }
                return returnValue;
            }


            Candera::UInt8 TextRenderHandleController::GetFreeArgsIndex()
            {
                UpdateAllRenderStates();
                // FEATSTD_DEBUG_ASSERT(CheckFreeIdxIsValidRequest());

                UInt8 resIdx = 0;
                UInt8 freeIdx = InvalidIndex();
                while (resIdx < InvalidIndex()) {
                    if (m_data[resIdx].m_state == TextRenderState::Waiting) {
                        break;
                    }
                    if (m_data[resIdx].m_state == TextRenderState::Free) {
                        freeIdx = resIdx;
                    }
                    ++resIdx;
                }

                if (resIdx != InvalidIndex())// Is a waiting state to be replaced
                {
                    FEATSTD_DEBUG_ASSERT(m_data[resIdx].m_state == TextRenderState::Waiting);
                    bool abortResult = AbortTask(resIdx);
                    if (!abortResult)//the waiting state has been changed during the checks
                    {
                        FEATSTD_DEBUG_ASSERT(freeIdx != InvalidIndex());
                        resIdx = freeIdx;
                    }
                }
                else {
                    resIdx = freeIdx;
                }
                if (resIdx == InvalidIndex()) {
                    // All tasks are Running or completed
                    // -> throw away a completed result
                    do {
                        FEATSTD_DEBUG_ASSERT(resIdx > 0);
                        --resIdx;
                        // All running? should not be possible (but nevertheless it should handle the case somehow)
                        // And its an overflow check thats why this check is how it is
                        if (resIdx >= InvalidIndex()) {
                            return InvalidIndex();
                        }
                    } while (m_data[resIdx].m_state != TextRenderState::Completed);
                    FEATSTD_DEBUG_ASSERT(m_data[resIdx].m_state == TextRenderState::Completed);
                    ResetIndex(resIdx);
                    resIdx = ShiftDataToNewestPosition(resIdx);
                }
                FEATSTD_DEBUG_ASSERT(m_data[resIdx].m_handle.PointsToNull() && m_data[resIdx].m_state == TextRenderState::Free);
                FEATSTD_LINT_NEXT_EXPRESSION(661, "Violates MISRA C++ 2008 Required Rule 5-0-16: code before this call already checks boundary")
                    m_data[resIdx].m_state = TextRenderState::Preparing;
                return resIdx;
            }

            void TextRenderHandleController::UpdateAllRenderStates()
            {
                for (UInt8 idx = 0; idx < InvalidIndex(); idx++) {
                    static_cast<void>(CheckRenderState(idx));
                }
            }

            Candera::UInt8 TextRenderHandleController::ShiftDataToNewestPosition(UInt8 index)
            {
                if ((index == 0) || (index == InvalidIndex())) {
                    return index;
                }
                Data tmpData;
                tmpData.MoveFrom(m_data[index]);
                while ((index - 1 >= 0) && (m_data[index - 1].m_state != TextRenderState::Free)) {
                    FEATSTD_DEBUG_ASSERT((m_data[index - 1].m_state != TextRenderState::PostAnalysis) ||
                                         (m_data[index - 1].m_state != TextRenderState::Preparing) ||
                                         (m_data[index - 1].m_state != TextRenderState::Waiting));
                    m_data[index].MoveFrom(m_data[index - 1]);
                    --index;
                }
                m_data[index].MoveFrom(tmpData);
                return index;
            }

            bool TextRenderHandleController::IsFreeIdxIsValidRequest() const
            {
                UInt8 completedCount = 0;
                UInt8 runningCount = 0;
                UInt8 waitingCount = 0;
                UInt8 idx = 0;
                while (idx < InvalidIndex()) {
                    if ((m_data[idx].m_state == TextRenderState::Preparing) ||
                        (m_data[idx].m_state == TextRenderState::PostAnalysis)) {
                        return false;
                    }
                    if (m_data[idx].m_state == TextRenderState::Completed) {
                        completedCount++;
                    }
                    if (m_data[idx].m_state == TextRenderState::Running) {
                        runningCount++;
                        if (runningCount >= InvalidIndex()) {
                            return false;
                        }
                    }
                    if (m_data[idx].m_state == TextRenderState::Waiting) {
                        waitingCount++;
                        if (waitingCount >= InvalidIndex()) {
                            return false;
                        }
                    }
                    idx++;
                }
                return completedCount != idx;
            }

            void TextRenderHandleController::ResetIndex(UInt8 idx)
            {
                if (idx >= InvalidIndex()) {
                    return;
                }
                if (!m_data[idx].m_handle.PointsToNull()) {
                    m_data[idx].m_handle.Release();
                }
                m_data[idx].m_state = TextRenderState::Free;
            }

            TextRenderState::Enum TextRenderHandleController::CheckRenderState(UInt8 idx)
            {
                TextRenderState::Enum currentState = m_data[idx].m_state;
                if ((currentState == TextRenderState::Waiting) || (currentState == TextRenderState::Running)) {
                    FEATSTD_DEBUG_ASSERT(!m_data[idx].m_handle.PointsToNull());
                    //Update state
                    FeatStd::AsyncRequestBase::State handleState = m_data[idx].m_handle->GetState();

                    switch (handleState) {
                    case FeatStd::AsyncRequestBase::Waiting:
                        currentState = TextRenderState::Waiting;
                        break;
                    case FeatStd::AsyncRequestBase::Running:
                        currentState = TextRenderState::Running;
                        break;
                    case FeatStd::AsyncRequestBase::Completed:
                        currentState = TextRenderState::Completed;
                        break;
                    case FeatStd::AsyncRequestBase::Aborted:
                        // Aborted should also free the slot
                        CANDERA_LINT_FALLTHROUGH()
                    case FeatStd::AsyncRequestBase::Invalid:
                        FreeIndex(idx);
                        currentState = TextRenderState::Free;
                        break;
                    }
                    m_data[idx].m_state = currentState;
                }
                FEATSTD_DEBUG_ASSERT(m_data[idx].m_state >= TextRenderState::Free);
                return m_data[idx].m_state;
            }
        }
    }
}
