//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef Candera_TextEngine_TextRenderDispatcher_h
#define Candera_TextEngine_TextRenderDispatcher_h

#include <Candera/Environment.h>
#include <FeatStd/Async/AsyncRequest.h>
#include <FeatStd/MemoryManagement/SharedPointer.h>
#include <Candera/System/EntityComponentSystem/EntitySystem.h>
#include <Candera/TextEngine/Async/TextRenderArgs.h>

//AsyncTextRenderDispatcher
#include <Candera/System/UpdateSystem/UpdateSystem.h>

//ThreadingTextRenderHandler
//#define CANDERA_TEXTENGINE_WORKER_THREAD_ENABLED
#ifdef FEATSTD_THREADSAFETY_ENABLED
namespace FeatStd {
    namespace Internal {
        class CriticalSection;
} }
#endif
namespace Candera {
    namespace TextRendering {
        typedef FeatStd::Internal::ResultCallbackAsyncRequest1<bool, TextRenderArgs::SharedPointer> TRenderProcess;
        typedef FeatStd::MemoryManagement::SharedPointer<TRenderProcess> TextRenderHandle;

        /** @addtogroup CanderaTextEngine
        *  @{
        */
        /**
        * @brief TextRenderDispatcher is used to dispatch a text into a dispatcher.
        * It contains a settings object which defines how the dispatcher should work.
        */
        class TextRenderDispatcher : public EntityComponentSystem::Entity {
        public:

            /**
            * @brief TextRenderSettings are settings which have direct influence on the textRendering process
            *   Current settings:
            *   Default text renderer: Sets a default renderer: The standard is set with CMake
            *                          or Target restrictions (No Threads available)
            *   Async loop count:      Sets the amount of render requests of the AsyncTextRenderer queue
            *                          --> Single Threaded asynchronous text rendering
            *                          which are handled in a single update call.
            */
            class TextRenderSettings {
            public:
                /**
                * Get the Settings object of text render process
                */
                static TextRenderSettings& GetInstance();

                /**
                * @return Get the default text renderer which is used when no other text renderer explicitly is specified.
                */
                Candera::TextRendering::TextRenderDispatcher * GetDefaultTextRenderer() const;

                /**
                * Sets the default text renderer which is used when no other text renderer explicitly is specified.
                */
                void SetDefaultTextRenderer(Candera::TextRendering::TextRenderDispatcher * val);

                /**
                * @return  Get the amount of entries which are processed in a single update call
                *          --> Only relevant for AsyncTextRenderer (SingleThreaded/Update-routine)
                */
                UInt8 GetAsyncLoopCount() const { return m_asyncLoopCount; }

                /**
                * Sets he amount of entries which are processed in a single update call
                *          --> Only relevant for AsyncTextRenderer (SingleThreaded/Update-routine)
                */
                void SetAsyncLoopCount(UInt8 const val);

                /**
                * Set the priority of the rendering thread
                * @param priority
                * @param callback the callback function to be used for setting priority. See FeatsStd::Internal::Thread for more info on callback function
                */
                bool SetPriority(FeatStd::Internal::ThreadPriority::Enum priority,
                    FeatStd::Internal::Thread::SetPriorityCallback callback);

            private:
                TextRenderSettings();
                FEATSTD_MAKE_CLASS_UNCOPYABLE(TextRenderSettings);

                TextRenderDispatcher * m_defaultTextRenderer;
                UInt8 m_asyncLoopCount;
            };

            virtual bool SetPriority(FeatStd::Internal::ThreadPriority::Enum priority, FeatStd::Internal::Thread::SetPriorityCallback callback)
            {
                FEATSTD_UNUSED2(priority, callback);
                return false;
            }

            /**
            * Get the Settings object of textrender process
            */
            static TextRenderSettings& GetTextRenderSettings();

#ifdef FEATSTD_THREADSAFETY_ENABLED
            static FeatStd::Internal::CriticalSection * GetCsLock();
#endif

            /**
            * Handles async render requests with priority
            * Current priorities (lower value == higher prio): 0, 1
            * Async request is standard 1
            * Sync requests which could be added to the async dispatcher queue would have prio 0
            */
            static TextRenderHandle RenderAsync(TRenderProcess::CallbackType callback, TextRenderArgs::SharedPointer& args, UInt32 const priority);

            /**
            * Handles async render requests with priority
            * Selects own render dispatcher
            * Current priorities (lower value == higher prio): 0, 1
            * Async request is standard 1
            * Sync requests which could be added to the async dispatcher queue would have prio 0
            */
            static TextRenderHandle RenderAsync(TextRenderDispatcher * dispatcher, TRenderProcess::CallbackType callback, TextRenderArgs::SharedPointer& args, UInt32 const priority);


        protected:
            /**
            * Returns the async dispatcher of the render object
            */
            virtual FeatStd::AsyncRequestDispatcher * GetDispatcher() = 0;

            /**
            * Waits until a specific render process has finished
            */
            virtual void WaitForHandleFinished(TextRenderHandle handle) = 0;

        };

        /** @} */ // end of CanderaTextEngine
    }// namespace TextRendering
}// namespace Candera
#endif // Candera_TextEngine_TextRenderDispatcher_h
