//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "TextRenderArgs.h"

#include <Candera/System/Diagnostics/Log.h>

#include <Candera/TextEngine/TextRenderContext.h>
#include <Candera/TextEngine/TextProperties.h>
#include <Candera/TextEngine/TextRenderContexts/GlyphDataContainer.h>

// TODO: refactor - TextNode2D in Candera Core part; not part of textengine
#include <Candera/Engine2D/Core/TextNode2D.h>
#include <Candera/Engine2D/Core/TextNodeRenderer/TextNodeRenderer.h>
#include <FeatStd/Util/FeatLimits.h>
#include <Candera/TextEngine/Internal/TextProcessProperties.h>


namespace Candera {
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaTextEngine);
    namespace TextRendering {
        class DummyCacheContext : public TextRenderContext {
        public:
            DummyCacheContext(Internal::TextProcessProperties::TextCache * glyphCache) : m_glyphCache(glyphCache) {}

            virtual const TextRect& GetClipRect() const override { 
                FEATSTD_SYNCED_STATIC_OBJECT(TextRect, s_rect);
                return s_rect;
            }
            virtual void Blit(Int16 /*x*/, Int16 /*y*/, const GlyphBitmap &/*glyph*/) override { }


            virtual GlyphCacheAccess* GetGlyphCacheAccess() const override
            {
                return m_glyphCache;
            }
        private:
            Internal::TextProcessProperties::TextCache * m_glyphCache;

        };

        bool TextRenderArgs::MeasureInit(TextNode2D& node, TextSize const& clientArea, LayoutingOptions layoutBase, TruncationToGlyphIteratorContext::SharedPointer& truncationContext)
        {
            Cleanup();
            m_textNode = &node;
            m_style = node.GetStyle();
            m_clientArea = clientArea;
            m_layoutingOptions = layoutBase;
            m_textProps = TextProperties(node.GetText().GetCString());
            m_textLength = static_cast<TextPosition>(node.GetText().GetCharCount());
            m_cacheContext = (node.GetTextNodeRenderer() != 0)?node.GetTextNodeRenderer()->GetMeasureReferenceContext():0;
            m_truncationContext = truncationContext;
            m_isAsyncCall = m_textNode->IsAsyncPreRenderEnabled();
            if (m_textVector.PointsToNull()) {
                m_textVector = GlyphDataContainer::Create();
            }
            return true;
        }




        bool TextRenderArgs::MeasureInit(SharedStyle::SharedPointer& style, 
                                         TextSize const& clientArea, 
                                         TextProperties& txtProps, 
                                         Internal::TextProcessProperties::TextCache* cacheContext, 
                                         LayoutingOptions layoutBase, 
                                         TruncationToGlyphIteratorContext::SharedPointer& truncationContext)
        {
            Cleanup();
            m_textNode = 0;
            m_style = style;
            m_clientArea = clientArea;
            m_layoutingOptions = layoutBase;
            m_textProps = txtProps;
            m_textLength = static_cast<TextPosition>(FeatStd::Internal::String::TLength(txtProps.GetTChar()));
            FEATSTD_DEBUG_ASSERT(m_cacheContext == 0);
            if (cacheContext != 0) {
                m_dummyCacheContext = TEXTENGINE_TRANSIENT_NEW(DummyCacheContext)(cacheContext);
                m_cacheContext = m_dummyCacheContext;
                if (m_cacheContext == 0) {
                    FEATSTD_LOG_ERROR("Error allocating glyph cache wrapper!");
                }
            }
            m_truncationContext = truncationContext;
            m_isAsyncCall = false;
            if (m_textVector.PointsToNull()) {
                m_textVector = GlyphDataContainer::Create();
            }
            return true;
        }

        bool TextRenderArgs::SetResults(TextRect const& layoutRect, TextRect const& textRect)
        {
            FEATSTD_DEBUG_ASSERT(m_textNode != 0);
            m_LayoutRectangle = layoutRect;
            m_textRectangle = textRect;
            if (m_textNode == 0) {
                return false;
            }
            m_textNode->SetPreparationState(TextNode2DRenderState::Finished);
            if (m_isAsyncCall) {
                if (m_validationHelper != 0) {
                    m_validationHelper->SetValid();
                }
                else {
                    m_textNode->TriggerValidUpdate();
                }
            }
            return true;
        }


        void TextRenderArgs::Cleanup()
        {
            if (m_dummyCacheContext != 0) {
                TEXTENGINE_FREE(m_dummyCacheContext);
                m_dummyCacheContext = 0;
            }
        }

        TextRenderArgs::TextRenderArgs() :
            m_validationHelper(),
            m_LayoutRectangle(TextRect::GetMin()),
            m_textRectangle(TextRect::GetMin()),
            m_textProps(""),
            m_clientArea(-1, -1),
            m_textLength(0),
            m_style(),
            m_textNode(0),
            m_cacheContext(0),
            m_dummyCacheContext(0),
            m_textVector(),
            m_truncationContext(),
            m_isAsyncCall(false)
        {

        }

        bool TextRenderArgs::DefaultInit(TextNode2D& node)
        {
            TruncationToGlyphIteratorContext::SharedPointer trunc;
            TextSize clientArea(-1, -1);
            return MeasureInit(node, clientArea, LayoutingOptions(), trunc);
        }

        TextRenderArgs::~TextRenderArgs()
        {
            Cleanup();
        }

    }
}
