//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_SIMPLEPROPERTYEX_H)
#define CANDERA_SIMPLEPROPERTYEX_H

#include <Candera/Environment.h>

namespace Candera {

    /**
     *  Extends SimpleProperty with a flag that is set whenever the property has changed.
     */
    template <typename T>
    class SimplePropertyEx {
        public:
            /**
             *  Constructor
             */
            SimplePropertyEx();

            /**
             *  Constructor
             *  @param value The value that is assigned.
             */
            SimplePropertyEx(const T& value);

            /**
             *  Destructor
             */
            ~SimplePropertyEx();

            /**
             *  Sets the value.
             *  @param value The value that is assigned.
             */
            void Set(const T& value);

            /**
             *  Gets the current value.
             *  @return The current value.
             */
            const T& Get() const;

            /**
             *  Retrieves whether the value has changed.
             *  @return true, if the value has changed, false, if not.
             */
            bool IsChanged() const;

            /**
             *  Sets the changed value.
             *  @param changed The value which is assigned.
             */
            void SetChanged(bool changed);

            operator const T() const;
            const T& operator()() const;

            const T& operator=(const T& value);

        private:
            T m_value;
            bool m_changed;

            FEATSTD_MAKE_CLASS_UNCOPYABLE(SimplePropertyEx);
    };

    template<typename T>
    SimplePropertyEx<T>::SimplePropertyEx() :
        m_value(T()),
        m_changed(true)
    {
    }

    template<typename T>
    SimplePropertyEx<T>::SimplePropertyEx(const T& value) :
        m_value(value),
        m_changed(true)
    {
    }

    template<typename T>
    SimplePropertyEx<T>::~SimplePropertyEx()
    {
    }

    template<typename T>
    void SimplePropertyEx<T>::Set(const T& value)
    {
        m_value = value;
        m_changed = true;
    }

    template<typename T>
    inline const T& SimplePropertyEx<T>::Get() const
    {
        return m_value;
    }

    template<typename T>
    inline bool SimplePropertyEx<T>::IsChanged() const
    {
        return m_changed;
    }

    template<typename T>
    inline void SimplePropertyEx<T>::SetChanged(bool changed)
    {
        m_changed = changed;
    }

    template<typename T>
    inline SimplePropertyEx<T>::operator const T() const
    {
        return Get();
    }
    
    template<typename T>
    inline const T& SimplePropertyEx<T>::operator()() const
    {
        return Get();
    }

    template<typename T>
    const T& SimplePropertyEx<T>::operator=(const T& value)
    {
        Set(value);
        return m_value;
    }

}   // namespace Candera

#endif  // CANDERA_SIMPLEPROPERTYEX_H
