//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_SHAREDPTRPROPERTY_H)
    #define CANDERA_SHAREDPTRPROPERTY_H

#include <Candera/Environment.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>

namespace Candera {
    
    /**
     *  @brief SharedPtrProperty is used for shared pointer type properties where the destruction of the value is handled by a SharedPointer.
     */
    template <typename T>
    class SharedPtrProperty {
        public:
            typedef MemoryManagement::SharedPointer<T> TPtr;

            /**
             *  Constructor
             */
            SharedPtrProperty();

            explicit SharedPtrProperty(T* value) :
                m_value(TPtr(value))
            {}

            /**
             *  Destructor
             */
            ~SharedPtrProperty();

            /**
             *  Sets the value.
             *  @param value The value that is assigned.
             */
            void Set(const TPtr& value);

            /**
             *  Gets the current value.
             *  @return The current value.
             */
            const T* Get() const;

            /**
             *  Gets the current value.
             *  @return The current value.
             */
            T* Get();

            operator T*();
            operator const T*() const;

            T* operator()();
            const T* operator()() const;

            T* operator=(T* value);

        private:
            TPtr m_value;

            FEATSTD_MAKE_CLASS_UNCOPYABLE(SharedPtrProperty);
    };

    template<typename T>
    SharedPtrProperty<T>::SharedPtrProperty() :
        m_value(TPtr())
    {
    }

    template<typename T>
    SharedPtrProperty<T>::~SharedPtrProperty()
    {
    }

    template<typename T>
    inline void SharedPtrProperty<T>::Set(const TPtr& value)
    {
        m_value = value;
    }

    template<typename T>
    inline const T* SharedPtrProperty<T>::Get() const
    {
        return m_value.GetPointerToSharedInstance();
    }

    template<typename T>
    inline T* SharedPtrProperty<T>::Get()
    {
        return m_value.GetPointerToSharedInstance();
    }

    template<typename T>
    inline SharedPtrProperty<T>::operator T*() 
    { 
        return Get(); 
    }
    
    template<typename T>
    inline SharedPtrProperty<T>::operator const T*() const 
    { 
        return Get(); 
    }

    template<typename T>
    inline T* SharedPtrProperty<T>::operator()() 
    { 
        return Get(); 
    }
    
    template<typename T>
    inline const T* SharedPtrProperty<T>::operator()() const 
    { 
        return Get(); 
    }

    template<typename T>
    T* SharedPtrProperty<T>::operator=(T* value) 
    { 
        Set(value); 
        return m_value.GetPointerToSharedInstance(); 
    }

}   // namespace Candera

#endif  // CANDERA_SHAREDPTRPROPERTY_H
