//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_OBSERVABLE_SIMPLEPROPERTY_H)
#define CANDERA_OBSERVABLE_SIMPLEPROPERTY_H

#include <Candera/Environment.h>
#include <Candera/System/Container/SingleLinkedList.h>

namespace Candera {

    class ObservableSimplePropertyListener {
    public:
        virtual void SimplePropertyChanged() = 0;
    };

    /**
     *  @brief ObservableSimpleProperty is used for value and reference type properties.
     *         Additionally it allows to specify listeners which can react on setting
     *         the contained property.
     */
    template <typename T>
    class ObservableSimpleProperty {
        public:
            /**
             *  Constructor
             */
            ObservableSimpleProperty();

            /**
             *  Constructor
             *  @param value The value that is assigned.
             */
            ObservableSimpleProperty(const T& value);

            /**
             *  Destructor
             */
            ~ObservableSimpleProperty();


            /**
             * Adds a listener that can react on setting the property.
             * @param listener The listener to be added.
             * @return True if the listener could be added, false in any other case.
             */
            bool AddListener(ObservableSimplePropertyListener* listener);

            /**
             * Removes a listener from the property.
             * @param listener The listener to be removed.
             * @return True if the listener could be removed, false in any other case.
             */
            bool RemoveListener(ObservableSimplePropertyListener* listener);

            /**
             *  Sets the value.
             *  @param value The value that is assigned.
             */
            void Set(const T& value);

            /**
             *  Gets the current value.
             *  @return The current value.
             */
            const T& Get() const;

            /**
             *  Gets the current value.
             *  @return The current value.
             */
            T& Get();

            operator T();
            operator const T() const;

            T& operator()();
            const T& operator()() const;

            const T& operator=(const T& value);

        private:
            T m_value;
            FeatStd::Internal::SingleLinkedList<ObservableSimplePropertyListener*> m_listeners;

            FEATSTD_MAKE_CLASS_UNCOPYABLE(ObservableSimpleProperty);
    };

    template<typename T>
    ObservableSimpleProperty<T>::ObservableSimpleProperty() :
        m_value(T())
    {
    }

    template<typename T>
    ObservableSimpleProperty<T>::ObservableSimpleProperty(const T& value) :
        m_value(value)
    {
    }

    template<typename T>
    ObservableSimpleProperty<T>::~ObservableSimpleProperty()
    {
        m_listeners.Clear();
    }

    template<typename T>
    inline void ObservableSimpleProperty<T>::Set(const T& value)
    {
        m_value = value;
        for (FeatStd::Internal::SingleLinkedList<ObservableSimplePropertyListener*>::Iterator it = m_listeners.Begin(); it != m_listeners.End(); ++it) {
            ObservableSimplePropertyListener* listener = *it;
            if (listener != 0) {
                listener->SimplePropertyChanged();
            }
        }
    }

    template<typename T>
    inline const T& ObservableSimpleProperty<T>::Get() const
    {
        return m_value;
    }

    template<typename T>
    inline T& ObservableSimpleProperty<T>::Get()
    {
        return m_value;
    }

    template<typename T>
    inline ObservableSimpleProperty<T>::operator T()
    {
        return Get();
    }
    
    template<typename T>
    inline ObservableSimpleProperty<T>::operator const T() const
    {
        return Get();
    }

    template<typename T>
    inline T& ObservableSimpleProperty<T>::operator()()
    {
        return Get();
    }
    
    template<typename T>
    inline const T& ObservableSimpleProperty<T>::operator()() const
    {
        return Get();
    }

    template<typename T>
    const T& ObservableSimpleProperty<T>::operator=(const T& value)
    {
        Set(value);
        return m_value;
    }

    template<typename T>
    bool ObservableSimpleProperty<T>::AddListener(ObservableSimplePropertyListener* listener)
    {
        return m_listeners.Append(listener);
    }

    template<typename T>
    bool ObservableSimpleProperty<T>::RemoveListener(ObservableSimplePropertyListener* listener)
    {
        return m_listeners.Remove(listener);
    }

}   // namespace Candera

#endif  // CANDERA_SIMPLEPROPERTY_H
