//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_DISPOSABLEPROPERTYEX_H)
#define CANDERA_DISPOSABLEPROPERTYEX_H

#include <Candera/Environment.h>
#include <Candera/System/MemoryManagement/Disposer.h>

namespace Candera {
    namespace Internal {
        struct TextBrushProperties;
    }

    /**
     *  Extends DisposableProperty with a flag that is set whenever the property has changed.
     */
    template <typename T>
    class DisposablePropertyEx {
        public:
            typedef void(*DisposerFn)(T);

            /**
             *  Constructor
             */
            DisposablePropertyEx();

            /**
             *  Destructor
             */
            ~DisposablePropertyEx();

            /**
             *  Sets the value and the disposer.
             *  @param value        The value that is assigned.
             *  @param disposerFn   The disposer function that is assigned.
             */
            void Set(const T& value, DisposerFn disposerFn);

            /**
             *  Gets the current value.
             *  @return The current value.
             */
            const T& Get() const;

            /**
             *  Retrieves whether the value has changed.
             *  @return true, if the value has changed, false, if not.
             */
            bool IsChanged() const;

            /**
             *  Sets the changed value.
             *  @param changed The value that is assigned.
             */
            void SetChanged(bool changed);

            operator const T() const;
            const T& operator()() const;

            DisposerFn GetDisposerFn() const { return m_disposerFn; }

        private:
            T m_value;
            DisposerFn m_disposerFn;
            bool m_changed;

            FEATSTD_MAKE_CLASS_UNCOPYABLE(DisposablePropertyEx);
    };

    template<typename T>
    DisposablePropertyEx<T>::DisposablePropertyEx() :
        m_value(T()),
        m_disposerFn(0),
        m_changed(true)
    {
    }

    template<typename T>
    DisposablePropertyEx<T>::~DisposablePropertyEx()
    {
        if (m_disposerFn != 0) {
            m_disposerFn(m_value);
        }
    }
    
    template<typename T>
    void DisposablePropertyEx<T>::Set(const T& value, DisposerFn disposerFn)
    {
        if (m_disposerFn != 0) {
            m_disposerFn(m_value);
        }

        m_value = value;
        m_disposerFn = disposerFn;
        m_changed = true;
    }

    template<typename T>
    inline const T& DisposablePropertyEx<T>::Get() const
    {
        return m_value;
    }

    template<typename T>
    inline bool DisposablePropertyEx<T>::IsChanged() const
    {
        return m_changed;
    }

    template<typename T>
    inline void DisposablePropertyEx<T>::SetChanged(bool changed)
    {
        m_changed = changed;
    }

    template<typename T>
    inline DisposablePropertyEx<T>::operator const T() const
    {
        return Get();
    }

    template<typename T>
    inline const T& DisposablePropertyEx<T>::operator()() const
    {
        return Get();
    }

}   // namespace Candera

#endif  // CANDERA_DISPOSABLEPROPERTYEX_H
