//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_AGREGATOR_H)
#define CANDERA_AGREGATOR_H

#include <Candera/Environment.h>

#include <Candera/System/MetaInfo/HashFunctions.h>
#include <FeatStd/Util/ArrayOperation.h>
#include <FeatStd/Util/BinarySearch.h>
#include <FeatStd/Platform/String.h>
#include <FeatStd/Container/Vector.h>
#include <FeatStd/Diagnostics/Debug.h>
#include <FeatStd/Diagnostics/ErrorHandling.h>

#ifdef _MSC_VER
#pragma warning( disable : 4503 )
#endif

namespace Candera {
namespace Internal {
    /**
     * AIP is the aggregation instance provider with a parameter. The name is kept short to reduce the space the symbol used in .lib files.
     */
    class AIRBase
    {
    protected:
        AIRBase(void* ref) :
            m_ref(ref)
        {
        }

        const void* Get() const
        {
            return m_ref;
        }

    private:
        AIRBase(const AIRBase& ref);
        AIRBase& operator=(const AIRBase& ref);
        void* m_ref;
    };

    template<typename T>
    class AIR : public AIRBase
    {
    public:
        typedef typename T::BaseMetaInfoProvider BaseMetaInfoProvider;
        AIR() :
            AIRBase(reinterpret_cast<void*>(&I()))
        {
        }

        static T& I()
        {
            static T s_i(BaseMetaInfoProvider::GetMetaInfoImpl());
            return s_i;
        }

    private:
        AIR(const AIR&);
        AIR& operator=(const AIR&);
    };
}

namespace MetaInfo {
/// @addtogroup MetaInfo
/// @{


template<typename BaseType>
struct AggregatableComparer {
    static Int Compare(const Char *left, const BaseType *right) {
        if ((0 == left) || (0 == right)) {
            return -1;
        }
        return (left != right->GetName()) ? FeatStd::Internal::String::CompareStrings(left, right->GetName()) : 0;
    }
};

template<typename BaseType>
struct AggregatableComperatorLess {
    bool operator()(const BaseType* left, const BaseType* right) const {
        return (left != right) && (0 != left) && AggregatableComparer<BaseType>::Compare(left->GetName(), right) < 0;
    }
};

template <typename T>
T& AggregatableInstanceProvider();

namespace Internal {

class EmptyAggregation
{
};

class AggregationInstanceRefBase
{
protected:
    AggregationInstanceRefBase(void* ref) :
        m_ref(ref)
    {
    }

    const void* Get() const
    {
        return m_ref;
    }

private:
    AggregationInstanceRefBase(const AggregationInstanceRefBase& ref);
    AggregationInstanceRefBase& operator=(const AggregationInstanceRefBase& ref);
    void* m_ref;
};

template<typename T>
class AggregationInstanceRef : public AggregationInstanceRefBase
{
public:
    AggregationInstanceRef() :
        AggregationInstanceRefBase(reinterpret_cast<void*>(&AggregatableInstanceProvider<T>()))
    {
    }
};

template<bool firstTypeHasRegister, bool secondTypeHasRegister, typename T1, typename T2>
struct TypeAggregatorRegisterPolicy;

template<bool firstTypeHasRegisterItems, bool secondTypeHasRegisterItems, typename T1, typename T2>
struct TypeAggregatorRegisterItemsPolicy;

template<typename T1, typename T2>
struct TypeAggregatorRegisterPolicy<false, false, T1, T2>
{
    typedef typename T1::BaseType BaseType;
    inline static void* Register(BaseType**)
    {
        return 0;
    }
};

template<typename T1, typename T2>
struct TypeAggregatorRegisterPolicy<false, true, T1, T2>
{
    typedef typename T1::BaseType BaseType;
    inline static void* Register(BaseType** arr)
    {
        typedef void* (*RegisterCallback)(BaseType**);
        RegisterCallback registerCallback = T2::Register;
        // Since T2 also has a Register method that expects its items have the index range [0..T2.Count[
        // the pointer to the array has to be moved to the position of the first item of T2 (arr + T1.Count)
        BaseType** arrT2 = arr + Int32(T1::Count);
        while (0 != registerCallback) {
            registerCallback = reinterpret_cast<RegisterCallback>(registerCallback(arrT2));
        }
        return 0;
    }
};

template<typename T1, typename T2>
struct TypeAggregatorRegisterPolicy<true, false, T1, T2>
{
    typedef typename T1::BaseType BaseType;
    inline static void* Register(BaseType**)
    {
        return reinterpret_cast<void*>(T1::Register);
    }
};

template<typename T1, typename T2>
struct TypeAggregatorRegisterPolicy<true, true, T1, T2>
{
    typedef typename T1::BaseType BaseType;
    inline static void* Register(BaseType** arr)
    {
        typedef void* (*RegisterCallback)(BaseType**);
        RegisterCallback registerCallback = T2::Register;
        // Since T2 also has a Register method that expects its items have the index range [0..T2.Count[
        // the pointer to the array has to be moved to the position of the first item of T2 (arr + T1.Count)
        BaseType** arrT2 = arr + Int32(T1::Count);
        while (0 != registerCallback) {
            registerCallback = reinterpret_cast<RegisterCallback>(registerCallback(arrT2));
        }
        return reinterpret_cast<void*>(T1::Register);
    }
};

template<typename T1, typename T2>
struct TypeAggregatorRegisterItemsPolicy<false, false, T1, T2>
{
    typedef typename T1::BaseType BaseType;
    inline static void RegisterItems(BaseType**)
    {
    }
};

template<typename T1, typename T2>
struct TypeAggregatorRegisterItemsPolicy<false, true, T1, T2> {
    typedef typename T1::BaseType BaseType;
    inline static void RegisterItems(BaseType** arr)
    {
        T2::RegisterItems(arr, T1::Count);
    }
};

template<typename T1, typename T2>
struct TypeAggregatorRegisterItemsPolicy<true, false, T1, T2>
{
    typedef typename T1::BaseType BaseType;
    inline static void RegisterItems(BaseType** arr)
    {
        T1::RegisterItems(arr, T2::Count);
    }
};

template<typename T1, typename T2>
struct TypeAggregatorRegisterItemsPolicy<true, true, T1, T2>
{
    typedef typename T1::BaseType BaseType;
    inline static void RegisterItems(BaseType **, Int32 insertAt)
    {
        FEATSTD_COMPILETIME_ASSERT(((T1::HasRegisterItems != 1) && (T2::HasRegisterItems != 1)));
    }
};

template<typename _ItemType>
struct AggregationContainerPolicy
{
    typedef _ItemType ItemType;
    typedef ItemType* ItemTypePtr;
    typedef void* (*RegisterCallback)(ItemTypePtr*);

    static ItemType* Lookup(const Char *name, ItemTypePtr* sortedItems, Int32 count) {
        typedef AggregatableComparer<ItemType> Comparer;
        ItemType* item = 0;
        if ((0 != name) && (0 != sortedItems)) {
            Int32 i = FeatStd::Internal::BinarySearch<Comparer>(name, sortedItems, count);
            if (i < count) {
                item = sortedItems[i];
            }
        }
        return item;
    }

    static UInt32 GetHash(ItemTypePtr* items, Int32 count) {
        UInt32 hash = 0;
        if (0 != items) {
            for (Int32 i = 0; i < count; ++i) {
                hash = Internal::Fnv1aHash::MixHash(hash, items[i]->GetHash());
            }
        }
        return hash;
    }

    static void InitItems(ItemTypePtr* items, RegisterCallback _registerCallback) {
        if (0 != items) {
            RegisterCallback registerCallback = _registerCallback;
            while (0 != registerCallback) {
                registerCallback = reinterpret_cast<RegisterCallback>(registerCallback(items));
            }
        }
    }

    static void InitSortedItems(ItemTypePtr* sortedItems, ItemTypePtr* items, Int32 count) {
        if ((0 != sortedItems) && (0 != items)) {
            memcpy(sortedItems, items, sizeof(ItemTypePtr) * count);
            SortItems(sortedItems, count);
        }
    }

    static void SortItems(ItemTypePtr* sortedItems, Int32 count) {
        typedef AggregatableComperatorLess< ItemType > Comperator;
        typedef FeatStd::Internal::VectorPrivate::IntroSort< ItemTypePtr, ItemTypePtr*, Comperator > IntroSort;
        if (0 != sortedItems) {
            Comperator comperator;
            IntroSort sort(comperator);
            sort.Sort(&sortedItems[0], &sortedItems[count]);
        }
    }
};

template <typename _ItemType>
class Aggregation {
public:
    typedef _ItemType ItemType;
    typedef Candera::Int32 (*ItemsCallback)(_ItemType*** items, _ItemType*** sortedItems);

    Aggregation(ItemsCallback itemsCallback) :
        m_itemsCallback(itemsCallback)
    {
        FEATSTD_PANIC_IF(0 == m_itemsCallback, "call back for aggregation is not allowed to be null!");
    }

    virtual Candera::Int32 Items(ItemType***, ItemType***)
    {
        return 0;
    }

    ItemType* Item(Candera::Int32 idx)
    {
        ItemType** items = 0;
        (void)m_itemsCallback(&items, 0);
        return items[idx];
    }

    ItemType* Lookup(const Candera::Char *name)
    {
        ItemType** sortedItems = 0;
        Candera::Int32 count = m_itemsCallback(0, &sortedItems);
        return Internal::AggregationContainerPolicy<ItemType>::Lookup(name, sortedItems, count);
    }

    Candera::UInt32 GetHash()
    {
        ItemType** items = 0;
        Candera::Int32 count = m_itemsCallback(&items, 0);
        return Internal::AggregationContainerPolicy<ItemType>::GetHash(items, count);
    }

    Candera::Int32 Count()
    {
        return m_itemsCallback(0, 0);
    }

private:
    ItemsCallback m_itemsCallback;
};

template<typename ItemType>
class StaticAggregationBase
{
protected:
    static void ItemsBase(Candera::Int32 count, ItemType*** items, ItemType*** sortedItems, ItemType** s_items, ItemType** s_sortedItems, bool& initializeSortedItems);
};

template<typename _ItemType, typename _ItemAggregation>
class StaticAggregation : public StaticAggregationBase<_ItemType>
{
public:
    typedef _ItemType ItemType;
    typedef _ItemAggregation ItemAggregation;
    typedef StaticAggregationBase<_ItemType> Base;

    static ItemType* Item(Candera::Int32 idx);

    static ItemType* Lookup(const Candera::Char *name);

    static Candera::UInt32 GetHash();

    static Candera::Int32 Count();

private:
    static Candera::Int32 Items(ItemType*** items, ItemType*** sortedItems);

    static Aggregation<_ItemType>& AggregationInstance();
};

template<typename ItemType>
void StaticAggregationBase<ItemType>::ItemsBase(Candera::Int32 count, ItemType*** items, ItemType*** sortedItems, ItemType** s_items, ItemType** s_sortedItems, bool& initializeSortedItems)
{
    if (0 != items) {
        *items = (count > 0) ? s_items : 0;
    }
    if (0 != sortedItems) {
        *sortedItems = (count > 0) ? s_sortedItems : 0;
        if (initializeSortedItems) {
            initializeSortedItems = false;
            if (count > 0) {
                Candera::MetaInfo::Internal::AggregationContainerPolicy<ItemType>::SortItems(*sortedItems, count);
            }
        }
    }
}

template<typename _ItemType, typename _ItemAggregation>
_ItemType* StaticAggregation<_ItemType, _ItemAggregation>::Item(Candera::Int32 idx)
{
    return AggregationInstance().Item(idx);
}

template<typename _ItemType, typename _ItemAggregation>
_ItemType* StaticAggregation<_ItemType, _ItemAggregation>::Lookup(const Candera::Char *name)
{
    static const Candera::Char *lastName = 0;
    static ItemType* lastItem = 0;
    if (lastName != name) {
        lastItem = AggregationInstance().Lookup(name);
        lastName = name;
    }
    return lastItem;
}

template<typename _ItemType, typename _ItemAggregation>
Candera::UInt32 StaticAggregation<_ItemType, _ItemAggregation>::GetHash()
{
    return AggregationInstance().GetHash();
}

template<typename _ItemType, typename _ItemAggregation>
Candera::Int32 StaticAggregation<_ItemType, _ItemAggregation>::Count()
{
    return AggregationInstance().Count();
}

template<typename _ItemType, typename _ItemAggregation>
Candera::Int32 StaticAggregation<_ItemType, _ItemAggregation>::Items(_ItemType*** items, _ItemType*** sortedItems)
{
    static ItemAggregation s_items;
    static ItemAggregation s_sortedItems;
    static bool initializeSortedItems = true;
    const Candera::Int32 count = sizeof(s_items) / sizeof(ItemType*);
    // reviewed by SESA see RTC Bug 293163
    // coverity[overrun-buffer-arg]
    Base::ItemsBase(count, items, sortedItems, reinterpret_cast<ItemType**>(&s_items), reinterpret_cast<ItemType**>(&s_sortedItems), initializeSortedItems);
    return count;
}

template<typename _ItemType, typename _ItemAggregation>
Aggregation<_ItemType>& StaticAggregation<_ItemType, _ItemAggregation>::AggregationInstance()
{
    static Aggregation<_ItemType> s_aggregation(Items);
    return s_aggregation;
}

} // namespace Internal

/**
 *  Aggregates type T1 and T2 to an aggregatable type
 */
template<typename T1, typename T2>
struct TypeAggregator {
    enum { Count = Int32(T1::Count) + Int32(T2::Count) };
    enum { HasRegister = 1 };
    enum { HasRegisterItems = 0 };
    typedef typename T1::BaseType BaseType;

    /**
     *  Register stores pointers to member _t1 and _t2 to the given BaseType array
     *  @param arr array of pointers
     *  @return incremented pointer to array of pointers
     */
    static void* Register(BaseType** arr) {
        Internal::TypeAggregatorRegisterItemsPolicy<T1::HasRegisterItems == 1, T2::HasRegisterItems == 1, T1, T2>::RegisterItems(arr);
        return Internal::TypeAggregatorRegisterPolicy<T1::HasRegister == 1, T2::HasRegister == 1, T1, T2>::Register(arr);
    }
};

/**
 *  NullAggregatable starts an Aggregatable type with an empty definition
 */
template<typename _BaseType>
struct NullAggregatable {
    enum { Count = 0 };
    enum { HasRegister = 0 };
    enum { HasRegisterItems = 0 };
    typedef _BaseType BaseType;
};

/**
 *  Function used to provided an aggregatable type and external users
 *  with the same instance of the source type.
 *  @return Instance of the
 */
template <typename T>
T& AggregatableInstanceProvider()
{
    static T instance;
    return instance;
}

/**
 *  Aggregatable promotes the given type to an aggregatable type
 */
template<typename T, typename _BaseType>
struct Aggregatable {
    enum { Count = 1 };
    enum { HasRegister = 0 };
    enum { HasRegisterItems = 1 };
    typedef _BaseType BaseType;

    static void RegisterItems(BaseType **arr, Int32 insertAt) {
        FEATSTD_DEBUG_ASSERT(0 == arr[insertAt]);
        arr[insertAt] = &AggregatableInstanceProvider<T>();
    }
};

/**
 *  @brief Defines access functions to the single items of an aggregated type
 */
template<typename Aggregation, typename _ItemType, Int count = Aggregation::Count>
class AggregationContainer {
public:
    typedef _ItemType ItemType;

    /**
     *  Returns item at index idx
     *  @param idx index of item to retrieve
     *  @return item at index idx
     */
    static ItemType* Item(Int32 idx) {
        return Items()[idx];
    }

    /**
     *  returns the number of types aggregated in this type
     *  @return number of aggregated items
     */
    static Int32 Count() { return Int32(count); }

    /**
     *  Lookup an item with the given name
     *  @param name name of the item of interest
     *  @return item instance or null otherwise
     */
    static ItemType* Lookup(const Char *name) {
        return Internal::AggregationContainerPolicy<ItemType>::Lookup(name, SortedItems(), count);
    }

    static UInt32 GetHash() {
        return Internal::AggregationContainerPolicy<ItemType>::GetHash(Items(), count);
    }

private:
    /**
     *  Returns array of pointers to the items in the aggregated type
     *  At first invocation, the pointer array will be initialized by calling
     *  Register of the aggregated type. The dimension of the array is defined
     *  by Count of the aggregated type.
     *  @return array of pointers to the aggregated types.
     */
    static ItemType** Items() {
        static ItemType* s_items[count] = { 0 };
        if (0 == *s_items) {
            Internal::AggregationContainerPolicy<ItemType>::InitItems(s_items, Aggregation::Register);
        }
        return s_items;
    }

    static ItemType** SortedItems() {
        static ItemType* s_sortedItems[count] = { 0 };
        if (0 == *s_sortedItems) {
            Internal::AggregationContainerPolicy<ItemType>::InitSortedItems(s_sortedItems, Items(), count);
        }
        return s_sortedItems;
    }
};

template<typename Aggregation, typename _ItemType>
class AggregationContainer<Aggregation, _ItemType, 0> {
    public:
        typedef _ItemType ItemType;

        /**
         *  Returns item at index idx
         *  @param idx index of item to retrieve
         *  @return item at index idx
         */
        static ItemType* Item(Int32 /*idx*/) {
            return 0;
        }

        /**
         *  returns the number of types aggregated in this type
         *  @return number of aggregated items
         */
        static Int32 Count() { return 0; }

        /**
         *  Lookup an item with the given name
         *  @param name name of the item of interest
         *  @return item instance or null otherwise
         */
        static ItemType* Lookup(const Char * /*name*/) {
            return 0;
        }

        static UInt32 GetHash() {
            return 0;
        }
};
/// @}
}} // namespace Candera::Internal
#endif    // CANDERA_AGREGATOR_H
