//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef ENUM_DATA_TYPE_BASE_H
#define ENUM_DATA_TYPE_BASE_H

#include <Candera/Environment.h>
#include <CanderaPlatform/OS/SerializationPlatform.h>

/// @addtogroup MetaInfo
/// @{

namespace Candera {
    namespace MetaInfo {
        namespace Internal {

/**
 * @brief EnumDataTypeBase is a class that implements the base serialization of
 * an enumeration.
 * @tparam Enumeration Type of the enumeration that is serialized.
 * @tparam EditorFunc A function that returns an editor string for the enumeration.
 */
 template <typename Enumeration, const Char* (*EditorFunc)()>
struct EnumDataTypeBase {
    static bool ConvertToString(const Enumeration val, Char *buf, UInt bufLen) {
        return SerializationPlatform::FromBinary(buf, bufLen, (Int)val) == 1;
    }
    static bool ConvertFromString(Enumeration &val, const Char *buf) {
        Int i = 0;
        UInt n = SerializationPlatform::ToBinary(buf, i);
        val = static_cast<Enumeration>(i);
        return n == 1;
    }
    static const Char* GetEditor() { return EditorFunc(); }
};

}}} // namespaces

// Macros for generating editor functions.

/**
 * Start the description of an editor string. Such a string should be surrounded
 * by a function before passed to EnumDataTypeBase. See EnumDataType for usage
 * example.
 * @param ENUM Name of the enumeration.
 */
#define ENUM_DATA_TYPE_EDITOR_BEGIN(ENUM) \
    "builtin://EnumEditor?name="                                \
        FEATSTD_STRINGIZE(ENUM) "&values="                       \

/**
 * Define an enumeration item. Should only appear between 
 * ENUM_DATA_TYPE_EDITOR_BEGIN and ENUM_DATA_TYPE_EDITOR_END.
 * @param ITEM Name of the enumeration item.
 */
#define ENUM_DATA_TYPE_EDITOR_ITEM(ITEM) \
    FEATSTD_STRINGIZE(ITEM) ",,"
/**
 * Define an enumeration item. Should only appear between 
 * ENUM_DATA_TYPE_EDITOR_BEGIN and ENUM_DATA_TYPE_EDITOR_END.
 * This allows specifying the value of the enumeration item.
 * @param ITEM Name of the enumeration item.
 * @param VALUE Value of the enumeration item.
 *      Must be an integer. May be hexadecimal.
 */
#define ENUM_DATA_TYPE_EDITOR_ITEM_VALUE(ITEM, VALUE) \
    FEATSTD_STRINGIZE(ITEM) "," FEATSTD_STRINGIZE(VALUE) ","

/**
 * End the description of an editor string.
 */
#define ENUM_DATA_TYPE_EDITOR_END()

/**
 * Defines a data type specialization for an Enumeration,
 * using a specified editor function.
 * @param ENUM_TYPE Type of the enumeration.
 * @param EDITOR_FUNCTION Pointer to a function returning an editor string.
 *          Signature must be const Char* (*)().
 */
#define ENUM_DATA_TYPE_DEF(ENUM_TYPE, EDITOR_FUNCTION) \
    template<> struct DataType<ENUM_TYPE> : \
        public Internal::EnumDataTypeBase<ENUM_TYPE, EDITOR_FUNCTION> {}

/// @}

#endif //ENUM_DATA_TYPE_BASE_H
