//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################


#if !defined(Candera_Endianness_h)
#define Candera_Endianness_h


#if !defined(CANDERA_MICRO_CONTROLLER_ENDIANNESS)
    /**
     *  define micro controller endianness [LittleEndian | BigEndian]
     *  there is no portable way to auto detect endianness at compile time. therefore specify
     *  CANDERA_MICRO_CONTROLLER_ENDIANNESS to either LittleEndian or BigEndian.
     *  The big default is LittleEndian
     */
#    define CANDERA_MICRO_CONTROLLER_ENDIANNESS LittleEndian
#elif CANDERA_MICRO_CONTROLLER_ENDIANNESS != LittleEndian && CANDERA_MICRO_CONTROLLER_ENDIANNESS != BigEndian
#    error invalid definition of CANDERA_MICRO_CONTROLLER_ENDIANNESS=<LittleEndian|BigEndian>
#endif


namespace Candera {
    namespace MemoryManagement {
        namespace Internal {
            struct LittleEndian {
                union ByteLayout16 {
                    UInt16 val;
                    struct {
                        UInt8 b0;
                        UInt8 b1;
                    } bytes;
                };

                union ByteLayout32 {
                    UInt32 val;
                    struct {
                        UInt8 b0;
                        UInt8 b1;
                        UInt8 b2;
                        UInt8 b3;
                    } bytes;
                };
            };

            struct BigEndian {
                union ByteLayout16 {
                    UInt16 val;
                    struct {
                        UInt8 b1;
                        UInt8 b0;
                    } bytes;
                };

                union ByteLayout32 {
                    UInt32 val;
                    struct {
                        UInt8 b3;
                        UInt8 b2;
                        UInt8 b1;
                        UInt8 b0;
                    } bytes;
                };
            };

            typedef CANDERA_MICRO_CONTROLLER_ENDIANNESS HostEndianness;

            template<typename from, typename to, typename T, Int tSize = sizeof(T)> struct Converter {
            };

            template<typename T, typename endian> struct Converter<endian, endian, T, 2> {
                static inline T Convert(T val) { return val; }
            };

            template<typename T, typename endian> struct Converter<endian, endian, T, 4> {
                static inline T Convert(T val) { return val; }
            };


            template<typename from, typename to, typename T> struct Converter<from, to, T, 2> {
                static inline T Convert(T val) { return (val << 8) | (val >> 8); }
            };

            template<typename from, typename to, typename T> struct Converter<from, to, T, 4> {
                static inline T Convert(T val) {
                    typename to::ByteLayout32 r;

                    r.bytes.b0 = reinterpret_cast<typename from::ByteLayout32*>(&val)->bytes.b0;
                    r.bytes.b1 = reinterpret_cast<typename from::ByteLayout32*>(&val)->bytes.b1;
                    r.bytes.b2 = reinterpret_cast<typename from::ByteLayout32*>(&val)->bytes.b2;
                    r.bytes.b3 = reinterpret_cast<typename from::ByteLayout32*>(&val)->bytes.b3;

                    return r.val;
                }
            };

        }    // namespace Internal

        /** @addtogroup MemoryMgmtSystem
         *  @{
         */
        using Internal::BigEndian;
        using Internal::LittleEndian;
        using Internal::HostEndianness;

        template<typename T> inline T HostToBigEndian(T val) { return Internal::Converter<HostEndianness, BigEndian, T>::Convert(val); }
        template<typename T> inline T HostToLittleEndian(T val) { return Internal::Converter<HostEndianness, LittleEndian, T>::Convert(val); }

        template<typename T> inline T BigToHostEndian(T val) { return Internal::Converter<BigEndian, HostEndianness, T>::Convert(val); }
        template<typename T> inline T LittleToHostEndian(T val) { return Internal::Converter<LittleEndian, HostEndianness, T>::Convert(val); }

        template<typename T> inline T BigToLittleEndian(T val) { return Internal::Converter<BigEndian, LittleEndian, T>::Convert(val); }
        template<typename T> inline T LittleToBigEndian(T val) { return Internal::Converter<BigEndian, LittleEndian, T>::Convert(val); }
        /** @} */ // end of MemoryMgmtSystem

    }    // namespace MemoryManagement
}    // namespace Candera


#endif    // Candera_Endianness_h

