//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_VECTOR3_H)
#define CANDERA_VECTOR3_H

#include <Candera/Environment.h>
#include <FeatStd/Diagnostics/Debug.h>
#include <Candera/System/Mathematics/Math.h>
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/Mathematics/MathStringBufferAppenders.h>

namespace Candera {
/** @addtogroup MathematicsSystem
 *  @{
 */

class Matrix4;
class Vector4;

/**
 * @brief The default Vector3 class.
 */
class Vector3
{
    FEATSTD_LOG_SET_REALM(Candera::Diagnostics::LogRealm::CanderaSystem);

    public:
        /**
         *  The data of this vector.
         */
        struct Data
        {
            Data() : x(0.F), y(0.F), z(0.F) {}
            Data(Float ix, Float iy, Float iz) : x(ix), y(iy), z(iz) {}

            Float x;
            Float y;
            Float z;
        };

        /**
         *  Destructor
         */
        ~Vector3() {}

        /**
         *  CopyConstructor from another vector.
         *  @param vector The other vector.
         */
        Vector3(const Vector3 &vector) : m_data(vector.m_data) {}

        /**
         *  Constructor
         */
        Vector3() {};

        /**
         *  Constructor with given parameters.
         *  @param xValue The x-value of the vector.
         *  @param yValue The y-value of the vector.
         *  @param zValue The z-value of the vector.
         */
        Vector3(Float xValue, Float yValue, Float zValue) : m_data(xValue, yValue, zValue) {};

        /**
         *  CopyConstructor from another vector4.
         *  @param vector The other vector4 (The fourth value will be ignored).
         */
        explicit Vector3(const Vector4& vector);

        /**
         *  Calculates the Length of the vector.
         *  @see Vector3::GetSquaredLength()
         *  @return         The length of the vector.
         */
        Float GetLength() const {return Math::SquareRoot(m_data.x * m_data.x + m_data.y * m_data.y + m_data.z * m_data.z);}

        /**
         *  Calculates the squared Length of the vector.
         *  This method is faster then Vector3::GetLength(),
         *  and can be used e.g. for comparing the length of 2 Vectors.
         *  @see Vector3::GetLength()
         *  @return         The squared length of the vector.
         */
        Float GetSquaredLength() const;

        /**
         *  Calculates the distance to another vector.
         *  @see Vector3::GetSquaredDistanceTo()
         *  @param  vector  The target vector.
         *  @return         The distance to the passed vector.
         */
        Float GetDistanceTo(const Vector3& vector) const;

        /**
         *  Calculates the squared distance to another vector.
         *  This method is faster then Vector3::GetDistanceTo().
         *  Use this if you want to compare distances between vectors
         *  without incurring the square root.
         *  @see Vector3::GetDistanceTo()
         *  @param  vector  The target vector.
         *  @return         The squared distance to the passed vector.
         */
        Float GetSquaredDistanceTo(const Vector3& vector) const;

        /**
         *  Calculates the scalar dot product of this vector with another vector.
         *  @remarks        Can be used to calculate the angle between two vectors.
         *  @param  vector  The other vector.
         *  @return         The dot product value.
         */
        Float GetDotProduct(const Vector3& vector) const;

        /**
         *  Calculates the cross product of this vector with another vector.
         *  @remarks        The cross product operation between two vectors results
         *                  in a third vector perpendicular to the two input vectors.
         *  @param  vector  The other vector.
         *  @return         A vector perpendicular to this vector and the passed vector.
         */
        Vector3 GetCrossProduct(const Vector3& vector) const;

        /**
         *  Normalizes the vector to a length of a 1.0F approximation.
         *  @return         The length of the vector before the normalization.
         */
        Float Normalize();

        /**
         *  Transforms the vector by the given Matrix, projecting the result back to w = 1.
         *  @param matrix4  The Matrix to transform with.
         */
        void TransformCoordinate(const Matrix4& matrix4);

        /**
         *  Sets all values of the Vector3 to 0.
         */
        void SetZero();

        Vector3 operator + (const Vector3& rhs) const;
        Vector3 operator - (const Vector3& rhs) const;
        Vector3 operator * (const Vector3& rhs) const;

        Vector3& operator = (const Vector3& rhs);
        Vector3& operator += (const Vector3& rhs);
        Vector3& operator -= (const Vector3& rhs);
        Vector3& operator *= (const Vector3& rhs);

        bool operator == (const Vector3& rhs) const;
        bool operator != (const Vector3& rhs) const;

        Vector3& operator = (const Float f);
        Vector3& operator *= (const Float f);
        Vector3 operator * (const Float f) const;
        Vector3& operator /= (const Float f);
        Vector3 operator / (const Float f) const;

        /**
         *  Return the negative of this Vector3.
         *  @return The negative of this Vector3.
         */
        Vector3 operator - () const;
        const Vector3& operator + () const;

        /**
         *  Provides access to the properties by specifying an index.
         *  Index 0 returns x and accordingly.
         *  @param index The index of the desired element.
         *  @return The desired element.
         */
        Float& operator [] (UInt8 index);

        /**
         *  Provides access to the properties by specifying an index.
         *  Index 0 returns x and accordingly.
         *  @param index The index of the desired element.
         *  @return The desired element.
         */
        Float operator [] (UInt8 index) const;

        /**
         *  Retrieves the x-value of this vector.
         *  @return the x-value of this vector.
         */
        Float GetX() const { return m_data.x; }

        /**
         *  Retrieves the y-value of this vector.
         *  @return the y-value of this vector.
         */
        Float GetY() const { return m_data.y; }

        /**
         *  Retrieves the z-value of this vector.
         *  @return the z-value of this vector.
         */
        Float GetZ() const { return m_data.z; }

        /**
         *  Sets the x-value of this vector.
         *  @param xValue The value that is set.
         */
        void SetX(Float xValue) { m_data.x = xValue; }

        /**
         *  Sets the y-value of this vector.
         *  @param yValue The value that is set.
         */
        void SetY(Float yValue) { m_data.y = yValue; }

        /**
         *  Sets the z-value of this vector.
         *  @param zValue The value that is set.
         */
        void SetZ(Float zValue) { m_data.z = zValue; }

        /**
         *  Retrieves the data of this vector.
         *  @return The data of this vector.
         */
        const Data* GetData() const { return &m_data; }

    private:
        Vector3::Data m_data;
};

 /** @} */ // end of MathematicsSystem

inline Vector3& Vector3::operator += (const Vector3& rhs)
{
    m_data.x += rhs.m_data.x;
    m_data.y += rhs.m_data.y;
    m_data.z += rhs.m_data.z;
    return *this;
}

inline Vector3& Vector3::operator -= (const Vector3& rhs)
{
    m_data.x -= rhs.m_data.x;
    m_data.y -= rhs.m_data.y;
    m_data.z -= rhs.m_data.z;
    return *this;
}

inline Vector3& Vector3::operator *= (const Vector3& rhs)
{
    m_data.x *= rhs.m_data.x;
    m_data.y *= rhs.m_data.y;
    m_data.z *= rhs.m_data.z;
    return *this;
}

inline Vector3 Vector3::operator + (const Vector3& rhs) const
{
    return Vector3(
        m_data.x + rhs.m_data.x,
        m_data.y + rhs.m_data.y,
        m_data.z + rhs.m_data.z);
}

inline Vector3 Vector3::operator - (const Vector3& rhs) const
{
    return Vector3(
        m_data.x - rhs.m_data.x,
        m_data.y - rhs.m_data.y,
        m_data.z - rhs.m_data.z);
}

inline Vector3 Vector3::operator * (const Vector3& rhs) const
{
    return Vector3(
        m_data.x * rhs.m_data.x,
        m_data.y * rhs.m_data.y,
        m_data.z * rhs.m_data.z);
}

inline Vector3& Vector3::operator = (const Vector3& rhs)
{
    //no check for self assignment due to performance reasons
    m_data.x = rhs.m_data.x;
    m_data.y = rhs.m_data.y;
    m_data.z = rhs.m_data.z;
    return *this;
}

inline bool Vector3::operator == (const Vector3& rhs) const
{
    return (Math::FloatAlmostEqual(m_data.x, rhs.m_data.x) 
         && Math::FloatAlmostEqual(m_data.y, rhs.m_data.y) 
         && Math::FloatAlmostEqual(m_data.z, rhs.m_data.z));
}

inline bool Vector3::operator != (const Vector3& rhs) const
{
    return !(*this == rhs);
}

inline Vector3& Vector3::operator = (const Float f)
{
    m_data.x = f;
    m_data.y = f;
    m_data.z = f;
    return *this;
}

inline Vector3& Vector3::operator *= (const Float f)
{
    m_data.x *= f;
    m_data.y *= f;
    m_data.z *= f;
    return *this;
}

inline Vector3 Vector3::operator * (const Float f) const
{
    return Vector3(
        m_data.x * f,
        m_data.y * f,
        m_data.z * f);
}

inline Vector3& Vector3::operator /= (const Float f)
{
    if (f == 0.0F) {
        FEATSTD_LOG_ERROR("Vector3 operator, division by zero.");
        return *this;
    }

    //Inverse computation: 1 division + 3 multiplications is faster then 3 divisions
    Float inv = 1.0F / f;

    m_data.x *= inv;
    m_data.y *= inv;
    m_data.z *= inv;

    return *this;
}

inline Vector3 Vector3::operator / (const Float f) const
{
    if (f == 0.0F) {
        FEATSTD_LOG_ERROR("Vector3 operator, division by zero.");
        return *this;
    }

    //Inverse computation: 1 division + 3 multiplications is faster then 3 divisions
    Float inv = 1.0F / f;

    return Vector3(
        m_data.x * inv,
        m_data.y * inv,
        m_data.z * inv);
}

inline Vector3 Vector3::operator - () const
{
    return Vector3(-m_data.x, -m_data.y, -m_data.z);
}

inline const Vector3& Vector3::operator + () const
{
    return *this;
}

inline Float& Vector3::operator [] (UInt8 index)
{
    switch (index) {
        FEATSTD_LINT_CURRENT_SCOPE(1960, "Violates MISRA C++ 2008 Required Rule 6-5-3: all paths return")
        case 0:
            return m_data.x;
        case 1:
            return m_data.y;
        case 2:
            return m_data.z;
        default:
            FEATSTD_DEBUG_FAIL();
            return m_data.x;
    }
}

inline Float Vector3::operator [] (UInt8 index) const
{
    switch (index) {
        FEATSTD_LINT_CURRENT_SCOPE(1960, "Violates MISRA C++ 2008 Required Rule 6-5-3: all paths return")
        case 0:
            return m_data.x;
        case 1:
            return m_data.y;
        case 2:
            return m_data.z;
        default:
            FEATSTD_DEBUG_FAIL();
            return 0.0F;
    }
}
} // namespace Candera

#endif    // CANDERA_VECTOR3_H
