//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Rectangle.h"
#include <Candera/System/Mathematics/Matrix3x2.h>
#include <Candera/System/Mathematics/Math.h>

namespace Candera {

/******************************************************************************
 *  Constructor
 ******************************************************************************/
Rectangle::Rectangle() :
    m_left(0.0F),
    m_top(0.0F),
    m_width(0.0F),
    m_height(0.0F)
{
}

Rectangle::Rectangle(Float left, Float top, Float width, Float height) :
    m_left(left),
    m_top(top),
    m_width(width),
    m_height(height)
{
}

Rectangle::Rectangle(const Vector2& position, const Vector2& size) :
    m_left(position.GetX()),
    m_top(position.GetY()),
    m_width(size.GetX()),
    m_height(size.GetY())
{
}

Rectangle::Rectangle(const Rectangle& rectangle) :
    m_left(rectangle.m_left),
    m_top(rectangle.m_top),
    m_width(rectangle.m_width),
    m_height(rectangle.m_height)
{
}

/******************************************************************************
 *  Destructor
 ******************************************************************************/
Rectangle::~Rectangle()
{
}

/******************************************************************************
 *  operator =
 ******************************************************************************/
Rectangle& Rectangle::operator = (const Rectangle& rhs)
{
    if (this == &rhs) {
        return *this;
    }
    m_left   = rhs.m_left;
    m_top    = rhs.m_top;
    m_width  = rhs.m_width;
    m_height = rhs.m_height;
    return *this;
}

/******************************************************************************
 *  operator ==
 ******************************************************************************/
bool Rectangle::operator==(const Rectangle& rhs) const
{
    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(777, intendeed)
    return ((m_left == rhs.m_left) && (m_top == rhs.m_top) && (m_width == rhs.m_width) && (m_height == rhs.m_height));
}


/******************************************************************************
 *  operator !=
 ******************************************************************************/
bool Rectangle::operator!=(const Rectangle& rhs) const
{
    return !operator==(rhs);
}

/******************************************************************************
 *  UnionAxis
 ******************************************************************************/
 static void UnionAxis(Float& dstXRef, Float& dstWRef, const Float srcX, const Float srcW) {
    const Float dstX = dstXRef;
    const Float dstW = dstWRef;

    if(Math::IsNan(dstX)) {
        dstXRef = dstX;
    }
    else {
        dstXRef = (dstX < srcX) ? dstX : srcX;
    }

    if ((dstW < 0.0F) || (srcW < 0.0F)) {
        dstWRef = -1.0F;
    }
    else {
        Float dstR = (dstX + dstW) - dstXRef;
        if(Math::IsNan(dstR)) {
            dstWRef = dstR;
        }
        else {
            Float srcR = (srcX + srcW) - dstXRef;
            dstWRef = (dstR > srcR) ? dstR : srcR;
        }
    }
 }

/******************************************************************************
 *  Union
 ******************************************************************************/
void Rectangle::Union(const Rectangle& src)
{
    if ((src.GetWidth() == 0.0F) || (src.GetHeight() == 0.0F)) {
        return;
    }

    if ((GetWidth() == 0.0F) || (GetHeight() == 0.0F)) {
        *this = src;
        return;
    }

    UnionAxis(m_left, m_width, src.GetLeft(), src.GetWidth());
    UnionAxis(m_top, m_height, src.GetTop(), src.GetHeight());
}

/******************************************************************************
 *  IntersectAxis
 ******************************************************************************/
 static void IntersectAxis(Float& dstXRef, Float& dstWRef, const Float srcX, const Float srcW) {
    const Float dstX = dstXRef;
    const Float dstW = dstWRef;
    
    dstXRef = (dstX > srcX) ? dstX : srcX;
    
    if ((dstW < 0.0F) && (srcW < 0.0F)) {
        dstWRef = -1.0F;
    }
    else {
        Float dstR = (dstW < 0.0F) ? Math::MaxFloat() : ((dstX + dstW) - dstXRef);
        Float srcR = (srcW < 0.0F) ? Math::MaxFloat() : ((srcX + srcW) - dstXRef);
        
        dstWRef = (dstR < srcR) ? dstR : srcR;
        if (dstWRef < 0.0F) {
            dstWRef = 0.F;
        }
        if (dstWRef == 0.0F) {
            dstXRef = 0.0F;
        }
    }
 }

/******************************************************************************
 *  Intersect
 ******************************************************************************/
void Rectangle::Intersect(const Rectangle& src)
{
    IntersectAxis(m_left, m_width, src.GetLeft(), src.GetWidth());
    IntersectAxis(m_top, m_height, src.GetTop(), src.GetHeight());
}

/******************************************************************************
 *  IntersectChild
 ******************************************************************************/
void Rectangle::IntersectChild(const Rectangle& child)
{
    Rectangle rect(child);
    rect.SetPosition(GetLeft() + rect.GetLeft(), GetTop() + rect.GetTop());
    Intersect(rect);
}

/******************************************************************************
 *  Transform
 ******************************************************************************/
void Rectangle::Transform(const Matrix3x2& mat)
{
    Float l = GetLeft();
    Float t = GetTop();
    Float r = l + GetWidth();
    Float b = t + GetHeight();

    Float lx = l * mat(0, 0);
    Float ly = l * mat(0, 1);
    Float tx = t * mat(1, 0);
    Float ty = t * mat(1, 1);
    Float rx = r * mat(0, 0);
    Float ry = r * mat(0, 1);
    Float bx = b * mat(1, 0);
    Float by = b * mat(1, 1);

    Float minxx; Float maxxx;
    if (lx < rx) { minxx = lx; maxxx = rx; } else { minxx = rx; maxxx = lx; }
    Float minxy; Float maxxy;
    if (tx < bx) { minxy = tx; maxxy = bx; } else { minxy = bx; maxxy = tx; }
    Float minyx; Float maxyx;
    if (ly < ry) { minyx = ly; maxyx = ry; } else { minyx = ry; maxyx = ly; }
    Float minyy; Float maxyy;
    if (ty < by) { minyy = ty; maxyy = by; } else { minyy = by; maxyy = ty; }

    Float minx = minxx + minxy;
    Float miny = minyx + minyy;

    Float maxx = maxxx + maxxy;
    Float maxy = maxyx + maxyy;

    SetLeft(minx + mat(2, 0));
    SetTop(miny + mat(2, 1));
    SetWidth(maxx - minx);
    SetHeight(maxy - miny);
}

/******************************************************************************
 *  Contains
 ******************************************************************************/
bool Rectangle::Contains(const Vector2& point) const
{
    const Float x = point.GetX();
    const Float y = point.GetY();
    
    return (x >= m_left) &&
        (y >= m_top) &&
        ((m_width < 0.0F) || (x < (m_left + m_width))) &&
        ((m_height < 0.0F) || (y < (m_top + m_height)));
}

Float& Rectangle::operator [] (UInt8 index)
{
    Float* result = &m_left;
    switch (index) {
    case 0: { result = &m_left; break; }
    case 1: { result = &m_top; break; }
    case 2: { result = &m_width; break; }
    case 3: { result = &m_height; break; }
    default:
        FEATSTD_DEBUG_FAIL();
    }
    return *result;
}

Float Rectangle::operator [] (UInt8 index) const
{
    Float result = 0.0F;
    switch (index) {
    case 0: { result = m_left; break; }
    case 1: { result = m_top; break; }
    case 2: { result = m_width; break; }
    case 3: { result = m_height; break; }
    default:
        FEATSTD_DEBUG_FAIL();
    }
    return result;
}

}  //end of namespace Candera
