//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Plane.h"

namespace Candera {


Plane::Plane() :
    m_distance(0.0F)
{
    m_direction.SetZero();
}

Plane::~Plane()
{
}

Plane::Plane(const Plane& plane) : m_distance(plane.GetDistance()), m_direction(plane.GetDirection())
{
}

Plane::Plane(const Vector3& direction, Float distance) : m_distance(distance), m_direction(direction) 
{

}

Float Plane::GetDotProduct(const Vector4& vector) const 
{
    Vector4 temp(m_direction.GetX(), m_direction.GetY(), m_direction.GetZ(), m_distance);
    return temp.GetDotProduct(vector);
}

bool Plane::IsIntersectingLine(Vector3& point, const Vector3& start, const Vector3& end, Float accuracy) const
{
    Vector3 direction = end - start;
    Float dot;
    Float temp;

    Vector3 normalized_direction(m_direction); // to keep this method const
    static_cast<void>(normalized_direction.Normalize());
    
    dot = normalized_direction.GetDotProduct(direction);
    
    if (((dot < accuracy) && (dot > -accuracy)) || (dot == 0.0F)) {
        return false; //the line is almost parallel to the plane
    }

    temp = (m_distance + normalized_direction.GetDotProduct(start)) / dot;
    point.SetX(start.GetX() - (temp * direction.GetX()));
    point.SetY(start.GetY() - (temp * direction.GetY()));
    point.SetZ(start.GetZ() - (temp * direction.GetZ()));

    return true;
}

Plane& Plane::operator = (const Plane& rhs)
{
    if (this != &rhs) {
        m_direction = rhs.m_direction;
        m_distance = rhs.m_distance;
    }
    return *this;
}

bool Plane::operator==(const Plane& rhs) const
{
    CANDERA_SUPPRESS_LINT_FOR_CURRENT_SCOPE(777, intendeed)
    return (m_distance == rhs.m_distance) && (m_direction == rhs.m_direction);
}

} // namespace Candera

