//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_MATRIX3_H)
#define CANDERA_MATRIX3_H

#include <Candera/Environment.h>
#include <Candera/System/Mathematics/Math.h>
#include <CanderaPlatform/OS/MemoryPlatform.h>

namespace Candera {
    /** @addtogroup MathematicsSystem
    *  @{
    */

    //Forward declarations to prevent reciprocal dependencies
    class Matrix4;

    /**
    * @brief The default representation of a 3x3 Matrix.
    */
    class Matrix3
    {
    public:

        /**
         *  Constructor
         */
        Matrix3();

        /**
         *  Destructor
         */
        ~Matrix3(); // non-virtual

        /**
         *  CopyConstructor from another 3x3 Matrix.
         *  @param matrix3 The other 3x3 Matrix.
         */
        Matrix3(const Matrix3& matrix3);

        /**
         *  CopyConstructor from another 4x4 Matrix
         *  Elements outside the 3x3 Matrix will be ignored.
         *  @param matrix4 The other 4x4 Matrix.
         */
        Matrix3(const Matrix4& matrix4);

        /**
         *  Constructor each element of the matrix is set individually
         *  @param m00 Element 0,0
         *  @param m01 Element 0,1
         *  @param m02 Element 0,2
         *  @param m10 Element 1,0
         *  @param m11 Element 1,1
         *  @param m12 Element 1,2
         *  @param m20 Element 2,0
         *  @param m21 Element 2,1
         *  @param m22 Element 2,2
         */
        Matrix3(Float m00, Float m01, Float m02,
            Float m10, Float m11, Float m12,
            Float m20, Float m21, Float m22);

        /**
         *  Constructs a 3x3 Matrix3 out of a float array.
         *  @param arr9    The float array consisting of 16 values.
         */
        Matrix3(const Float arr9[]);

        /**
         *  Sets all values of this Matrix3 to 0.
         */
        void SetZero();

        /**
         *  Sets the Identity Matrix
         */
        void SetIdentity();

        /**
         *  Sets the Matrix3 to a rotated Matrix3 around the X-axis.
         *  The Matrix3 is at first initialized to an identity Matrix3
         *  and then the Rotation values are set.
         *  @param  angle       The angle in degree.
         */
        void SetRotationX(Float angle);

        /**
         *  Sets the Matrix3 to a rotated Matrix3 around the Y-axis.
         *  The Matrix3 is at first initialized to an identity Matrix3
         *  and then the Rotation values are set.
         *  @param  angle       The angle in degree.
         */
        void SetRotationY(Float angle);

        /**
         *  Sets the Matrix3 to a rotated Matrix3 around the Z-axis.
         *  The Matrix is at first initialized to an identity Matrix3
         *  and then the Rotation values are set.
         *  @param  angle       The angle in degree.
         */
        void SetRotationZ(Float angle);

        /**
         *  Sets the Matrix3 to a rotated Matrix3 around the X, Y and Z-axis.
         *  The Matrix3 is at first initialized to an identity Matrix3
         *  and then the Rotation values are set. This method is faster then
         *  using 3 consecutive calls to SetRotationX, SetRotationY and SetRotationZ.
         *  @param  x_angle       The angle to rotate around the x-axis in degree.
         *  @param  y_angle       The angle to rotate around the y-axis in degree.
         *  @param  z_angle       The angle to rotate around the z-axis in degree.
         */
        void SetRotationXYZ(Float x_angle, Float y_angle, Float z_angle);

        /**
         *  Sets the Matrix3 to a rotated Matrix3 around a specified axis.
         *  The Matrix3 is at first initialized to an identity Matrix3
         *  and then the Rotation values are set.
         *  @param  angle    The angle in degree.
         *  @param  x        The x value of the rotation axis.
         *  @param  y        The y value of the rotation axis.
         *  @param  z        The z value of the rotation axis.
         */
        void SetRotationAxis(Float angle, Float x, Float y, Float z);

        /**
         *  Sets the Matrix3 to a scaled Matrix3.
         *  The Matrix3 is at first initialized to an identity Matrix3
         *  and then the scaling values are set.
         *  @param  x           The scaling on the X-axis.
         *  @param  y           The scaling on the Y-axis.
         *  @param  z           The scaling on the Z-axis.
         */
        void SetScaling(Float x, Float y, Float z);

        /**
         *  Creates the inverse version of the Matrix3.
         */
        void Inverse();

        /**
         *  Creates the transpose version of the Matrix3.
         */
        void Transpose();

        /**
         *  Retrieves the determinant of the Matrix3.
         *  @return The Determinant of the Matrix3.
         */
        Float GetDeterminant() const;

        /**
         *  Gets the value at a specific row and column.
         *  The indexes of row and column start with 0.
         *  @param  row     The index of the row.
         *  @param  col     The index of the column.
         *  @return         The value at the intersection of row and col.
         */
        Float Get(UInt8 row, UInt8 col) const;

        /**
         *  Gets the Data out of the Matrix3.
         *  @return  The Data out of the Matrix3
         */
        const Float* GetData() const { return &m_data[0][0]; }

        /**
         *  Sets a value at a specific row and column.
         *  The indexes of row and column start with 0.
         *  @param  row     The index of the row.
         *  @param  col     The index of the column.
         *  @param  val     The value to set.
         */
        void Set(UInt8 row, UInt8 col, Float val);

        Matrix3 operator * (const Matrix3& rhs) const;
        Matrix3& operator *= (const Matrix3& rhs);
        Matrix3& operator = (const Matrix3& rhs);
        Float& operator () (UInt8 row, UInt8 col);
        Float  operator () (UInt8 row, UInt8 col) const;

        bool operator == (const Matrix3& rhs) const;
        bool operator != (const Matrix3& rhs) const;

    private:
        Float m_data[3][3];
    };

    /** @} */ // end of MathematicsSystem

    inline Matrix3& Matrix3::operator *= (const Matrix3& rhs)
    {
        *this = ((*this) * rhs);
        return *this;
    }

    inline Matrix3& Matrix3::operator = (const Matrix3& rhs)
    {
        MemoryPlatform::Copy(m_data, &rhs.m_data, sizeof(m_data));
        return *this;
    }

    inline Float Matrix3::Get(UInt8 row, UInt8 col) const
    {
        return m_data[row][col];
    }

    inline void Matrix3::Set(UInt8 row, UInt8 col, Float val)
    {
        m_data[row][col] = val;
    }

    inline Float& Matrix3::operator () (UInt8 row, UInt8 col)
    {
        return m_data[row][col];
    }

    inline Float Matrix3::operator () (UInt8 row, UInt8 col) const
    {
        return m_data[row][col];
    }

    inline bool Matrix3::operator == (const Matrix3& rhs) const
    {
        return MemoryPlatform::Compare(m_data, rhs.m_data, sizeof(m_data)) == 0;
    }

    inline bool Matrix3::operator!= (const Matrix3& rhs) const
    {
        return !(*this == rhs);
    }

} // namespace Candera

#endif  // CANDERA_MATRIX3_H
