//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_MATHDATATYPES_H)
    #define CANDERA_MATHDATATYPES_H

#include <Candera/System/MetaInfo/DataType.h>
#include <Candera/System/Mathematics/MathStringBufferAppenders.h>
#include <Candera/System/Mathematics/Rectangle.h>
#include <Candera/System/Mathematics/Variant.h>
#include <Candera/System/Mathematics/Vector2.h>
#include <Candera/System/Mathematics/Vector3.h>
#include <Candera/System/Mathematics/Vector4.h>

namespace Candera {

    namespace MetaInfo {

        /** 
         * @addtogroup DataTypesSystem
         * @{
         */

        /**
         * @brief DataType specialization for Rectangle.
         * 
         * The string representation of a Rectangle consists of the four Rectangle coordinates comma separated,
         *  e.g. "0.0,0.0,100.0,100.0", where left = 0.0F, top = 0.0F, width = 100.0F, height = 100.0F.
         */
        template<> struct DataType<Rectangle> {           
            /**
             * Convert a Rectangle to its string representation.
             * 
             * @param rect Rectangle to convert.
             * @param buf Buffer to write the string representation.
             * @param bufLen Buffer length.
             * @return true if conversion successful, false otherwise.
             */
            static bool ConvertToString(const Rectangle& rect, Char* buf, UInt bufLen);
            
            /**
             * Convert the string representation to a Rectangle.
             * 
             * @param rect converted Rectangle.
             * @param buf Buffer that contains the string representation.
             * @return true if conversion successful, false otherwise.
             */
            static bool ConvertFromString(Rectangle& rect, const Char* buf);
            
            /**
             * Get the PropertyEditor name as known by SceneComposer.
             * 
             * @return string containing the SceneComposer property editor name.
             */
            static const Char* GetEditor() { return "builtin://RectangleEditor"; }
        };

        /**
         * @brief DataType specialization for Vector2.
         * 
         * The string representation of a Vector2 consists of the two Vector2 components comma separated,
         *  e.g. "0.0,1.0", where x = 0.0F, y = 1.0F.
         */
        template<> struct DataType<Vector2> {          
            /**
             * Convert a Vector2 to its string representation.
             * 
             * @param val Vector2 to convert.
             * @param buf Buffer to write the string representation.
             * @param bufLen Buffer length.
             * @return true if conversion successful, false otherwise.
             */
            static bool ConvertToString(const Vector2 &val, Char *buf, UInt bufLen);
            
            /**
             * Convert the string representation to a Vector2.
             * 
             * @param val converted Vector2.
             * @param buf Buffer that contains the string representation.
             * @return true if conversion successful, false otherwise.
             */
            static bool ConvertFromString(Vector2 &val, const Char *buf);
            
            /**
             * Get the PropertyEditor name as known by SceneComposer.
             * 
             * @return string containing the SceneComposer property editor name.
             */
            static const Char* GetEditor() { return "builtin://Vector2Editor"; }
        };

        /**
         * @brief DataType specialization for Vector3.
         * 
         * The string representation of a Vector3 consists of the three Vector3 components comma separated,
         *  e.g. "0.0,0.0,1.0", where x = 0.0F, y = 0.0F, z = 1.0F.
         */
        template<> struct DataType<Vector3> {          
            /**
             * Convert a Vector3 to its string representation.
             * 
             * @param val Vector3 to convert.
             * @param buf Buffer to write the string representation.
             * @param bufLen Buffer length.
             * @return true if conversion successful, false otherwise.
             */
            static bool ConvertToString(const Vector3 &val, Char *buf, UInt bufLen);
            
            /**
             * Convert the string representation to a Vector3.
             * 
             * @param val converted Vector3.
             * @param buf Buffer that contains the string representation.
             * @return true if conversion successful, false otherwise.
             */
            static bool ConvertFromString(Vector3 &val, const Char *buf);
            
            /**
             * Get the PropertyEditor name as known by SceneComposer.
             * 
             * @return string containing the SceneComposer property editor name.
             */
            static const Char* GetEditor() { return "builtin://Vector3Editor"; }
        };    
        
        /**
        * @brief DataType specialization for Vector4.
        *
        * The string representation of a Vector4 consists of the four Vector4 components comma separated,
        *  e.g. "0.0,0.0,1.0,1.0", where x = 0.0F, y = 0.0F, z = 1.0F, w =1.0F.
        */
        template<> struct DataType<Vector4> {
            /**
            * Convert a Vector4 to its string representation.
            *
            * @param val Vector4 to convert.
            * @param buf Buffer to write the string representation.
            * @param bufLen Buffer length.
            * @return true if conversion successful, false otherwise.
            */
            static bool ConvertToString(const Vector4 &val, Char *buf, UInt bufLen);

            /**
            * Convert the string representation to a Vector4.
            *
            * @param val converted Vector4.
            * @param buf Buffer that contains the string representation.
            * @return true if conversion successful, false otherwise.
            */
            static bool ConvertFromString(Vector4 &val, const Char *buf);

            /**
            * Get the PropertyEditor name as known by SceneComposer.
            *
            * @return string containing the SceneComposer property editor name.
            */
            static const Char* GetEditor() { return "builtin://Vector4Editor"; }
        };

        /**
        * @brief DataType specialization for Variant.
        *
        * The string representation of a Variant consists of type represented by the value of the type (see VariantValueType) and the actual value.
        * 
        */
        template<> struct DataType<Variant> {
            /**
            * Convert a Variant to its string representation.
            *
            * @param val Variant to convert.
            * @param buf Buffer to write the string representation.
            * @param bufLen Buffer length.
            * @return true if conversion successful, false otherwise.
            */
            static bool ConvertToString(const Variant &val, Char *buf, UInt bufLen);

            /**
            * Convert the string representation to a Variant.
            *
            * @param val converted Variant.
            * @param buf Buffer that contains the string representation.
            * @return true if conversion successful, false otherwise.
            */
            static bool ConvertFromString(Variant &val, const Char *buf);

            /**
            * Get the PropertyEditor name as known by SceneComposer.
            *
            * @return string containing the SceneComposer property editor name.
            */
            static const Char* GetEditor()
            {
                return "builtin://VariantEditor"
                    "?Types{"
                    "(Float,builtin://FloatEditor),"
                    "(Int32,builtin://IntEditor),"
                    "(UInt32,builtin://UIntEditor),"
                    "(Bool,builtin://BoolEditor),"
                    "(Double,builtin://DoubleEditor),"
                    "(Int8,builtin://SByteEditor),"
                    "(Int16,builtin://ShortEditor),"
                    "(Int64,builtin://LongEditor),"
                    "(UInt8,builtin://ByteEditor),"
                    "(UInt16,builtin://UShortEditor),"
                    "(UInt64,builtin://ULongEditor),"
                    "(String,builtin://StringEditor),"
                    "}";
            }
        };
        /**
         * @} //end of DataTypesSystem
         */

    }   // namespace MetaInfo
}   // namespace Candera

#endif  // CANDERA_MATHDATATYPES_H

