//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "CultureDataType.h"
#include <Candera/System/GlobalizationBase/Culture.h>
#include <Candera/System/Container/Vector.h>
#include <FeatStd/Util/StaticObject.h>
#include <Candera/System/Diagnostics/Log.h>

namespace Candera {
    using namespace Globalization;
    FEATSTD_LOG_SET_REALM(Candera::Diagnostics::LogRealm::CanderaSystem);

    namespace MetaInfo {

        class CultureList {
            public:
                FEATSTD_TYPEDEF_SHARED_POINTER(CultureList);

                CultureList() : m_noCulture(0) {}
                ~CultureList();
                const Culture::SharedPointer& GetCulture(const Char* name);
                bool AddCulture(const Culture::SharedPointer& culture) { return m_cultureList.Add(culture); }
            private:
                Candera::Internal::Vector<Culture::SharedPointer> m_cultureList;
                const Culture::SharedPointer m_noCulture;
                CANDERA_SHARED_POINTER_DECLARATION();
        };

        CultureList::~CultureList()
        {
            for (UInt i = 0; i < static_cast<UInt>(m_cultureList.Size()); ++i) {
                FEATSTD_DELETE_ARRAY(m_cultureList[i]->GetDisplayName());
            }
            m_cultureList.Clear();
        }

        const Culture::SharedPointer& CultureList::GetCulture(const Char* name)
        {
            for (UInt i = 0; i < static_cast<UInt>(m_cultureList.Size()); ++i) {
                if (StringPlatform::CompareStrings(name, m_cultureList[i]->GetDisplayName()) == 0) {
                    return m_cultureList[i];
                }
            }

            return m_noCulture;
        }

        bool DataType<Culture::SharedPointer>::ConvertToString(const Culture::SharedPointer& culture, Char* buf, UInt bufLen)
        {
            if (culture == 0) {
                return false;
            }
            Int count = 0;
            static_cast<void>(SerializationPlatform::FromBinary(buf, bufLen, static_cast<Int>(culture->GetTextDirection()), &count));
            buf += count;

            if (count < static_cast<Int>(bufLen)) {
                return false;
            }

            buf[count] = ';';
            buf++;
            bufLen -= static_cast<UInt>(count + 1);

            static_cast<void>(SerializationPlatform::FromBinary(buf, bufLen, culture->GetLocale(), &count));

            return (buf[count] == '\0');
        }

        static CultureList::SharedPointer& GetCultureList()
        {
            FEATSTD_UNSYNCED_STATIC_OBJECT(CultureList::SharedPointer, s_cultureList, FEATSTD_NEW(CultureList));
            return s_cultureList;
        }

        CultureList::SharedPointer& s_forceInitCultureList = GetCultureList();


#ifdef FEATSTD_THREADSAFETY_ENABLED
        static FeatStd::Internal::CriticalSection& GetCriticalSection()
        {
            FEATSTD_UNSYNCED_STATIC_OBJECT(FeatStd::Internal::CriticalSection, cs);
            return cs;
        }

        static FeatStd::Internal::CriticalSection& s_forceInitCriticalSection = GetCriticalSection();

#endif

        bool DataType<Culture::SharedPointer>::ConvertFromString(Culture::SharedPointer& culture, const Char* buf)
        {
            culture.Release();
            if ((buf == 0) || (StringPlatform::CompareStrings(buf, "") == 0)) {
                return true;
            }

#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker lock(&GetCriticalSection());
#endif

            CultureList::SharedPointer& cultureList = GetCultureList();
            culture = cultureList->GetCulture(buf);
            if (culture != 0) {
                return true;
            }
            culture = Culture::Create();
            if (culture != 0) {
                SizeType length = StringPlatform::Length(buf);
                Char* cultureName = FEATSTD_NEW_ARRAY(Char, length + 1);
                if (cultureName == 0) {
                    return false;
                }
                StringPlatform::Copy(cultureName, buf);
                cultureName[length] = '\0';

                Int textDirection = 0;
                Int count = 0;
                static_cast<void>(SerializationPlatform::ToBinary(cultureName, textDirection, &count));

                culture->SetDisplayName(cultureName);
                culture->SetLocale(cultureName + count + 1);
                culture->SetTextDirection(static_cast<TextDirection>(textDirection));
                static_cast<void>(cultureList->AddCulture(culture));
            }
            else {
                FEATSTD_LOG_DEBUG("Culture creation were not possible");
            }

            return (culture != 0);
        }
    }   // namespace MetaInfo
}   // namespace Candera
