//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_ENTITYCOMPONENTSYSTEM_ENTITYSYSTEM_H
#define CANDERA_ENTITYCOMPONENTSYSTEM_ENTITYSYSTEM_H

#include <Candera/System/Container/Map.h>
#include <Candera/System/Container/Vector.h>
#include <Candera/System/EntityComponentSystem/AbstractComponentSystem.h>
#include <Candera/System/EntityComponentSystem/ComponentHandle.h>
#include <Candera/System/Rtti/Rtti.h>
#include <FeatStd/Platform/CriticalSection.h>

namespace Candera {

class Event;

namespace EntityComponentSystem
{

/** @addtogroup EntityComponentSystem
 *  @{
 */

class Entity;

/**
 * @brief  The EntitySystem is a globally accessible class that executes registered component systems, dispatches events
 *         to them, and keeps track of all components attached to entities. Using the entity system, components can be
 *         retrieved by type for any given entity.
 *         For general information about the Entity Component System see EntityComponentSystem.h
 */

class EntitySystem
{
public:
    /**
     *  Dispatches an event to all registered component systems.
     *  @param event  The event to be dispatched to all component systems.
     */
    static void OnEvent(const Event& event);

    /**
     *  Create a component system of the specified type.
     *  @return  The component system that was created, or 0 if a system of that type, or a system using the same
     *           component type, already exists.
     */
    template<typename ComponentSystem> static ComponentSystem* Create();

    /**
     *  Destroy a component system of the specified type.
     *  @return  True, if the operation was successful. False, if no system of the given type exists, or the operation failed.
     */
    template<typename ComponentSystem> static bool Destroy() { return Destroy(ComponentSystem::GetTypeId()); }

    /**
     *  Get a component system by type.
     *  @return  The component system of the given type, or 0 if none is registered in the entity system.
     */
    template<typename ComponentSystem> static ComponentSystem* Get() { return static_cast<ComponentSystem*>(Get(ComponentSystem::GetTypeId())); }

    /**
     *  Get the component handle of a component defined by the given type that is attached to the given entity.
     *  @param entity  The entity to return the requested component handle for.
     *  @return  The component handle of the given component type for the given entity. Returns a 'null' handle if
     *           no component of the requested type is attached to the entity.
     */
    template<typename ComponentType> static ComponentHandle<ComponentType> GetComponent(const Entity* entity);

    /**
     *  Get the component handle of a component defined by the given type that is attached to the given entity.
     *  @param entity  The entity to return the requested component handle for.
     *  @return  The component handle of the given component type for the given entity. Returns a 'null' handle if
     *           no component of the requested type is attached to the entity.
     */
    template<typename ComponentType, typename DerivedComponentType> static ComponentHandle<ComponentType> GetDerivedComponent(const Entity* entity);

    /**
     * @brief Iterator to get all components of a type from an entity
     */
    template<typename ComponentType>
    class ComponentIterator
    {
    public:
        ComponentIterator(const Entity* entity);

        bool Get(ComponentHandle<ComponentType>& handle) const;
        void Next() const { ++m_index; }

    private:
        Internal::Vector<ComponentHandle<ComponentType> > m_componentContainer;
        mutable SizeType m_index;
    };

    /**
     *  Deregister an entity, i.e. remove it from the entity system. All attached components of that entity are
     *  deregistered in their respective component system.
     *  @param entity The entity to be removed from the entity system.
     *  @return True, if the entity could be successfully deregistered, False otherwise.
     */
    static bool DeregisterEntity(const Entity* entity);

    /**
     *  Dispatches the call to all registered component systems.
     */
    static void HeartBeat();

    /**
     *  Dispatches the call to all registered component systems that are enabled.
     */
    static void Update();

private:
    // Calls AttachComponent(), DetachComponent(), and  OnComponentSystemPriorityChanged()
    friend class AbstractComponentSystem;

    /**
     *  Attaches a component identified by its handle and type for the given entity.
     *  @param entity  The entity to attach the component to.
     *  @param handle  The handle of the component to attach.
     *  @param typeId  The typeId of the component to attach.
     *  @return  True, if the component was successfully attached to the entity.
     *           False, if the operation failed, or there is already a component of the same type
     *           attached to the given entity.
     */
    static bool AttachComponent(const Entity* entity, const UInt32 handle, const TypeId typeId);

    /**
     *  Detaches a component identified by its type from the given entity
     *  @param entity  The entity to detach the component from.
     *  @param handle  The handle of the component to detach.
     *  @param typeId  The typeId of the component to detach.
     *  @return  True, if the component was successfully detached from the entity.
     *           False, if the operation failed, or there was no component of the given typeId
     *           attached to the given entity.
     */
    static bool DetachComponent(const Entity* entity, const UInt32 handle, const TypeId typeId);

    /**
     *  Notifies the entity system when the priority of a component system changed.
     */
    static void OnComponentSystemPriorityChanged();

    /**
     *  Destroy a component system of the specified type.
     *  @param typeId  The type of the component system to be destroyed.
     *  @return  True, if the operation was successful. False, if no system of the given type exists, or the operation failed.
     */
    static bool Destroy(const TypeId typeId);

    /**
     *  Get a component system by type.
     *  @param typeId  The type of the component system to get.
     *  @return  The component system of the given type, or 0 if none is registered in the entity system.
     */
    static AbstractComponentSystem* Get(const TypeId typeId);

    /**
     *  Get the component handles of all components defined by the given type that are attached to the given entity.
     *  @param entity                   The entity to return the requested component handles for.
     *  @param componentContainer       The container to return all component handles in.
     *  @param clearComponentContainer  If clearComponentContainer is true, the componentContainer will be cleared before use.
     */
    template<typename ComponentType>
    static void GetComponents(const Entity* entity, Internal::Vector<ComponentHandle<ComponentType> >& componentContainer);

    /**
     *  Get the component handles of all components defined by the given type that are attached to the given entity.
     *  @param entity                   The entity to return the requested component handles for.
     *  @param componentContainer       The container to return all component handles in.
     *  @param clearComponentContainer  If clearComponentContainer is true, the componentContainer will be cleared before use.
     */
    template<typename ComponentType, typename DerivedComponentType>
    static void GetDerivedComponents(const Entity* entity, Internal::Vector<ComponentHandle<ComponentType> >& componentContainer);

    typedef Internal::Vector<AbstractComponentSystem*> ComponentSystems;
    static ComponentSystems& GetComponentSystems();
    static ComponentSystems& s_forceInitComponentSystems;

    struct ComponentInfo {
        ComponentInfo(UInt32 handle, TypeId typeId) : m_handle(handle), m_typeId(typeId) {}
        UInt32 m_handle;
        TypeId m_typeId;
    };

    typedef Internal::Vector<ComponentInfo, FeatStd::Internal::LinearIncreasePolicy<1> > Components;
    typedef Internal::Map<const Entity*, Components> EntityComponents;
    static EntityComponents& GetEntityComponents();
    static EntityComponents& s_forceInitEntityComponents;

    EntitySystem();
    virtual ~EntitySystem() = 0; // Provoke abstract (static) class
};

template<typename ComponentSystem>
ComponentSystem* EntitySystem::Create()
{
    ComponentSystems& componentSystems = GetComponentSystems();
    for (SizeType i = 0; i < componentSystems.Size(); ++i) {
        if (ComponentSystem::GetTypeId() == componentSystems[i]->GetDynamicTypeId()) {
            // There is already a component system of that type
            return 0;
        }
    }

    ComponentSystem* componentSystem = FEATSTD_NEW(ComponentSystem);
    componentSystem->m_wasCreatedbyEntitySystem = true;
    for (SizeType i = 0; i < componentSystems.Size(); ++i) {
        if (componentSystem->GetComponentTypeId() == componentSystems[i]->GetComponentTypeId())
        {
            // There is already a component system for that component type
            FEATSTD_DELETE(componentSystem);
            return 0;
        }
    }

    componentSystems.Add(componentSystem);
    OnComponentSystemPriorityChanged();
    return componentSystem;
}

template<typename ComponentType>
ComponentHandle<ComponentType> EntitySystem::GetComponent(const Entity* entity)
{
    EntityComponents& entityComponents = GetEntityComponents();
    Components* components = entityComponents.Find(entity);
    if (0 != components) {
        for (SizeType i = 0; i < components->Size(); ++i) {
            if (ComponentType::GetTypeId() == (*components)[i].m_typeId) {
                return ComponentHandle<ComponentType>((*components)[i].m_handle);
            }
        }
    }

    return ComponentHandle<ComponentType>();
}

template<typename ComponentType, typename DerivedComponentType>
ComponentHandle<ComponentType> EntitySystem::GetDerivedComponent(const Entity* entity)
{
    EntityComponents& entityComponents = GetEntityComponents();
    Components* components = entityComponents.Find(entity);
    if (0 != components) {
        for (SizeType i = 0; i < components->Size(); ++i) {
            if (DerivedComponentType::GetTypeId() == (*components)[i].m_typeId) {
                return ComponentHandle<ComponentType>((*components)[i].m_handle);
            }
        }
    }

    return ComponentHandle<ComponentType>();
}

template<typename ComponentType>
void EntitySystem::GetComponents(const Entity* entity, Internal::Vector<ComponentHandle<ComponentType> >& componentContainer)
{
    EntityComponents& entityComponents = GetEntityComponents();
    Components* components = entityComponents.Find(entity);
    if (0 != components) {
        for (SizeType i = 0; i < components->Size(); ++i) {
            if (ComponentType::GetTypeId() == (*components)[i].m_typeId) {
                componentContainer.Add((*components)[i].m_handle);
            }
        }
    }
}

template<typename ComponentType, typename DerivedComponentType>
void EntitySystem::GetDerivedComponents(const Entity* entity, Internal::Vector<ComponentHandle<ComponentType> >& componentContainer)
{
    EntityComponents& entityComponents = GetEntityComponents();
    Components* components = entityComponents.Find(entity);
    if (0 != components) {
        for (SizeType i = 0; i < components->Size(); ++i) {
            if (DerivedComponentType::GetTypeId() == (*components)[i].m_typeId) {
                componentContainer.Add((*components)[i].m_handle);
            }
        }
    }
}

template<typename ComponentType>
EntitySystem::ComponentIterator<ComponentType>::ComponentIterator(const Entity* entity)
    :
    m_index(0)
{
    GetComponents<ComponentType>(entity, m_componentContainer);
}

template<typename ComponentType>
bool EntitySystem::ComponentIterator<ComponentType>::Get(ComponentHandle<ComponentType>& handle) const
{
    if (m_index < m_componentContainer.Size()) {
        handle = m_componentContainer[m_index];
        return true;
    }

    handle = ComponentHandle<ComponentType>();
    return false;
}

/** @} */ // end of EntityComponentSystem

} // namespace EntityComponentSystem

} // namespace Candera

#endif
