//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_ENTITYCOMPONENTSYSTEM_COMPONENTSYSTEM_H
#define CANDERA_ENTITYCOMPONENTSYSTEM_COMPONENTSYSTEM_H

#include <Candera/System/EntityComponentSystem/AbstractComponentSystem.h>
#include <Candera/System/EntityComponentSystem/ComponentHandle.h>
#include <Candera/System/Container/Vector.h>
#include <FeatStd/Util/NumericUtil.h>

namespace Candera {

namespace EntityComponentSystem {

/** @addtogroup EntityComponentSystem
 *  @{
 */

template<typename ComponentType, typename EntityType> class ComponentPointer;

/**
 * @brief  A ComponentSystem is an implementation of an AbstractComponentSystem that can be used with the EntitySystem.
 *         It is also a factory for a Component derived class that can be attached to any Entity.
 *         Component handles are used for referencing components. Their purpose is to avoid dangling pointers.
 *         Handles can be copied and assigned arbitrarily. They can be resolved using a ComponentPointer, which can only
 *         be dereferenced for use, but not copied or assigned. Resolving a handle to a pointer is a lightweight operation,
 *         which only requires two 'if' and a read from memory. A dangling handle (i.e. its referenced component has been
 *         destroyed) will resolve to a null pointer.
 *         For general information about the Entity Component System see EntityComponentSystem.h
 *
 * @tparam ComponentType Type of the Component derived object this component system can create, manage, and process.
 * @tparam EntityType Type of the Entity derived object this component systems's components can be attached to.
 */

template<typename ComponentType, typename EntityType>
class ComponentSystem : public AbstractComponentSystem
{
    FEATSTD_TYPEDEF_BASE(AbstractComponentSystem);

public:

    FEATSTD_LINT_NEXT_EXPRESSION(578, "Violates MISRA C++ 2008 Required Rule 2-10-2: ComponentSystem<>::Handle hides FeatStd::Handle")
    typedef ComponentHandle<ComponentType> Handle;
    typedef ComponentPointer<ComponentType, EntityType> Pointer;

    /**
     *  Create a component and return its handle.
     *  @return  The handle of the created component. If the creation failed, a null handle will be returned.
     */
    Handle CreateComponent();

    /**
     *  Destroy the component associated with the given handle.
     *  @param handle  The handle to identify the component.
     *  @return  True, if the component was successfully destroyed.
     *           False, if the operation failed, or if the handle is dangling.
     */
    bool DestroyComponent(const Handle handle);

    /**
     *  Attach the component identified by the handle to the given entity.
     *  @param handle  The handle to identify the component.
     *  @param entity  The entity to attach the component to.
     *  @return  True, if the component was successfully attached to the given entity.
     *           False, if the operation failed, or if the component is already attached to a different entity,
     *           or if the entity already has a component of the same type attached, or if the handle is dangling.
     */
    bool AttachComponent(const Handle handle, EntityType* const entity);

    /**
     *  Detach the component identified by the handle from its attached entity.
     *  @param handle  The handle to identify the component.
     *  @return  True, if the component was successfully detached from its attached entity.
     *           False, if the operation failed, or if the handle is dangling.
     */
    bool DetachComponent(const Handle handle);

    /** 
     *  Get the typeId of the component associated with this component system.
     *  @return  TypeId of the component associated with this component system.
     */
    virtual TypeId GetComponentTypeId() const override { return ComponentType::GetTypeId(); };

protected:
    /**
     *  Constructor
     */
    ComponentSystem();

    /**
     *  Destructor
     */
    virtual ~ComponentSystem() override;

    /**
     *  Callback for derived systems to perform actions after a component was attached.
     *  @param handle  The handle of the component that was attached.
     *  @param entity  The entity the component was attached to.
     *  @return  True, if the derived system successfully performed its actions, false otherwise.
     */
    virtual bool OnAttachComponent(Handle handle, EntityType* const entity) { FEATSTD_UNUSED(handle); FEATSTD_UNUSED(entity); return true; }

    /**
     *  Callback for derived systems to perform actions after a component was detached.
     *  @param handle  The handle of the component that was detached.
     *  @return  True, if the derived system successfully performed its actions, false otherwise.
     */
    virtual bool OnDetachComponent(Handle handle) { FEATSTD_UNUSED(handle); return true; }

    /**
     *  Creates a handle for the given component.
     *  @param component  The component to create the handle for.
     *  @return  The handle that was created for the component.
     */
    Handle CreateHandle(ComponentType* component);

    /**
     *  Removes the handle of a component. After the operation this or any duplicate handles
     *  will be dangling, and resolve to a null pointer.
     *  @param handle  The handle to remove.
     */
    void Remove(Handle handle);

    // Continuous array of components
    Internal::Vector<ComponentType*> m_components;

    /**
     *  Get the pointer from the handle.
     *  Important note: DO NOT keep copies of pointers. The point of using handles is to avoid dangling pointers.
     *  Therefore, keep as many copies of handles as required, but always resolve via GetPointer() before
     *  actual use. If a handle is dangling (i.e. the component it is referring to was destroyed), it will
     *  resolve to a null pointer which can be checked.
     *  @param handle  The handle to get the pointer for.
     *  @return  The pointer associated with the handle, or 0 if the handle is dangling.
     */
    inline ComponentType* GetPointer(const Handle handle) const;

    /**
     *  Gets the handle of the given component.
     *  @param component  The component to get the handle of.
     *  @return  The handle of the component.
     */
    Handle GetHandle(const ComponentType* component) const;

    /**
     *  Casts an unsigned integer to a component handle.
     *  @param handleRaw  The unsigned integer to cast to a component handle.
     *  @return  The handle of the component.
     */
    Handle CastToHandle(UInt32 handleRaw) const;

    /**
     *  Gets the component handle as an unsigned integer.
     *  @param handle  The component handle to get as an unsigned integer.
     *  @return  The unsigned integer that represents the component handle.
     */
    UInt32 GetHandleRaw(const Handle handle) const { return static_cast<UInt32>(handle); }

private:
    /**
     *  Deregister the entity from the component system. This also destroys any associated components.
     *  @param entity  The entity to be deregistered.
     */
    virtual void DeregisterEntity(const Entity* entity);

    /**
     *  Query if the given handle refers to a component of the system and is of the given type (which can be
     *  a derived type).
     *  @param handle           The handle to perform the query on.
     *  @param componentTypeId  The type id to query with the handle.
     *  @return  True, if the given handle refers to a component of the system and is of the given type.
     *           False, if the handle does not refer to a component of the system, or if the component is not
     *           of the given type.
     */
    virtual bool IsComponentTypeId(const UInt32 handle, TypeId componentTypeId) const;

    struct ComponentHandleEntry
    {
        explicit ComponentHandleEntry(UInt32 nextFreeIndex, UInt32 counter = 0)
            : m_nextFreeIndex(nextFreeIndex), m_counter(counter), m_component(0) {}

        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1960, UInt32 is an explicit unsigned integer)
            UInt32 m_nextFreeIndex : Handle::INDEX;
        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1960, UInt32 is an explicit unsigned integer)
            UInt32 m_counter : Handle::COUNTER;
        ComponentType* m_component;
    };

    Internal::Vector<ComponentHandleEntry> m_handleEntries;
    SizeType m_firstFreeHandleEntry;

    // ComponentPointer uses GetPointer() to expose non-copyable pointer
    friend class ComponentPointer<ComponentType, EntityType>;
};

template<typename ComponentType, typename EntityType>
ComponentSystem<ComponentType, EntityType>::ComponentSystem()
    :
    m_firstFreeHandleEntry(0)
{
    CreateHandle(0); // create the 'null' handle first
}

template<typename ComponentType, typename EntityType>
ComponentSystem<ComponentType, EntityType>::~ComponentSystem()
{
    for (SizeType i = 0; i < m_components.Size(); ++i) {
        FEATSTD_DEBUG_ASSERT(0 != m_components[i]);
        EntityType* entity = m_components[i]->m_entity;
        if (0 != entity) {
            static_cast<void>(DetachComponentFrom(entity, GetHandle(m_components[i]), m_components[i]->GetDynamicTypeId()));
        }

        FEATSTD_DELETE(m_components[i]);
    }

    m_components.Clear();
}

template<typename ComponentType, typename EntityType> inline
ComponentType* ComponentSystem<ComponentType, EntityType>::GetPointer(const Handle handle) const
{
    FEATSTD_DEBUG_ASSERT(m_wasCreatedbyEntitySystem); // A component system must be created using EntitySystem::Create().

    if (handle.GetIndex() < FeatStd::Internal::NumericConversion<UInt32>(m_handleEntries.Size())) {
        const ComponentHandleEntry& handleEntry = m_handleEntries[handle.GetIndex()];
        if (handleEntry.m_counter == handle.GetCounter()) {
            return handleEntry.m_component;
        }
    }

    return 0;
}

template<typename ComponentType, typename EntityType>
ComponentHandle<ComponentType> ComponentSystem<ComponentType, EntityType>::CreateComponent()
{
    FEATSTD_DEBUG_ASSERT(m_wasCreatedbyEntitySystem); // A component system must be created using EntitySystem::Create().

    Handle handle;
    ComponentType* component = FEATSTD_NEW(ComponentType);
    if (0 != component) {
        handle = CreateHandle(component);
        if (handle.IsNullHandle()) {
            FEATSTD_DELETE(component);
            return handle;
        }

        if (!m_components.Add(component)) {
            Remove(handle);
            handle = Handle();
            FEATSTD_DELETE(component);
            component = 0;
        }
    }

    return handle;
}

template<typename ComponentType, typename EntityType>
bool ComponentSystem<ComponentType, EntityType>::DestroyComponent(const Handle handle)
{
    ComponentType* component = GetPointer(handle);
    if (0 == component) {
        return false;
    }

    for (SizeType i = 0; i < m_components.Size(); ++i) {
        if (component == m_components[i]) {
            bool success = m_components.Remove(i);
            EntityType* entity = component->m_entity;
            if (0 != entity) {
                success = DetachComponentFrom(entity, handle, component->GetDynamicTypeId()) && success;
                component->m_entity = 0;
                success = OnDetachComponent(handle) && success;
            }

            Remove(handle);
            FEATSTD_DELETE(component);
            return success;
        }
    }

    return false;
}

template<typename ComponentType, typename EntityType>
bool ComponentSystem<ComponentType, EntityType>::AttachComponent(const Handle handle, EntityType* const entity)
{
    ComponentType* component = GetPointer(handle);
    if (0 == component) {
        return false;
    }

    for (SizeType i = 0; i < m_components.Size(); ++i) {
        if (component == m_components[i]) {
            if (0 == component->m_entity) {
                if (AttachComponentTo(entity, handle, component->GetDynamicTypeId())) {
                    component->m_entity = entity;
                    return OnAttachComponent(handle, entity);
                }
            }
            else {
                return false; // component was already attached to an entity, remove it first!
            }
        }
    }

    // component was not found, which means it was not created with CreateComponent()
    return false;
}

template<typename ComponentType, typename EntityType>
bool ComponentSystem<ComponentType, EntityType>::DetachComponent(const Handle handle)
{
    ComponentType* component = GetPointer(handle);
    if (0 == component) {
        return false;
    }

    for (SizeType i = 0; i < m_components.Size(); ++i) {
        if (component == m_components[i]) {
            EntityType* const entity = component->m_entity;
            if (0 != entity) {
                component->m_entity = 0;
                bool result = DetachComponentFrom(entity, handle, component->GetDynamicTypeId());
                return OnDetachComponent(handle) && result;
            }
            else {
                return false; // component was not attached
            }
        }
    }

    // component was not found, which means it was not created with CreateComponent()
    return false;
}

template<typename ComponentType, typename EntityType>
bool ComponentSystem<ComponentType, EntityType>::IsComponentTypeId(const UInt32 handle, TypeId componentTypeId) const
{
    ComponentType* component = GetPointer(handle);
    if (0 == component) {
        return false;
    }

    return component->IsTypeOf(componentTypeId);
}

template<typename ComponentType, typename EntityType>
void ComponentSystem<ComponentType, EntityType>::DeregisterEntity(const Entity* entity)
{
    for (SizeType i = 0; i < m_components.Size(); ++i) {
        ComponentType* component = m_components[i];
        FEATSTD_DEBUG_ASSERT(0 != component);
        if (entity == component->m_entity) {
            component->m_entity = 0;
            Handle handle = GetHandle(component);
            static_cast<void>(OnDetachComponent(handle));
            Remove(handle);
            static_cast<void>(m_components.Remove(i));
            FEATSTD_DELETE(component);
            return;
        }
    }
}

template<typename ComponentType, typename EntityType>
ComponentHandle<ComponentType> ComponentSystem<ComponentType, EntityType>::CreateHandle(ComponentType* component)
{
    if (m_handleEntries.Size() <= m_firstFreeHandleEntry) {
        const UInt32 newIndex = FeatStd::Internal::NumericConversion<UInt32>(m_handleEntries.Size());
        const UInt32 nextFreeIndex = newIndex + 1;

        ComponentHandleEntry handleEntry(nextFreeIndex);
        handleEntry.m_component = component;
        m_firstFreeHandleEntry = nextFreeIndex;
        return (m_handleEntries.Add(handleEntry)) ? Handle(newIndex, 0) : Handle(0, 0);
    }

    const UInt32 newIndex = FeatStd::Internal::NumericConversion<UInt32>(m_firstFreeHandleEntry);
    m_firstFreeHandleEntry = m_handleEntries[newIndex].m_nextFreeIndex;
    ++(m_handleEntries[newIndex].m_counter);
    m_handleEntries[newIndex].m_component = component;

    return Handle(newIndex, m_handleEntries[newIndex].m_counter);
}

template<typename ComponentType, typename EntityType>
ComponentHandle<ComponentType> ComponentSystem<ComponentType, EntityType>::GetHandle(const ComponentType* component) const
{
    for (UInt32 i = 0; i < FeatStd::Internal::NumericConversion<UInt32>(m_handleEntries.Size()); ++i) {
        if (m_handleEntries[i].m_component == component) {
            return Handle(i, m_handleEntries[i].m_counter);
        }
    }

    return Handle();
}

template<typename ComponentType, typename EntityType>
ComponentHandle<ComponentType> ComponentSystem<ComponentType, EntityType>::CastToHandle(UInt32 handleRaw) const
{
    Handle handle(handleRaw);
    if (handle.GetIndex() >= static_cast<UInt32>(m_handleEntries.Size())) {
        // the given raw 'handle' was at no point in time a valid handle
        return Handle();
    }

    return handle;
}

template<typename ComponentType, typename EntityType>
void ComponentSystem<ComponentType, EntityType>::Remove(Handle handle)
{
    const SizeType index = static_cast<SizeType>(handle.GetIndex());
    if (0 == index) {
        return;
    }

    FEATSTD_DEBUG_ASSERT(index < m_handleEntries.Size());
    FEATSTD_DEBUG_ASSERT(m_handleEntries[index].m_counter == handle.GetCounter());
    FEATSTD_DEBUG_ASSERT(0 != m_handleEntries[index].m_component);

    m_handleEntries[index].m_nextFreeIndex = m_firstFreeHandleEntry;
    m_handleEntries[index].m_component = 0;
    m_firstFreeHandleEntry = index;
}

/** @} */ // end of EntityComponentSystem

} // namespace EntityComponentSystem

} // namespace Candera
#endif
