//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_ENTITYCOMPONENTSYSTEM_COMPONENTHANDLE_H
#define CANDERA_ENTITYCOMPONENTSYSTEM_COMPONENTHANDLE_H

#include <CanderaPlatform/OS/CanderaTypes.h>

namespace Candera {

namespace EntityComponentSystem {

/** @addtogroup EntityComponentSystem
 *  @{
 */

/**
 * @brief  ComponentHandle abstracts pointers to Components to avoid dangling pointers.
 *         For general information about the Entity Component System see EntityComponentSystem.h
 *
 * @tparam ComponentType  Type of the Component derived object this component handle can encapsulate pointers for.
 */

template<typename ComponentType>
class ComponentHandle
{
public:
    /**
     *  Constructor creating 'null' handle
     */
    ComponentHandle() : m_index(0), m_counter(0) {}

    /**
     *  Check if the handle is the designated 'null' handle.
     *  @return  True, if the handle is a 'null' handle.
     *           False, if the handle is not a 'null' handle.
     */
    bool IsNullHandle() const { return ((m_index == 0) && (m_counter == 0)); }

    /**
     *  Compare the component handles for equality.
     *  @param other  The second component handle
     *  @return  True, if equal, false otherwise.
     */
    bool operator== (const ComponentHandle& other) const { return (static_cast<UInt32>(*this) == static_cast<UInt32>(other)); }

    /**
     *  Compare the component handles for inequality.
     *  @param other  The second component handle
     *  @return  True, if not equal, false otherwise.
     */
    bool operator!= (const ComponentHandle& other) const { return (static_cast<UInt32>(*this) != static_cast<UInt32>(other)); }

protected:
    enum Bits {
        INDEX = 16,
        COUNTER = 16
    };

    /**
     *  Constructor
     */
    ComponentHandle(UInt32 index, UInt32 counter) : m_index(index), m_counter(counter) {}

    /**
     *  Constructor casting UInt32 to handle
     */
    ComponentHandle(UInt32 handle) : m_index(handle & ((1 << INDEX) - 1)), m_counter(handle >> INDEX) {}

    /**
     *  Operator casting handle to UInt32
     */
    inline operator UInt32() const;

    /**
     *  Get the index that is encoded in the handle.
     */
    UInt32 GetIndex() const { return m_index; }

    /**
     *  Get the counter that is encoded in the handle.
     */
    UInt32 GetCounter() const { return m_counter; }

private:
    template<typename ComponentTypeAlt, typename EntityType> friend class ComponentSystem; // exclusively creates handles.
    friend class EntitySystem; // converts UInt32 to valid handles.

    UInt32 m_index : INDEX;
    UInt32 m_counter : COUNTER;
};

template<typename ComponentType>
ComponentHandle<ComponentType>::operator UInt32() const
{
    return m_counter << INDEX | m_index;
}

/** @} */ // end of EntityComponentSystem

} // namespace EntityComponentSystem

} // namespace Candera

#endif
