//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_ENTITYCOMPONENTSYSTEM_ABSTRACTCOMPONENTSYSTEM_H
#define CANDERA_ENTITYCOMPONENTSYSTEM_ABSTRACTCOMPONENTSYSTEM_H

#include <Candera/System/EntityComponentSystem/Entity.h>
#include <Candera/System/Rtti/Rtti.h>
#include <CanderaPlatform/OS/CanderaTypes.h>

namespace Candera {

class Event;

namespace EntityComponentSystem {

/** @addtogroup EntityComponentSystem
 *  @{
 */

/**
 * @brief  AbstractComponentSystem is the base class for component systems that can be used with the EntitySystem.
 *         For general information about the Entity Component System see EntityComponentSystem.h
 */

class AbstractComponentSystem : public Entity
{
public:
    /**
     *  Destructor
     */
    virtual ~AbstractComponentSystem() {}

    /**
     *  Receives an event that the derived component system can process.
     *  @param event  The event to be processed.
     */
    virtual void OnEvent(const Event&) {}

    /**
     *  Get the RTTI TypeId for the component that is associated with the derived component system.
     *  @return  Returns the TypeId for the component that is associated with the derived component system.
     */
    virtual TypeId GetComponentTypeId() const = 0;

    /**
     *  Regularly called by the entity system to provide a way for the derived component system to perform
     *  necessary tasks.
     */
    virtual void HeartBeat() {}

    /**
     *  If the component system is enabled, this function is called regularly by the entity system.
     */
    virtual void Update() {}

    /**
     *  Get the priority of the component system.
     *  @return  The priority of the component system.
     */
    Float GetPriority() const { return m_priority; }

    /**
     *  Set the priority of the component system.
     *  @param priority  The priority of the component system to be set.
     */
    void SetPriority(Float priority);

    /**
     *  Enable or disable the component system. Disabled component systems are not called via Update().
     *  @param isEnabled  The flag if this component system is enabled or disabled.
     */
    void SetEnabled(bool isEnabled) { m_isEnabled = isEnabled; }

    /**
     *  Returns if the component system is enabled or disabled.
     *  @return  True, if the component system is enabled.
     *           False, if the component system is disabled.
     */
    bool IsEnabled() const { return m_isEnabled; }

    FEATSTD_RTTI_DECLARATION();

protected:
    /**
     *  Constructor
     */
    AbstractComponentSystem();

    /**
     *  Attaches a component identified by its handle and type for the given entity.
     *  @param entity  The entity to attach the component to.
     *  @param handle  The handle of the component to attach.
     *  @param typeId  The typeId of the component to attach.
     *  @return  True, if the component was successfully attached to the entity.
     *           False, if the operation failed, or there is already a component of the same type
     *           attached to the given entity.
     */
    bool AttachComponentTo(const Entity* entity, const UInt32 handle, const TypeId typeId) const;

    /**
     *  Detaches a component identified by its type from the given entity.
     *  @param entity  The entity to detach the component from.
     *  @param handle  The handle of the component to detach.
     *  @param typeId  The typeId of the component to detach.
     *  @return  True, if the component was successfully detached from the entity.
     *           False, if the operation failed, or there was no component of the given typeId
     *           attached to the given entity.
     */
    bool DetachComponentFrom(const Entity* entity, const UInt32 handle, const TypeId typeId) const;

    /**
     *  Deregister the entity from the component system. This also destroys any associated components.
     *  @param entity  The entity to be deregistered.
     */
    virtual void DeregisterEntity(const Entity* entity) = 0;

    /**
     *  Query if the given handle refers to a component of the system and is of the given type (which can be
     *  a derived type).
     *  @param handle           The handle to perform the query on.
     *  @param componentTypeId  The type id to query with the handle.
     *  @return  True, if the given handle refers to a component of the system and is of the given type.
     *           False, if the handle does not refer to a component of the system, or if the component is not
     *           of the given type.
     */
    virtual bool IsComponentTypeId(const UInt32 handle, TypeId componentTypeId) const = 0;

    // A flag to catch component systems that were created manually instead of using the EntitySystem.
    bool m_wasCreatedbyEntitySystem;

private:

    friend class EntitySystem;     // Calls DeregisterEntity(), accesses m_wasCreatedbyEntitySystem

    Float m_priority;
    bool m_isEnabled;
};

/** @} */ // end of EntityComponentSystem

} // namespace EntityComponentSystem

} // namespace Candera

#endif
