//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef VideoMemoryStatistic_h
#define VideoMemoryStatistic_h

#include <Candera/Environment.h>

#include <Candera/System/Container/SingleLinkedList.h>
#include <Candera/Engine2D/Core/BitmapImage2D.h>

namespace FeatStd { namespace Diagnostics {
    class Measurable;
}}

namespace Candera { namespace Diagnostics {
/// @addtogroup DiagnosticsSystem
/// @{

/**
 * Interfaces for retrieving current and maximum size of content uploaded to VRAM
 */
    class VideoMemoryStatistic
    {
    public:
        friend class Candera::BitmapImage2D;
        /**
         * Retrieve current size of texture images uploaded to VRAM in bytes.
         * @return Current size of texture images uploaded to VRAM in bytes.
         */
        static UInt GetCurrentTextureImageSizeUploaded();

        /**
         * Retrieve maximum size of texture images uploaded to VRAM in bytes since start or last reset.
         * @return Maximum size of texture images uploaded to VRAM in bytes since start or last reset.
         */
        static UInt GetMaxTextureImageSizeUploaded() { return m_maxTextureImageSize; }

        /**
         * Retrieve current size of vertex buffers uploaded to VRAM in bytes.
         * @return Current size of vertex buffers uploaded to VRAM in bytes.
         */
        static UInt GetCurrentVertexBufferSizeUploaded();

        /**
         * Retrieve maximum size of vertex buffers uploaded to VRAM in bytes since start or last reset.
         * @return Maximum size of vertex buffers uploaded to VRAM in bytes since start or last reset.
         */
        static UInt GetMaxVertexBufferSizeUploaded() { return m_maxVertexBufferSize; }

        /**
         * Retrieve current size of frame buffer objects uploaded to VRAM in bytes.
         * @return Current size of frame buffer objects uploaded to VRAM in bytes.
         */
        static UInt GetCurrentFrameBufferObjectSizeUploaded();

        /**
         * Retrieve maximum size of frame buffers objects uploaded to VRAM in bytes since start or last reset.
         * @return Maximum size of frame buffers objects uploaded to VRAM in bytes since start or last reset.
         */
        static UInt GetMaxFrameBufferObjectSizeUploaded() { return m_maxFrameBufferObjectSize; }

        /**
         * Retrieve current size of window surfaces uploaded to VRAM in bytes.
         * @return Current size of window surfaces uploaded to VRAM in bytes.
         */
        static UInt GetCurrentWindowSurfaceSizeUploaded();

        /**
         * Retrieve maximum size of window surfaces uploaded to VRAM in bytes since start or last reset.
         * @return Maximum size of window surfaces uploaded to VRAM in bytes since start or last reset.
         */
        static UInt GetMaxWindowSurfaceSizeUploaded() { return m_maxWindowSurfaceSize; }


        /**
         * Retrieve current size of 2D BitmapImages uploaded to VRAM in bytes.
         * Note: BitmapImage2D are also used as glyph caches, which are therefore also considered.
         * @return Current size of 2D BitmapImages uploaded to VRAM in bytes.
         */
        static UInt GetCurrentBitmapImage2DSizeUploaded();

        /**
         * Retrieve maximum size of 2D BitmapImages uploaded to VRAM in bytes since start or last reset.
         * Note: BitmapImage2D are also used as glyph caches, which are therefore also considered.
         * @return Maximum size of 2D BitmapImages uploaded to VRAM in bytes since start or last reset.
         */
        static UInt GetMaxBitmapImage2DSizeUploaded() { return m_maxBitmapImage2DSize; }

        /**
         * Resets all statistics to default values.
         */
        static void Reset();

        // Texture related event functions.
        static void OnTextureImageUploaded(const FeatStd::Diagnostics::Measurable& measurable);
        static void OnTextureImageUnloaded(const FeatStd::Diagnostics::Measurable& measurable);
        static void AddTextureMemoryUsed(UInt size) { m_additionalTextureMemoryUsed += size; }
        static void SubtractTextureMemoryUsed(UInt size) { m_additionalTextureMemoryUsed -= size; };
        // Vertex buffer related event functions.
        static void OnVertexBufferUploaded(const FeatStd::Diagnostics::Measurable& measurable);
        static void OnVertexBufferUnloaded(const FeatStd::Diagnostics::Measurable& measurable);
        // Frame buffer object related event functions.
        static void OnFrameBufferObjectUploaded(const FeatStd::Diagnostics::Measurable& measurable);
        static void OnFrameBufferObjectUnloaded(const FeatStd::Diagnostics::Measurable& measurable);
        // Window surface related event functions.
        static void OnWindowSurfaceUploaded(const FeatStd::Diagnostics::Measurable& measurable);
        static void OnWindowSurfaceUnloaded(const FeatStd::Diagnostics::Measurable& measurable);
        // BitmapImage2D related event functions.
        static void OnBitmapImage2DUploaded(const FeatStd::Diagnostics::Measurable& measurable);
        static void OnBitmapImage2DUnloaded(const FeatStd::Diagnostics::Measurable& measurable);



    private:
        struct VramInfo
        {
            const FeatStd::Diagnostics::Measurable* measurable;
            UInt size;

            VramInfo() : measurable(0), size(0) { }
            VramInfo(const FeatStd::Diagnostics::Measurable* initMeasurable, UInt initSize)
              : measurable(initMeasurable), size(initSize) { }
            ~VramInfo() { }
            // operator== utilized by VramInfoList::remove() to compare handles.
            bool operator==(const VramInfo &rhs) const { return measurable == rhs.measurable; }
        };

        typedef Candera::Internal::SingleLinkedList<VramInfo> VramInfoList;

        static VramInfoList m_textureList;
        static UInt m_maxTextureImageSize;
        static UInt m_additionalTextureMemoryUsed;

        static VramInfoList m_vertexBufferList;
        static UInt m_maxVertexBufferSize;

        static VramInfoList m_frameBufferObjectList;
        static UInt m_maxFrameBufferObjectSize;

        static  VramInfoList m_windowSurfaceList;
        static UInt m_maxWindowSurfaceSize;

        static VramInfoList m_bitmapImage2DList;
        static UInt m_maxBitmapImage2DSize;

        static bool m_lockTextureImageStatistics;


        /**
        * 2D rendering is realized using Candera 3D on certain platform. In this case,
        * an image's VRAM size is counted as BitmapImage2D and BitmapTextureImage.
        * To prevent this, a flag can be set, that TextureImages are not counted for the VRAM statistics.
        * @param lock Boolean value to lock (true) or unlock (false) the collection of VRAM statistics for TextureImages.
        */
        static void SetTextureImageStatisticLocked(bool lock) { m_lockTextureImageStatistics = lock; }

        /**
        * 2D rendering is realized using Candera 3D on certain platform. In this case,
        * an image's VRAM size is counted as BitmapImage2D and BitmapTextureImage.
        * This function returns a flag that states if TextureImages are  counted or not for the VRAM statistics..
        * @return A flag that states if TextureImages are locked and not counted (true) or unlocked(true) for the VRAM statistics.
        */
        static bool IsTextureImageStatisticLocked() { return m_lockTextureImageStatistics; }

        FEATSTD_MAKE_CLASS_STATIC(VideoMemoryStatistic);
};
/// @}
}}
#endif
