//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "VideoMemoryStatistic.h"

#include <FeatStd/Diagnostics/Measurable.h>

namespace Candera { namespace Diagnostics {

using FeatStd::Diagnostics::Measurable;

VideoMemoryStatistic::VramInfoList VideoMemoryStatistic::m_textureList;
UInt VideoMemoryStatistic::m_maxTextureImageSize = 0;
UInt VideoMemoryStatistic::m_additionalTextureMemoryUsed = 0;

VideoMemoryStatistic::VramInfoList VideoMemoryStatistic::m_vertexBufferList;
UInt VideoMemoryStatistic::m_maxVertexBufferSize = 0;

VideoMemoryStatistic::VramInfoList VideoMemoryStatistic::m_frameBufferObjectList;
UInt VideoMemoryStatistic::m_maxFrameBufferObjectSize = 0;

VideoMemoryStatistic::VramInfoList VideoMemoryStatistic::m_windowSurfaceList;
UInt VideoMemoryStatistic::m_maxWindowSurfaceSize = 0;

VideoMemoryStatistic::VramInfoList VideoMemoryStatistic::m_bitmapImage2DList;
UInt VideoMemoryStatistic::m_maxBitmapImage2DSize = 0;

bool VideoMemoryStatistic::m_lockTextureImageStatistics = false;

void VideoMemoryStatistic::Reset()
{
    m_textureList.Clear();
    m_maxTextureImageSize = 0;
    m_additionalTextureMemoryUsed = 0;

    m_vertexBufferList.Clear();
    m_maxVertexBufferSize = 0;

    m_frameBufferObjectList.Clear();
    m_maxFrameBufferObjectSize = 0;

    m_windowSurfaceList.Clear();
    m_maxWindowSurfaceSize = 0;

    m_bitmapImage2DList.Clear();
    m_maxBitmapImage2DSize = 0;

}

void VideoMemoryStatistic::OnTextureImageUploaded(const Measurable& measurable)
{
#ifdef CANDERA_VIDEO_MEMORY_STATISTIC_ENABLED
    if (!IsTextureImageStatisticLocked()) {
        VramInfo vramInfo(&measurable, measurable.GetSize());
        static_cast<void>(m_textureList.Prepend(vramInfo));

        // Finally check if current size uploaded increases the max. size of textures uploaded yet.
        UInt curTexSize = GetCurrentTextureImageSizeUploaded();
        if (curTexSize > m_maxTextureImageSize) {
            m_maxTextureImageSize = curTexSize;
        }
    }
#else
    FEATSTD_UNUSED(measurable);
#endif
}

void VideoMemoryStatistic::OnTextureImageUnloaded(const Measurable& measurable)
{
#ifdef CANDERA_VIDEO_MEMORY_STATISTIC_ENABLED
    // Find texture and reset size to zero if found.
    VramInfoList::Iterator it = m_textureList.Begin();
    while((it != m_textureList.End()) && (&measurable != (*it).measurable)) {
        ++it;
    }
    if (it != m_textureList.End()) {
        // texture handle matches given texture image handle.
        static_cast<void>(m_textureList.Remove(*it));
    }
#else
    FEATSTD_UNUSED(measurable);
#endif
}

UInt VideoMemoryStatistic::GetCurrentTextureImageSizeUploaded()
{
    UInt sum = 0;
    for(VramInfoList::Iterator it = m_textureList.Begin(); it != m_textureList.End(); ++it) {
        sum += (*it).size ;
    }
    return sum + m_additionalTextureMemoryUsed;
}

void VideoMemoryStatistic::OnVertexBufferUploaded(const Measurable& measurable)
{
#ifdef CANDERA_VIDEO_MEMORY_STATISTIC_ENABLED
    VramInfo vramInfo(&measurable, measurable.GetSize());
    static_cast<void>(m_vertexBufferList.Prepend(vramInfo));

    // Finally check if current size uploaded increases the max. size of vertex buffer uploaded yet.
    UInt curVertexBufferSize = GetCurrentVertexBufferSizeUploaded();
    if ( curVertexBufferSize > m_maxVertexBufferSize) {
        m_maxVertexBufferSize = curVertexBufferSize;
    }
#else
    FEATSTD_UNUSED(measurable);
#endif
}

void VideoMemoryStatistic::OnVertexBufferUnloaded(const Measurable& measurable)
{
#ifdef CANDERA_VIDEO_MEMORY_STATISTIC_ENABLED
    // Find vertex buffer and reset size to zero if found.
    VramInfoList::Iterator it = m_vertexBufferList.Begin();
    while((it != m_vertexBufferList.End()) && (&measurable != (*it).measurable)) {
        ++it;
    }
    if (it != m_vertexBufferList.End()) {
        // vertex buffer handle matches given vertex buffer handle.
        static_cast<void>(m_vertexBufferList.Remove(*it));
    }
#else
    FEATSTD_UNUSED(measurable);
#endif
}

UInt VideoMemoryStatistic::GetCurrentVertexBufferSizeUploaded()
{
    UInt sum = 0;
    for(VramInfoList::Iterator it = m_vertexBufferList.Begin(); it != m_vertexBufferList.End(); ++it) {
        sum += (*it).size ;
    }
    return sum;
}

void VideoMemoryStatistic::OnFrameBufferObjectUploaded(const Measurable& measurable)
{
#ifdef CANDERA_VIDEO_MEMORY_STATISTIC_ENABLED
     //Check if frame buffer handle already exists in VramInfo.
    VramInfoList::Iterator it = m_frameBufferObjectList.Begin();
    while((it != m_frameBufferObjectList.End()) && (&measurable != (*it).measurable)) {
        ++it;
    }
    if (it != m_frameBufferObjectList.End()) {
        // frame buffer handle already exists in VramInfo.
        (*it).size = measurable.GetSize();
    }
    else {
        // frame buffer handle does not exist in VramInfo yet.
        VramInfo vramInfo(&measurable, static_cast<UInt>(measurable.GetSize()));
        static_cast<void>(m_frameBufferObjectList.Prepend(vramInfo));
    }

    // Finally check if current size uploaded increases the max. size of frame buffer objects uploaded yet.
    UInt curFrameBufferObjectSize = GetCurrentFrameBufferObjectSizeUploaded();
    if (curFrameBufferObjectSize > m_maxFrameBufferObjectSize) {
        m_maxFrameBufferObjectSize = curFrameBufferObjectSize;
    }
#else
    FEATSTD_UNUSED(measurable);
#endif
}

void VideoMemoryStatistic::OnFrameBufferObjectUnloaded(const Measurable& measurable)
{
#ifdef CANDERA_VIDEO_MEMORY_STATISTIC_ENABLED
    // Find frame buffer object and reset size to zero if found.
    VramInfoList::Iterator it = m_frameBufferObjectList.Begin();
    while((it != m_frameBufferObjectList.End()) && (&measurable != (*it).measurable)) {
        ++it;
    }
    if (it != m_frameBufferObjectList.End()) {
        // frame buffer object handle matches.
        static_cast<void>(m_frameBufferObjectList.Remove(*it));
    }
#else
    FEATSTD_UNUSED(measurable);
#endif
}

UInt VideoMemoryStatistic::GetCurrentFrameBufferObjectSizeUploaded()
{
    UInt sum = 0;
    for(VramInfoList::Iterator it = m_frameBufferObjectList.Begin(); it != m_frameBufferObjectList.End(); ++it) {
        sum += (*it).size ;
    }
    return sum;
}

void VideoMemoryStatistic::OnWindowSurfaceUploaded(const Measurable& measurable)
{
#ifdef CANDERA_VIDEO_MEMORY_STATISTIC_ENABLED
    //Check if window surface handle already exists in VramInfo.
    VramInfoList::Iterator it = m_windowSurfaceList.Begin();
    while((it != m_windowSurfaceList.End()) && (&measurable != (*it).measurable)) {
        ++it;
    }
    if (it != m_windowSurfaceList.End()) {
        // window surface handle already exists in VramInfo.
        (*it).size = measurable.GetSize();
    }
    else {
        // window surface handle does not exist in VramInfo yet.
        VramInfo vramInfo(&measurable, measurable.GetSize());
        static_cast<void>(m_windowSurfaceList.Prepend(vramInfo));
    }

    // Finally check if current size uploaded increases the max. size of window surfaces uploaded yet.
    UInt curWindowSurfaceSize = GetCurrentWindowSurfaceSizeUploaded();
    if (curWindowSurfaceSize > m_maxWindowSurfaceSize) {
        m_maxWindowSurfaceSize = curWindowSurfaceSize;
    }
#else
    FEATSTD_UNUSED(measurable);
#endif
}

void VideoMemoryStatistic::OnWindowSurfaceUnloaded(const Measurable& measurable)
{
#ifdef CANDERA_VIDEO_MEMORY_STATISTIC_ENABLED
    // Find window surface object and reset size to zero if found.
    VramInfoList::Iterator it = m_windowSurfaceList.Begin();
    while((it != m_windowSurfaceList.End()) && (&measurable != (*it).measurable)) {
        ++it;
    }
    if (it != m_windowSurfaceList.End()) {
        // window surface handle matches.
        static_cast<void>(m_windowSurfaceList.Remove(*it));
    }
#else
    FEATSTD_UNUSED(measurable);
#endif
}

UInt VideoMemoryStatistic::GetCurrentWindowSurfaceSizeUploaded()
{
    UInt sum = 0;
    for(VramInfoList::Iterator it = m_windowSurfaceList.Begin(); it != m_windowSurfaceList.End(); ++it) {
        sum += (*it).size ;
    }
    return sum;
}

void VideoMemoryStatistic::OnBitmapImage2DUploaded(const Measurable& measurable)
{
#ifdef CANDERA_VIDEO_MEMORY_STATISTIC_ENABLED
    //Check if BitmapImage2D handle already exists in VramInfo.
    VramInfoList::Iterator it = m_bitmapImage2DList.Begin();
    while((it != m_bitmapImage2DList.End()) && (&measurable != (*it).measurable)) {
        ++it;
    }
    if (it != m_bitmapImage2DList.End()) {
        // BitmapImage2D handle already exists in VramInfo.
        (*it).size = measurable.GetSize();
    }
    else {
        // BitmapImage2D handle does not exist in VramInfo yet.
        VramInfo vramInfo(&measurable, measurable.GetSize());
        static_cast<void>(m_bitmapImage2DList.Prepend(vramInfo));
    }

    // Finally check if current size uploaded increases the max. size of 2D bitmap images uploaded yet.
    UInt curBitmapImage2DSize = GetCurrentBitmapImage2DSizeUploaded();
    if (curBitmapImage2DSize > m_maxBitmapImage2DSize) {
        m_maxBitmapImage2DSize = curBitmapImage2DSize;
    }
#else
    FEATSTD_UNUSED(measurable);
#endif
}

void VideoMemoryStatistic::OnBitmapImage2DUnloaded(const Measurable& measurable)
{
#ifdef CANDERA_VIDEO_MEMORY_STATISTIC_ENABLED
    // Find window surface object and reset size to zero if found.
    VramInfoList::Iterator it = m_bitmapImage2DList.Begin();
    while((it != m_bitmapImage2DList.End()) && (&measurable != (*it).measurable)) {
        ++it;
    }
    if (it != m_bitmapImage2DList.End()) {
        // window surface handle matches.
        static_cast<void>(m_bitmapImage2DList.Remove(*it));
    }
#else
    FEATSTD_UNUSED(measurable);
#endif
}

UInt VideoMemoryStatistic::GetCurrentBitmapImage2DSizeUploaded()
{
    UInt sum = 0;
    for(VramInfoList::Iterator it = m_bitmapImage2DList.Begin(); it != m_bitmapImage2DList.End(); ++it) {
        sum += (*it).size ;
    }
    return sum;
}

}}
