//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <Candera/System/Diagnostics/ScreenshotTool.h>

#include <FeatStd/Util/StaticObject.h>
#include <FeatStd/Util/PointerUtil.h>
#include <FeatStd/Event/EventListener.h>
#include <FeatStd/Event/Event.h>

#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/EntityComponentSystem/EntitySystem.h>
#include <Candera/System/EntityComponentSystem/EntityComponentSystem.h>
#include <Candera/System/MemoryManagement/CanderaHeap.h>
#include <Candera/System/Diagnostics/Log.h>

#ifdef CANDERA_2D_OVER_3D_ENABLED
#include <CanderaPlatform/Device/Common/Internal/RenderDevice2DOver3D/Context2DOver3DDevicePool.h>
#include <CanderaPlatform/Device/Common/Base/RenderTarget2D.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>
#include <Candera/Engine2D/Core/Camera2D.h>
#include <Candera/Engine2D/Core/Renderer2D.h>
#endif

#ifdef CANDERA_3D_ENABLED
#include <CanderaPlatform/Device/Common/Base/RenderTarget3D.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>
#include <Candera/Engine3D/Core/Renderer.h>
#include <Candera/Engine3D/Core/PixelBuffer.h>
#include <Candera/Engine3D/Core/Camera.h>
#else
#include <CanderaPlatform/Device/Common/Stubs/DummyCandera3D.h>
#endif

using namespace FeatStd;

namespace Candera {
    namespace Diagnostics {
        FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaSystem);

        class ScreenshotTool::ScreenshotPrivateData {
        public:
            ScreenshotPrivateData(UInt8 index,
                                  TScreenshotID const& id,
                                  Int width,
                                  Int height) :
                                  m_screenshotData(id, width, height),
                                  m_pixelBuffer(0),
                                  m_index(index)
            {

#ifdef CANDERA_3D_ENABLED
                //4x because its RGBA format - it wont change - as this is a performance drop on the PBOs
                m_pixelBuffer = CANDERA_NEW(PixelBuffer)(static_cast<SizeType>(width * height * 4), true);
                if (m_pixelBuffer != 0) {
                    if (!m_pixelBuffer->Upload()) {
                        FEATSTD_LOG_WARN("Error uploading PixelBuffer for screenshots!");
                        CANDERA_DELETE(m_pixelBuffer);
                        m_pixelBuffer = 0;
                    }
                }
                else {
                    FEATSTD_LOG_WARN("Error creating buffer for screenshots! Out of memory!");
                }
#endif
            }

            ~ScreenshotPrivateData()
            {
#ifdef CANDERA_3D_ENABLED
                if (m_pixelBuffer != 0) {
                    static_cast<void>(m_pixelBuffer->Unload());
                    CANDERA_DELETE(m_pixelBuffer);
                    m_pixelBuffer = 0;
                }
                if (!m_syncFence.PointsToNull()) {
                    m_syncFence.Release();
                }
                FEATSTD_DEBUG_ASSERT(m_syncFence.PointsToNull());
#endif
            }

            ScreenshotData m_screenshotData;
#ifdef CANDERA_3D_ENABLED
            SynchronizationFence::SharedPointer m_syncFence;
            Candera::PixelBuffer * m_pixelBuffer;
#else
            void * m_pixelBuffer;
#endif
            UInt8 m_index;
        };


#ifdef CANDERA_3D_ENABLED
        static void GetActiveRenderTargetDimension(Int& width, Int& height)
        {
            width = 0;
            height = 0;
#ifdef CANDERA_2D_OVER_3D_ENABLED
            for (SizeType i = 0; i < Renderer2D::GetCameraCount(); i++) {
                Camera2D const* activeCamera2D = Renderer2D::GetCamera(static_cast<UInt32>(i));
                if (activeCamera2D != 0) {
                    const Candera::RenderTarget2D * renderTarget = activeCamera2D->GetRenderTarget();
                    if (renderTarget != 0) {
                        if (width < renderTarget->GetWidth()) {
                            width = renderTarget->GetWidth();
                        }
                        if (height < renderTarget->GetHeight()) {
                            height = renderTarget->GetHeight();
                        }
                    }
                }
            }
#endif
// #ifdef CANDERA_3D_ENABLED
            Camera const* activeCamera = RenderDevice::GetActiveCamera();
            if (activeCamera != 0) {
                Candera::RenderTarget3D * renderTarget = activeCamera->GetRenderTarget();
                if (renderTarget != 0) {
                    width = renderTarget->GetWidth();
                    height = renderTarget->GetHeight();
                }
            }
//#endif
        }
#endif

        ScreenshotTool::ScreenshotData::ScreenshotData(TScreenshotID const& id, Int width, Int height) :
            m_id(id),
            m_data(0),
            m_width(width),
            m_height(height)
        {}

        void ScreenshotTool::TakeScreenshot(ScreenshotTool& customScreenshotTool, TScreenshotID const& id)
        {
            for (UInt8 i = 0; i < 2; i++) {
                // check if there is an empty slot for a screenshot
                if (customScreenshotTool.m_screenshotData[i] == 0) {
                    if (customScreenshotTool.TakeScreenshotStart(id, i)) {
                        if (!customScreenshotTool.m_isAsync) {
                            static_cast<void>(customScreenshotTool.TakeScreenshotResult(i));
                        }
                    }
                    break;
                }
            }
        }

        void ScreenshotTool::TakeScreenshot(TScreenshotID const& id)
        {
            TakeScreenshot(GetInstance(), id);
        }

        ScreenshotTool::ScreenshotTool(bool isAsync) :m_currentIndex(0), m_isAsync(isAsync), m_isInitialized(false)
        {
            for (UInt8 i = 0; i < 2; i++) {
                m_screenshotData[i] = 0;
            }
        }

        ScreenshotTool::ScreenshotTool() :m_currentIndex(0), m_isAsync(false), m_isInitialized(false)
        {
            //  if () {
            m_isAsync = true;
            // }
            // else {
            //     m_isAsync = false;
            // }

            for (UInt8 i = 0; i < 2; i++) {
                m_screenshotData[i] = 0;
            }
        }

        ScreenshotTool::~ScreenshotTool()
        {
            static_cast<void>(DetachDispatcher());
            for (UInt8 i = 0; i < 2; i++) {
                if (m_screenshotData[i] != 0) {
                    CANDERA_DELETE(m_screenshotData[i]);
                }
            }
        }

        void ScreenshotTool::HandleFinishedScreenshots()
        {
            if (m_isAsync) {
                for (UInt8 i = 0; i < 2; i++) {
                    if (IsTakeScreenshotFinished(i)) {
                        static_cast<void>(TakeScreenshotResult(i));
                    }
                }
            }
        }


        void ScreenshotTool::Initialize()
        {
            static_cast<void>(AttachDispatcher());
        }

        bool ScreenshotTool::IsTakeScreenshotFinished(UInt8 screenshotIdx) const
        {
#ifdef CANDERA_3D_ENABLED
            if (!m_isAsync) {
                return false;
            }
            ScreenshotPrivateData * data = m_screenshotData[screenshotIdx];
            if (data != 0) {
                // The IsTakeScreenshotFinished method should only be triggered when the screenshot is taken asynchronously
                FEATSTD_DEBUG_ASSERT(!data->m_syncFence.PointsToNull());
                FEATSTD_DEBUG_ASSERT(m_isAsync);
                if (!data->m_syncFence.PointsToNull()) {
                    return data->m_syncFence->IsProcessed();
                }
            }
#else
            FEATSTD_UNUSED(screenshotIdx);
#endif
            return false;
        }

        bool ScreenshotTool::TakeScreenshotStart(TScreenshotID const& id, UInt8 screenshotIdx)
        {
            if (!m_isInitialized) {
                m_isInitialized = true;
                Initialize();
            }
#ifdef CANDERA_3D_ENABLED
            bool result = true;
            Int width;
            Int height;
            GetActiveRenderTargetDimension(width, height);
            if ((width == 0) || (height == 0)) {
                FEATSTD_LOG_WARN("Error in screenshot: could not find rendertarget dimensions!");
                return false;
            }
            FEATSTD_DEBUG_ASSERT(m_screenshotData[screenshotIdx] == 0);
            m_screenshotData[screenshotIdx] = CANDERA_NEW(ScreenshotPrivateData)(screenshotIdx, id, width, height);
            ScreenshotPrivateData * data = m_screenshotData[screenshotIdx];
            if ((data != 0) && (data->m_pixelBuffer != 0)) {

                if (Renderer::BindPixelBuffer(*(data->m_pixelBuffer))) {
                    if (!RenderDevice::ReadPixels(0, 0, data->m_screenshotData.m_width, data->m_screenshotData.m_height, Bitmap::RgbaUnsignedBytePixelFormat, Bitmap::PackAlignment1, 0)) {
                        FEATSTD_LOG_WARN("Error in screenshot: read pixels failed!");
                        result = false;
                    }   
                    if (!Renderer::UnbindPixelBuffer(*(data->m_pixelBuffer))) {
                        FEATSTD_LOG_WARN("Error in screenshot: unbind pixelbuffer failed!");
                        result = false;
                    }
                }
                else {
                    FEATSTD_LOG_WARN("Error in screenshot: bind to pixelbuffer failed!");
                    result = false;
                }
                if (result) {
                    if (m_isAsync) {
                        data->m_syncFence = SynchronizationFence::Create();
                    }
                }
                else {
                    CANDERA_DELETE(m_screenshotData[screenshotIdx]);
                    m_screenshotData[screenshotIdx] = 0;
                }
            }
            else {
                FEATSTD_LOG_WARN("Error in screenshot: no data given.");
                result = false;
            }
            return result;
#else
            FEATSTD_LOG_WARN("ScreenshotTool is not supported: Candera 3D is disabled!");
            FEATSTD_UNUSED2(id, screenshotIdx);
            return false;
#endif
        }

        bool ScreenshotTool::TakeScreenshotResult(UInt8 screenshotIdx)
        {
            FEATSTD_DEBUG_ASSERT(m_isInitialized);
            if (!m_isInitialized) {
                m_isInitialized = true;
                Initialize();
            }
#ifdef CANDERA_3D_ENABLED
            bool result = true;
            ScreenshotPrivateData * data = m_screenshotData[screenshotIdx];
            if (data != 0) {
                if (data->m_pixelBuffer != 0) {
                    FEATSTD_DEBUG_ASSERT(m_isAsync == (!data->m_syncFence.PointsToNull()));
                    if (!data->m_syncFence.PointsToNull()) {
                        if (!data->m_syncFence->IsProcessed()) {
                            FEATSTD_LOG_WARN("Performance issue in screenshot: Try to retrieve screenshot whilst generating the buffer is not done yet!");
                        }
                        data->m_syncFence.Release();
                        // There should be only a single reference to the sync fence - if not, it is a bug
                        FEATSTD_DEBUG_ASSERT(data->m_syncFence.PointsToNull());
                    }
                    if (Renderer::BindPixelBuffer(*(data->m_pixelBuffer))) {

                        data->m_screenshotData.m_data = FeatStd::Internal::PointerToPointer<UInt8*>(Renderer::MapPixelBuffer(*(data->m_pixelBuffer)));
                        if (data->m_screenshotData.m_data != 0) {
                            ScreenshotToolEvent evt(&data->m_screenshotData);
                            result = HandleScreenshotCallback(data->m_screenshotData);
                            m_eventSource.DispatchEvent(evt);
                            if (!Renderer::UnmapPixelBuffer(*(data->m_pixelBuffer))) {
                                FEATSTD_LOG_WARN("Error in screenshot: unmap pixelbuffer failed!");
                                result = false;
                            }
                        }
                        else {
                            FEATSTD_LOG_WARN("Error in screenshot: map pixelbuffer failed!");
                            result = false;
                        }
                        if (!Renderer::UnbindPixelBuffer(*(data->m_pixelBuffer))) {
                            FEATSTD_LOG_WARN("Error in screenshot: unbind pixelbuffer failed!");
                        }
                    }
                    else {
                        FEATSTD_LOG_WARN("Error in screenshot: bind to pixelbuffer failed for retrieving data!");
                    }
                }
                CANDERA_DELETE(m_screenshotData[screenshotIdx]);
                m_screenshotData[screenshotIdx] = 0;
            }
            return result;
#else
            FEATSTD_UNUSED(screenshotIdx);
            return false;
#endif
        }

        bool ScreenshotTool::AttachDispatcher()
        {
            if (!m_isAsync) {
                return true;
            }
            bool result = true;
            if (m_updateHandle.IsNullHandle()) {
                UpdateSystem* updateSystem = EntityComponentSystem::EntitySystem::Get<UpdateSystem>();
                FEATSTD_DEBUG_ASSERT(updateSystem != 0);
                if (updateSystem == 0) {
                    return false;
                }
                m_updateHandle = updateSystem->CreateComponent();
                CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1025, Candera::UpdateDelegate::Update, "False positive because the template arguments match.")
                result = updateSystem->SetComponentUpdateDelegate(m_updateHandle, UpdateSystem::Delegate::Update<ScreenshotTool, &ScreenshotTool::HandleFinishedScreenshots>());
                result = (result) && (updateSystem->AttachComponent(m_updateHandle, this));
                if (!result) {
                    static_cast<void>(updateSystem->DestroyComponent(m_updateHandle));
                    m_updateHandle = UpdateSystem::Handle();
                }
            }
            return result;
        }

        bool ScreenshotTool::DetachDispatcher()
        {
            if (!m_isAsync) {
                return true;
            }
            if (m_updateHandle.IsNullHandle()) {
                return true;
            }
            UpdateSystem* updateSystem = EntityComponentSystem::EntitySystem::Get<UpdateSystem>();
            if (updateSystem == 0) {
                return true;
            }
            static_cast<void>(updateSystem->DetachComponent(m_updateHandle));
            static_cast<void>(updateSystem->DestroyComponent(m_updateHandle));
            m_updateHandle = UpdateSystem::Handle();
            return true;
        }

        Candera::Diagnostics::ScreenshotTool& ScreenshotTool::GetInstance()
        {
            FEATSTD_SYNCED_STATIC_OBJECT(ScreenshotTool, s_instance);
            return s_instance;
        }

        bool ScreenshotTool::AddEventListener(EventListener* eventListener)
        {
            return m_eventSource.AddEventListener(eventListener);
        }

        bool ScreenshotTool::RemoveEventListener(EventListener* eventListener, bool waitForListenerRelease /*= true*/)
        {
            return m_eventSource.RemoveEventListener(eventListener, waitForListenerRelease);
        }

        FEATSTD_RTTI_DEFINITION(ScreenshotToolEvent, FeatStd::Event)

    }
}
