//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ControllerSystem.h"

#include <Candera/System/EntityComponentSystem/EntitySystem.h>
#include <Candera/System/UpdateSystem/UpdateSystem.h>

namespace Candera {

    FEATSTD_RTTI_DEFINITION(ControllerSystem, Base)

ControllerSystem::ControllerSystem()
{
    using namespace EntityComponentSystem;

    UpdateSystem* updateSystem = EntitySystem::Get<UpdateSystem>();
    if (0 != updateSystem) {
        UpdateSystem::Handle updateComponent = updateSystem->CreateComponent();

        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1025, Candera::UpdateDelegate::Update, "False positive because the template arguments match.")
        UpdateSystem::Delegate fixedUpdateDelegate = UpdateSystem::Delegate::Update<ControllerSystem, &ControllerSystem::FixedUpdateDelegate>();
        bool success = updateSystem->SetComponentFixedUpdateDelegate(updateComponent, fixedUpdateDelegate);

        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1025, Candera::UpdateDelegate::UpdateWithSeconds, "False positive because the template arguments match.")
        UpdateSystem::Delegate updateDelegate = UpdateSystem::Delegate::UpdateWithSeconds<ControllerSystem, &ControllerSystem::UpdateDelegate>();
        success = updateSystem->SetComponentUpdateDelegate(updateComponent, updateDelegate) && success;

        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1025, Candera::UpdateDelegate::Update, "False positive because the template arguments match.")
        UpdateSystem::Delegate lateUpdateDelegate = UpdateSystem::Delegate::Update<ControllerSystem, &ControllerSystem::LateUpdateDelegate>();
        success = updateSystem->SetComponentLateUpdateDelegate(updateComponent, lateUpdateDelegate) && success;

        success = updateSystem->AttachComponent(updateComponent, this) && success;
        FEATSTD_DEBUG_ASSERT(success);
        FEATSTD_UNUSED(success);
    }
}

void ControllerSystem::FixedUpdateDelegate()
{
    for (SizeType i = 0; i < m_components.Size(); ++i) {
        ControllerComponent* component = m_components[i];
        ControllerEntity* entity = component->GetEntity();
        if (0 != entity) {
            component->FixedUpdate(entity);
        }
    }
}

void ControllerSystem::UpdateDelegate(Double applicationTime, Double deltaTime)
{
    for (SizeType i = 0; i < m_components.Size(); ++i) {
        ControllerComponent* component = m_components[i];
        ControllerEntity* entity = component->GetEntity();
        if (0 != entity) {
            component->Update(entity, applicationTime, deltaTime);
        }
    }
}

void ControllerSystem::LateUpdateDelegate()
{
    for (SizeType i = 0; i < m_components.Size(); ++i) {
        ControllerComponent* component = m_components[i];
        ControllerEntity* entity = component->GetEntity();
        if (0 != entity) {
            component->LateUpdate(entity);
        }
    }
}

bool ControllerSystem::SetComponentPriority(Handle handle, Float priority)
{
    ControllerComponent* component = GetPointer(handle);
    if (0 == component) {
        return false;
    }

    FEATSTD_LINT_NEXT_EXPRESSION(777, CANDERA_LINT_REASON_FLOATCOMPARING)
        if (priority != component->m_priority) {
            component->m_priority = priority;
            SortComponents();
        }

    return true;
}

struct ComponentPriorityComparator
{
    bool operator()(const ControllerComponent* a, const ControllerComponent* b) const {
        return (a->GetPriority() > b->GetPriority());
    }
};

void ControllerSystem::SortComponents()
{
    ComponentPriorityComparator comparator;
    m_components.Sort(comparator);
}



} // namespace Candera
