//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_CONTROLLERCOMPONENT_H
#define CANDERA_CONTROLLERCOMPONENT_H

#include <Candera/EngineBase/Common/CanderaObject.h>
#include <Candera/System/EntityComponentSystem/Component.h>
#include <Candera/System/EntityComponentSystem/ComponentSystem.h>
#include <CanderaPlatform/OS/CanderaTypes.h>

namespace Candera {

typedef CanderaObject ControllerEntity;

/** @addtogroup ControllerSystem
*  @{
*/

/**
* @brief  ControllerComponent is the abstract base class for user derived classes that can be attached
*         to any CanderaObject. A derived class can override FixedUpdate/Update/LateUpdate functions
*         which are called via the UpdateSystem.
*         The purpose of a ControllerComponent derived class is to easily attach code at runtime to an
*         entity without bloating every instance of the entity class. Think about it like a native 'script'.
*/

class ControllerComponent : public EntityComponentSystem::Component<ControllerEntity>
{

public:
    FEATSTD_RTTI_DECLARATION();

    /**
     *  Pure destructor to make this class abstract.
     */
    virtual ~ControllerComponent() = 0;

    /**
     *  FixedUpdate is called 1/FixedTimeStep times per second independent of the actual render frame rate.
     *  @param entity  The entity this component is attached to.
     */
    virtual void FixedUpdate(ControllerEntity* entity) { FEATSTD_UNUSED(entity); };

    /**
     *  Update is called every render frame.
     *  @param entity           The entity this component is attached to.
     *  @param applicationTime  The application time measured since the start of the application in seconds.
     *  @param deltaTime        The time measured since the last render frame in seconds.
     */
    virtual void Update(ControllerEntity* entity, Double applicationTime, Double deltaTime) { FEATSTD_UNUSED(entity); FEATSTD_UNUSED(applicationTime); FEATSTD_UNUSED(deltaTime); }

    /**
     *  LateUpdate is called every render frame after all Update calls have been executed.
     *  @param entity  The entity this component is attached to.
     */
    virtual void LateUpdate(ControllerEntity* entity) { FEATSTD_UNUSED(entity); };

    /**
     *  Get the priority of the component system.
     *  @return  The priority of the component system.
     */
    Float GetPriority() const { return m_priority; }

protected:
    /**
     *  Constructor
     */
    ControllerComponent() : m_priority(0.0F) {}

private:
    friend class EntityComponentSystem::ComponentSystem<ControllerComponent, ControllerEntity>;
    friend class ControllerSystem;

    Float m_priority;
};

/** @} */ // end of UpdateSystem

} // namespace Candera

#endif
