//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_LAYOUTER_H)
    #define CANDERA_LAYOUTER_H

#include <Candera/Environment.h>
#include <Candera/System/Mathematics/Vector2.h>
#include <Candera/System/Mathematics/Rectangle.h>
#include <Candera/EngineBase/Layout/Margin.h>

#ifdef CANDERA_2D_ENABLED
#include <Candera/Engine2D/Core/Node2D.h>
#endif
#include <Candera/EngineBase/Layout/Margin.h>
#include <Candera/EngineBase/Layout/LayoutDataTypes.h>
#include <Candera/System/Mathematics/MathDataTypes.h>
#include <Candera/EngineBase/Layout/LayoutTypes.h>
#include <Candera/EngineBase/Common/Alignment.h>
#include <Candera/EngineBase/Common/CanderaObject.h>
#include <FeatStd/Event/EventSource.h>
#include <FeatStd/Event/Event.h>
#include <Candera/EngineBase/Common/AbstractNodePointer.h>

#ifdef CANDERA_IGNORE_LAYOUTER_DEPRECATED_3_4_2_ENABLED
#define CANDERA_LAYOUTER_DEPRECATED_3_4_2(msg, func) func
#else
#define CANDERA_LAYOUTER_DEPRECATED_3_4_2(msg, func) CANDERA_DEPRECATED_3_4_2(msg, func)
#endif

namespace Candera {
    namespace Internal {
        template <typename T> class DelegatingLayouter; // forward declaration for lookup of friend declaration
    }
/** @addtogroup Layout
 *  @{
 */

    class Node2D;
#ifdef CANDERA_3D_ENABLED
    class Node;
#endif

    /**
     *  @brief A non-zero padding applies space inside the element layout's width and height.
     *  The values can be set separately for left, top, right and bottom so the padding can be asymmetric.
     */
    //typedef will be deprecated later on, interfaces for padding already marked with deprecation macros.
    typedef Margin Padding;

    /**
     *  @brief To layout a group of elements an instance of a Layouter has to be attached to a node. Child nodes act
     *  automatically as element for the parent layouter. The order of the child nodes defines the order of
     *  the layout items.
     */
    class Layouter : public CanderaObject {
#if defined(CANDERA_LAYOUT_ENABLED)

        public:
            /**
            * Set the behavior of all layout directions that are configured as AutomaticDirection.
            * NOTE: The default is InheritDirection.
            * @param layoutDirection The layout direction that has to be used for AutomaticDirection.
            */
            static void SetAutomaticDirectionBehavior(LayoutAlignment::LayoutDirection::Enum layoutDirection);

            /**
            * Returns the behavior of all layout directions that are configured as AutomaticDirection.
            * NOTE: The default is InheritDirection.
            * @return The layout direction that is used for AutomaticDirection.
            */
            static LayoutAlignment::LayoutDirection::Enum GetAutomaticDirectionBehavior();

            /**
             * Disposes the instance of this class.
             */
            virtual void Dispose() { FEATSTD_DELETE(this); }

            /**
             * Return the event source of all layout related events
             * @return returns the event source of all layout related events
             */
            static FeatStd::EventSource& GetEventSource();

            /**
             *  Triggers to layout the child nodes. Calls Measure() and Arrange() internally.
             *  @param node Node which child nodes shall be layouted.
             */
            void Layout(const AbstractNodePointer& node);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                void Layout(Node2D& node));
#endif

            /**
             *  Has to be called in OnMeasure() for all child nodes. This invokes the call of OnMeasure()
             *  which has to be implemented for each layout strategy.
             *  @param  node (Child) node which shall be measured.
             *  @param  clientSize      The width and height which is available for the particular child node.
             */
            void Measure(const AbstractNodePointer& node, const Vector2& clientSize);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                void Measure(Node2D& node, const Vector2& clientSize));
#endif

            /**
             *  Has to be called in OnArrange() for all child nodes. This invokes the call of OnArrange()
             *  which has to be implemented for each layout strategy.
             *  @param  node            (Child) node which shall be arranged.
             *  @param  clientArea      The rectangle which can be filled by the particular child node.
             *  @param  topLevel        Only the top level layouter should use true here. This is usually only Layouter::Layout().
             */
            void Arrange(const AbstractNodePointer& node, const Rectangle& clientArea, bool topLevel = false);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                void Arrange(Node2D& node, const Rectangle& clientArea, bool topLevel = false));
#endif

            /**
             *  Gets the parent space axis aligned bounding rectangle of a given node
             *  @param node The node that is calculated
             *  @param rect The rectangle that will be the parent space axis aligned bounding rectangle
             */
            static void GetParentSpaceAxisAlignedBoundingRect(const AbstractNodePointer& node, Rectangle& rect);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                static void GetParentSpaceAxisAlignedBoundingRect(const Node2D& node, Rectangle& rect));
#endif

            /**
             * Sets the vertical alignment of a given node within its parent container.
             *  @param node The node to set the vertical alignment for.
             *  @param alignment The vertical alignment of this node within a parent container.
             */
            static void SetVerticalAlignment(CanderaObject& node, VerticalAlignment alignment);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the CanderaObject version.",
                static void SetVerticalAlignment(Node2D& node, VerticalAlignment alignment));
#endif

            /**
             *  Retrieves the vertical alignment of a given node within its parent container.
             *  @param node The node to retrieve the vertical alignment for.
             *  @return The vertical alignment of node.
             */
            static VerticalAlignment GetVerticalAlignment(const CanderaObject& node);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the CanderaObject version.",
                static VerticalAlignment GetVerticalAlignment(const Node2D& node));
#endif

            /**
             *  Sets the horizontal alignment of a given node within its parent container.
             *  @param node The node to set the horizontal alignment for.
             *  @param horizontalAlignment The horizontal alignment of the given node within its parent container.
             */
            static void SetHorizontalAlignment(CanderaObject& node, HorizontalAlignment horizontalAlignment);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the CanderaObject version.",
                static void SetHorizontalAlignment(Node2D& node, HorizontalAlignment horizontalAlignment));
#endif

            /**
             *  Retrieves the horizontal alignment of a given node within a parent container.
             *  @param node The node whose horizontal alignment is sought.
             *  @return The horizontal alignment of node within its parent container.
             */
            static HorizontalAlignment GetHorizontalAlignment(const CanderaObject& node);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the CanderaObject version.",
                static HorizontalAlignment GetHorizontalAlignment(const Node2D& node));
#endif
            /**
             *  Retrieves the fixed size defined for a given node.
             *  @param node The node whose fixed size is sought.
             *  @return The fixed size of node. Note: -1.0F/-1.0F means that the fixed size is not set!
             *  @see IsSizeSet()
             */
            static const Vector2& GetSize(const CanderaObject& node);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the CanderaObject version.",
                static const Vector2& GetSize(const Node2D& node));
#endif

            /**
             *  Sets the fixed size for a given node.
             *  @param   node    The node whose fixed size shall be set.
             *  @param   size    The fixed size for this node.
             */
            static void SetSize(CanderaObject& node, const Vector2& size);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the CanderaObject version.",
                static void SetSize(Node2D& node, const Vector2& size));
#endif

            /**
             *  Sets the fixed size for this node.
             *  @param   node     The node whose fixed size shall be set.
             *  @param   width    The fixed width for this node.
             *  @param   height   The fixed height for this node.
             */
            static void SetSize(CanderaObject& node, Float width, Float height)
            {
                SetSize(node, Vector2(width, height));
            }
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the CanderaObject version.",
                static void SetSize(Node2D& node, Float width, Float height));
#endif

            /**
             *  Gets if the size of node is set.
             *  @param  node The Node whose size status is sought.
             *  @return     true, if the size was set, false if the size is reset to its default value.
             */
            static bool IsSizeSet(const CanderaObject &node);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the CanderaObject version.",
                static bool IsSizeSet(const Node2D &node));
#endif

            /**
             *  Retrieves the minimum size occupied by a given node node.
             *  @param  node The node, whose minimum size is sought.
             *  @return The minimum size occupied by a give node node.
             */
            static const Vector2& GetMinimumSize(const CanderaObject& node);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the CanderaObject version.",
                static const Vector2& GetMinimumSize(const Node2D& node));
#endif

            /**
             *  Sets the minimum size occupied by a given node node.
             *  @param node The node whose minimum size shall be set.
             *  @param minSize The minimal extent to occupy.
             */
            static void SetMinimumSize(CanderaObject& node, const Vector2& minSize);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the CanderaObject version.",
                static void SetMinimumSize(Node2D& node, const Vector2& minSize));
#endif

            /**
             *  Sets the minimum size occupied by this node.
             *  @param node   The node, whose minimum size shall be set.
             *  @param width  The minimal width to occupy.
             *  @param height The minimal height to occupy.
             */
            static void SetMinimumSize(CanderaObject& node, Float width, Float height)
            {
                SetMinimumSize(node, Vector2(width, height));
            }
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the CanderaObject version.",
                static void SetMinimumSize(Node2D& node, Float width, Float height));
#endif
            /**
             *  Retrieves the maximum size occupied by this node.
             *  @param node The node whose maximum size is sought.
             *  @return The maximum size occupied by this node.
             */
            static const Vector2& GetMaximumSize(const CanderaObject& node);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the CanderaObject version.",
                static const Vector2& GetMaximumSize(const Node2D& node));
#endif

            /**
             *  Retrieves whether the maximum size is set or not
             *  @param node The node whose maximum size is checked
             *  @return true, if the maximum size is set, false if not
             */
            static bool IsMaximumSizeSet(const CanderaObject& node);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the CanderaObject version.",
                static bool IsMaximumSizeSet(const Node2D& node));
#endif

            /**
             *  Sets the maximum size to occupy by this node.
             *  @param node   The node, whose maximum size shall be set.
             *  @param maxSize The maximal extent to occupy.
             */
            static void SetMaximumSize(CanderaObject& node, const Vector2& maxSize);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the CanderaObject version.",
                static void SetMaximumSize(Node2D& node, const Vector2& maxSize));
#endif

            /**
             *  Sets the maximum size occupied by this node.
             *  @param node   The node, whose minimum size shall be set.
             *  @param width The maximal width to occupy.
             *  @param height The maximal height to occupy.
             */
            static void SetMaximumSize(CanderaObject& node, Float width, Float height)
            {
                SetMaximumSize(node, Vector2(width, height));
            }
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the CanderaObject version.",
                static void SetMaximumSize(Node2D& node, Float width, Float height));
#endif

            /**
             *  Retrieves the margin used for layout this node.
             *  @param node The node whose margin is sought.
             *  @return The margin used for layout this node.
             */
            static const Margin& GetMargin(const CanderaObject& node);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the CanderaObject version.",
                static const Margin& GetMargin(const Node2D& node));
#endif

            /**
             *  Sets the margin used for layout this node.
             *  @param node   The node, whose margin shall be set.
             *  @param margin The margin used for layout this node.
             */
            static void SetMargin(CanderaObject& node, const Margin& margin);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the CanderaObject version.",
                static void SetMargin(Node2D& node, const Margin& margin));


            /**
            *  Retrieves whether the margin is set or not
            *  @param node The node whose margin size is checked
            *  @return true, if the margin is set, false if not
            */
            static bool IsMarginSet(const CanderaObject& node);

#endif
            /**
             * Sets the stretch behavior for layout this node.
             * @param node            The node, whose stretch behavior shall be set.
             * @param stretchBehavior StretchBehavior to be applied to the node, defines
             *                        if node gets stretched or not, and when stretched if it should keep it's aspect ratio or shall fill the
             *                        the stretched area.
             */
            static void SetStretchBehavior(CanderaObject& node, LayoutAlignment::StretchBehavior::Enum stretchBehavior);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the CanderaObject version.",
                static void SetStretchBehavior(Node2D& node, LayoutAlignment::StretchBehavior::Enum stretchBehavior));
#endif

            /**
             *  Retrieves the stretch behavior used for layout this node.
             *  @param node The node whose stretch behavior is sought.
             *  @return     The stretch behavior used for layout this node.
             */
            static LayoutAlignment::StretchBehavior::Enum GetStretchBehavior(const CanderaObject& node);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the CanderaObject version.",
                static LayoutAlignment::StretchBehavior::Enum GetStretchBehavior(const Node2D& node));
#endif

            /**
             *  Sets the direction (left-to-right or right-to-left) for the layout.
             *  It can be set either fixed left-to-right or right-to-left or dependent on the currently set language.
             *  @param  node        The node whose layout direction shall be set.
             *  @param  direction   The layout orientation.
             */
            static void SetLayoutDirection(CanderaObject& node, LayoutAlignment::LayoutDirection::Enum direction);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the CanderaObject version.",
                static void SetLayoutDirection(Node2D& node, LayoutAlignment::LayoutDirection::Enum direction));
#endif

            /**
             *  Retrieves the set layout direction.
             *  @param  node    The node whose layout direction is sought.
             *  @return         The layout direction.
             */
            static LayoutAlignment::LayoutDirection::Enum GetLayoutDirection(const CanderaObject& node);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the CanderaObject version.",
                static LayoutAlignment::LayoutDirection::Enum GetLayoutDirection(const Node2D& node));
#endif

            /**
            *  Indicates if the layout direction is set on this node.
            *  @param  node    The node whose layout direction is sought.
            *  @return         true if the layout direction is set on this node.
            */
            static bool IsLayoutDirectionSet(const CanderaObject& node);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the CanderaObject version.",
                static bool IsLayoutDirectionSet(const Node2D& node));
#endif

            /**
             *  \brief  Clears the set layout direction.
             *  The property falls back to the set value of the next parent node in the hierarchy or to default value.
             *  @param  node        The node whose layout direction shall be cleared.
             */
            static void ClearLayoutDirection(CanderaObject& node);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the CanderaObject version.",
                static void ClearLayoutDirection(Node2D& node));
#endif

            /**
            *  Sets the collapsible flag.
            *  It specifies if the node should be collapsed if the effective rendering is disabled (if the layout is collapsed then no layout will be performed on any child node).
            *  @param  node        The node whose layout direction shall be set.
            *  @param  collapsible The new collapsible flag value.
            */
            static void SetCollapsible(CanderaObject& node, bool collapsible);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the CanderaObject version.",
                static void SetCollapsible(Node2D& node, bool collapsible));
#endif

            /**
            *  Retrieves the collapsible flag.
            *  @param  node    The node whose layout direction is sought.
            *  @return         The layout direction.
            */
            static bool IsCollapsible(const CanderaObject& node);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the CanderaObject version.",
                static bool IsCollapsible(const Node2D& node));


#endif
            /**
             *  Gets the preferred size of the given node.
             *  @param  node    The node to retrieve preferred size from.
             *  @return         The preferred size.
             */
            Vector2 GetPreferredSize(const AbstractNodePointer& node) const { return node.GetPreferredSize(); }
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                Vector2 GetPreferredSize(const Node2D& node) const);
#endif

            /**
             *  Gets the preferred size + margin of the given node.
             *  @param node The node to retrieve preferred size + margin from.
             *  @return     The preferred size + margin.
             */
            static Vector2 GetClientSize(const AbstractNodePointer& node);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                Vector2 GetClientSize(const Node2D& node) const);
#endif

            /**
            *  Sets the cached flag. A cached layout will be only layed out again as soon as the layout of an inner node is marked as invalid.
            *  @param  node        The node whose layout shall be cached.
            *  @param  cached      The new cached flag value.
            */
            static void SetCached(CanderaObject& node, bool cached);

            /**
            *  Retrieves the cached flag.
            *  @param  node    The node.
            *  @return         True if the layout is cached. False otherwise.
            */
            static bool IsCached(const CanderaObject& node);

            /**
             *  Provides the parent for dynamic property scene graph inheritance.
             *  @param  host    The host, whose parent is sought.
             *  @return         The parent of host if available, null otherwise.
             */
            static const Candera::DynamicProperties::DynamicPropertyHost* ParentProvider(const Candera::DynamicProperties::DynamicPropertyHost* host);

            /**
             *  Clones this Layouter by creating a new instance with the same parameters than this instance.
             *  @return     Pointer to the cloned layouter if successful, null otherwise.
             */
            virtual Layouter* Clone() const = 0;

            /**
             *  Marks the layout of the given node as invalid.
             *  @param node
             */
            static void InvalidateLayout(const AbstractNodePointer& node);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                static void InvalidateLayout(const Node2D* node));
#endif
            static void InvalidateLayout(DynamicPropertyHost* host);

            /**
             *  Returns the clipping area that has been set for this node by the layouter.
             *  @param node
             *  @return The clipping area.
             */
            static const Rectangle& GetClippingArea(const CanderaObject& node);

        protected:

            /**
            *  Sets the clipping area. This only has to be called when the clipping area is set by a derived layouter in the overwritten OnClipping method and if the base OnClipping method is not called by that overwritten OnClipping method.
            *  @param node The node.
            *  @param clippingArea The clipping area.
            *  @return returns a bool value
            */
            static bool SetClippingArea(CanderaObject& node, const Rectangle& clippingArea);

            // Call Create() to instanciate a layouter object.
            Layouter() {}
            // Call Dispose() to delete a layouter instance.
            virtual ~Layouter() override {}

            /**
             *  This method is called as first layout pass to collect all the data which is needed to layout the child nodes.
             *  @param node The node which child nodes shall be layouted.
             *  @param clientArea Area where layout shall be applied.
             *  @return Vector describing height and width of whole layouted rectangle.
             */
            virtual Vector2 OnMeasure(const AbstractNodePointer& node, const Vector2& clientArea) = 0;
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                Vector2 OnMeasure(Node2D& node, const Vector2& clientArea));
#endif

            /**
             *  This method is called as second layout pass to arrange the child nodes (position and scale).
             *  @param node Node which children shall be arranged.
             *  @param clientArea Rectangle where layout shall be applied.
             */
            virtual void OnArrange(const AbstractNodePointer& node, const Rectangle& clientArea) = 0;
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                void OnArrange(Node2D& node, const Rectangle& clientArea));
#endif
#if defined(CANDERA_LAYOUT_CLIPPING_ENABLED)
            /**
             * OnClipping is called at the end of Arrange. The default implementation sets the clipping rectangle on RenderNodes.
             */
            virtual void OnClipping(const AbstractNodePointer& node, const Rectangle& clientArea);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                void OnClipping(Node2D& node, const Rectangle& clientArea));
#endif
#endif
            /**
             * OnSetPosition is called by the Arrange to set the node position.
             * @param node
             * @param position
             */
            virtual void OnSetPosition(const AbstractNodePointer& node, const Vector2& position);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                void OnSetPosition(Node2D& node, const Vector2& position));
#endif

            /**
            *  This method is called at the end of Measure. It limits the preferredSize to the min and max size values.
            *  @param node the node for which to limit the preferred size.
            *  @param preferredSize the calculated preferred size, will be limited by this function.
            *  @param nodeSize the calculated size of the node.
            *  @param nodeMinSize the minimum size of the node.
            *  @param nodeMaxSize the maximum size of the node.
            */
            virtual void OnLimitPreferredSize(CanderaObject& node, Vector2& preferredSize, const Vector2& nodeSize, const Vector2& nodeMinSize, const Vector2& nodeMaxSize);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the candera object node version. Please use the �CanderaObject version.",
                void OnLimitPreferredSize(Node2D& node, Vector2& preferredSize, const Vector2& nodeSize, const Vector2& nodeMinSize, const Vector2& nodeMaxSize));
#endif
            /**
             * Returns true if the layouter itself handles the scale. Otherwise the scale will be considered by the general layouter code.
             * @return true if the layouter itself handles the scale, false otherwise.
             */
            virtual bool IsHandlingScale()
            {
                return false;
            }

            static void AlignRectangle(const Rectangle& outerRect, Rectangle& innerRect, const AbstractNodePointer& node);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                static void AlignRectangle(const Rectangle& outerRect, Rectangle& innerRect, Node2D& node));
#endif

            static void SetNodePosition(const AbstractNodePointer& node, const Vector2& position);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                static void SetNodePosition(Node2D& node, const Vector2& position));
#endif
            static void SetArrangeActualSize(const Vector2& size);
            static const Vector2& GetArrangeActualSize();
            static void SetNodeScale(const AbstractNodePointer& node, const Vector2& scale);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                static void SetNodeScale(Node2D& node, const Vector2& scale));
#endif

            static Vector2 Vector2Min(const Vector2& size1, const Vector2& size2);
            static Vector2 Vector2Max(const Vector2& size1, const Vector2& size2);

            /**
             *  Retrieves the layout orientation dependent on the currently set language.
             *  If parameter LayoutOrientation is set to LayoutOrientation_Automatic, this method returns left-to-right or right-to-left
             *  dependent on the system language.
             *  @param node Node which LanguageSensitiveDirection shall be get.
             *  @return     LayoutOrientation_LeftToRight or LayoutOrientation_RightToLeft.
             */
            static LayoutAlignment::LayoutDirection::Enum GetLanguageSensitiveDirection(const AbstractNodePointer& node);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                static LayoutAlignment::LayoutDirection::Enum GetLanguageSensitiveDirection(const Node2D& node));
#endif

            /**
            *  Retrieves the parent layout orientation dependent on the currently set language.
            *  If parameter LayoutOrientation is set to LayoutOrientation_Automatic, this method returns left-to-right or right-to-left
            *  dependent on the system language.
            *  @param node Node which LanguageSensitiveDirection shall be get.
            *  @return     LayoutOrientation_LeftToRight or LayoutOrientation_RightToLeft.
            */
            static LayoutAlignment::LayoutDirection::Enum GetParentLanguageSensitiveDirection(const AbstractNodePointer& node);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                static LayoutAlignment::LayoutDirection::Enum GetParentLanguageSensitiveDirection(const Node2D& node));
#endif

            static Int16 GetLanguageSensitiveLeftMargin(const AbstractNodePointer& node, const Margin& margin);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                static Int16 GetLanguageSensitiveLeftMargin(const Node2D& node, const Margin& margin));
#endif

            static Vector2 GetAxisAlignedDimension(const Vector2& size, Float rotation);

            /**
             * Returns the preferred size before it is rotated and axis aligned.
             * It is only set when a node is rotated.
             * This value is only required if operations in the client area are required.
             * E.g. the calculation of the rotated base line.
             * @param node
             * @return returns the preferred size before it is rotate and axis aligned
             */
            static const Vector2& GetOriginalPreferredSize(const CanderaObject& node);

        private:
            friend void MemoryManagement::Internal::Destructor<Layouter, false>::Destruct(Layouter* object);
            friend class LayouterDynamicProperties;
            friend class AbstractNodePointer;
            friend class BitmapBrush;
#ifdef CANDERA_3D_ENABLED
            friend class CanvasSprite;
#endif
            friend class DefaultLayouter;
            template <typename T> friend class Internal::DelegatingLayouter;

            static void SetLayoutValid(Candera::CanderaObject& node, bool layoutValid);

            static void SetActualSize(const AbstractNodePointer& node, const Vector2& size);
            static const Vector2& GetActualSize(const AbstractNodePointer& node);

            static bool IsActualSizeAvailable(AbstractNodePointer const& node);

                // avoid assignment
            Layouter& operator=(const Layouter&);

            static const Vector2& MinimumSizeDefault();
            static const Vector2& MaximumSizeDefault();
            static const Vector2& SizeDefault();
            static const Margin& MarginDefault();
            static const Padding& PaddingDefault();

            static void OnVerticalAlignmentChanged(DynamicPropertyHost* obj, const DynamicProperties::ValueChangedArgs<VerticalAlignment>& /*args*/) { InvalidateLayout(obj); }
            static void OnHorizontalAlignmentChanged(DynamicPropertyHost* obj, const DynamicProperties::ValueChangedArgs<HorizontalAlignment>& /*args*/) { InvalidateLayout(obj); }
            static void OnSizeChanged(DynamicPropertyHost* obj, const DynamicProperties::ValueChangedArgs<Vector2>& /*args*/) { InvalidateLayout(obj); }
            static void OnMarginChanged(DynamicPropertyHost* obj, const DynamicProperties::ValueChangedArgs<Margin>& /*args*/) { InvalidateLayout(obj); }
            static void OnPaddingChanged(DynamicPropertyHost* obj, const DynamicProperties::ValueChangedArgs<Padding>& /*args*/) { InvalidateLayout(obj); }
            static void OnStretchBehaviorChanged(DynamicPropertyHost* host, const DynamicProperties::ValueChangedArgs<LayoutAlignment::StretchBehavior::Enum>& args);
            static void OnLayoutDirectionChanged(DynamicPropertyHost* obj, const DynamicProperties::ValueChangedArgs<LayoutAlignment::LayoutDirection::Enum>& /*args*/) { InvalidateLayout(obj); }
            static void OnCollapsibleChanged(DynamicPropertyHost* obj, const DynamicProperties::ValueChangedArgs<bool>& /*args*/) { InvalidateLayout(obj); }
            static void OnCachedChanged(DynamicPropertyHost* obj, const DynamicProperties::ValueChangedArgs<bool>& /*args*/) { InvalidateLayout(obj); }

            static bool IsNodeTransformationAllowed(const AbstractNodePointer& node);

#ifdef CANDERA_2D_ENABLED
            friend class Node2D;
#endif
#ifdef CANDERA_3D_ENABLED
            friend class Node;
#endif
            CdaDynamicPropertiesDeclaration();
#endif  // CANDERA_LAYOUT_ENABLED
    };

namespace Internal { namespace LayoutEvents {
    class LayoutEvent : public FeatStd::Event
    {
    public:
        FEATSTD_RTTI_DECLARATION();

        enum State
        {
            LayoutValidationBegin,
            LayoutBegin,
            MeassureBegin,
            MeassureEnd,
            ArrangeBegin,
            ArrangeSetActualSize,
            ArrangeSetClippingRectangle,
            ArrangeEnd,
            LayoutEnd,
            LayoutValidationEnd,
            PreSetNodePosition,
            PreSetNodeScale
        };

        LayoutEvent(State state, const AbstractNodePointer& node) :
            m_state(state),
            m_node(node)
        {
        }

        virtual ~LayoutEvent()
        {
        }

        State GetState() const
        {
            return m_state;
        }

        const AbstractNodePointer& GetNode() const
        {
            return m_node;
        }

    private:
        FEATSTD_MAKE_CLASS_STATIC(LayoutEvent);
        FEATSTD_MAKE_CLASS_UNCOPYABLE(LayoutEvent);

        State m_state;
        const AbstractNodePointer m_node;
    };

    class SizeLayoutEvent : public LayoutEvent
    {
    public:
        FEATSTD_RTTI_DECLARATION();

        SizeLayoutEvent(State state, const AbstractNodePointer& node, const Vector2& size) :
            LayoutEvent(state, node),
            m_size(size)
        {
        }

        virtual ~SizeLayoutEvent()
        {
        }

        const Vector2& GetSize() const
        {
            return m_size;
        }

    private:
        FEATSTD_MAKE_CLASS_STATIC(SizeLayoutEvent);
        FEATSTD_MAKE_CLASS_UNCOPYABLE(SizeLayoutEvent);

        const Vector2& m_size;
    };

    class ArrangeEvent : public SizeLayoutEvent
    {
    public:
        FEATSTD_RTTI_DECLARATION();

        ArrangeEvent(State state, const AbstractNodePointer& node, const Vector2& position, const Vector2& size) :
            SizeLayoutEvent(state, node, size),
            m_position(position)
        {
        }

        virtual ~ArrangeEvent()
        {
        }

        const Vector2& GetPosition() const
        {
            return m_position;
        }

    private:
        FEATSTD_MAKE_CLASS_STATIC(ArrangeEvent);
        FEATSTD_MAKE_CLASS_UNCOPYABLE(ArrangeEvent);

        const Vector2& m_position;
    };

}}    // namespace LayoutEvents namepace Internal
 /** @} */ // end of Layout
}   // namespace Candera

#endif  // CANDERA_LAYOUTER_H
