//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_LAYOUTDATATYPES_H)
#define CANDERA_LAYOUTDATATYPES_H

#include <Candera/EngineBase/Layout/LayoutTypes.h>
#include <Candera/EngineBase/Layout/Margin.h>
#include <CanderaPlatform/OS/SerializationPlatform.h>
#include <Candera/System/MetaInfo/DataTypeStringBufferAppenderMacros.h>

namespace Candera {
    class Layouter;

    namespace MetaInfo {

        /** @addtogroup DataTypesBase
         *  @{
         */

        /**
         * @brief DataType specialization for Margin.
         *
         * The string representation of a Margin consists of the four margin coordinates separated by a comma,
         *  e.g. "0,0,100,100"; left = 0, top = 0, right = 100, bottom = 100.
         */
        template<> struct DataType<Margin> {
            /**
             * Convert a Margin to its string representation.
             *
             * @param margin Margin to convert.
             * @param buf    Buffer to write the string representation.
             * @param bufLen Buffer length.
             * @return true if conversion successful, false otherwise.
             */
            static bool ConvertToString(const Margin& margin, Char* buf, UInt bufLen) {
                Int tab[4] = { margin.GetLeft(), margin.GetTop(), margin.GetRight(), margin.GetBottom() };
                UInt n = SerializationPlatform::FromBinary(buf, bufLen, ",", tab, tab + 4);
                return n == 4;
            }

            /**
             * Convert the string representation to a Margin.
             *
             * @param margin converted Margin.
             * @param buf Buffer that contains the string representation.
             * @return true if conversion successful, false otherwise.
             */
            static bool ConvertFromString(Margin& margin, const Char* buf) {
                Int tab[4];
                UInt n = SerializationPlatform::ToBinary(buf, ",", tab, tab + 4);
                margin = Margin((Int16)tab[0], (Int16)tab[1], (Int16)tab[2], (Int16)tab[3]);
                return n == 4;
            }

            /**
             * Get the PropertyEditor name as known by SceneComposer.
             *
             * @return string containing the SceneComposer property editor name.
             */
            static const Char* GetEditor() {
                return "builtin://MarginEditor";
            }
        };

        template<>
        struct DataType<Layouter*> {
            static bool ConvertToString(Layouter* /*val*/, Char* /*buf*/, UInt /*size*/) {
                return false;
            }
            static bool ConvertFromString(Layouter*& /*val*/, const Char* /*buf*/) {
                return false;
            }
            static const Char* GetEditor() {
                return 0;
            }
        };

        /**
         * @} //end of DataTypes
         */

    } // namespace MetaInfo
}   // namespace Candera

CdaDeclareStringBufferAppender(Candera::Margin)
CdaDeclareStringBufferAppender(Candera::Layouter*)

#endif  // CANDERA_LAYOUTDATATYPES_H
