//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_GRIDLAYOUTER_H)
    #define CANDERA_GRIDLAYOUTER_H

#include <Candera/EngineBase/Layout/Layouter.h>
#include <Candera/EngineBase/DynamicProperties/TypeInfo.h>

#define ENUM_DATA_TYPE \
    ENUM_DATA_TYPE_BEGIN(GridAutoArrangement)   \
        ENUM_DATA_TYPE_ITEM(GridAutoArrangementDisabled)           \
        ENUM_DATA_TYPE_ITEM(GridAutoArrangementHorizontal)         \
        ENUM_DATA_TYPE_ITEM(GridAutoArrangementVertical)         \
    ENUM_DATA_TYPE_END(GridAutoArrangement)

#include <Candera/System/MetaInfo/EnumDataType.h>

FEATSTD_LINT_NEXT_EXPRESSION(1577, "MISRA C++ 2008 Required Rule 14-7-3: specialization is done in TypeInfo.h") \
CdaDefPrimitiveTypeInfo(GridAutoArrangement, GridAutoArrangementDisabled)

namespace Candera {

/** @addtogroup Layout
 *  @{
 */

    /**
     *  @brief GridLayouter arranges its elements in a grid. The amount of rows and columns for the grid must be specified.
     *
     *  Each row and column can be configured with one float value:
     *  - Automatic size (value < 0):
     *      The row height (or column width) will be defined by the highest value of its child elements. So
     *      the row or column is given exactly the amount of space it needs, and no more.
     *  - Proportional size (0.0 < value < 1.0):
     *      The space is divided between a group of rows or columns.
     *  - Absolute size (value >= 1.0):
     *      Defines an exact (fixed) size in pixels.
     */
    class GridLayouter : public Layouter {
        FEATSTD_TYPEDEF_BASE(Layouter);

        public:
            /**
             *  Creates an instance of this class.
             *  @return A pointer to the created object.
             */
            static GridLayouter* Create();

            /**
             *  Sets the layout.
             *  @param rows The rows of this layout.
             *  @param columns The columns of this layout.
             */
            void SetLayout(UInt8 rows, UInt8 columns);

            /**
             *  Retrieves the columns of this layout.
             *  @return The columns of this layout.
             */
            UInt8 GetColumnCount() const { return m_columns; }

            /**
             *  Retrieves the rows of this layout.
             *  @return The rows of this layout.
             */
            UInt8 GetRowCount() const { return m_rows; }

            /**
             *  Sets the height of the specified row using the rules below.
             *  @param  row     The row index.
             *  @param  height  height < 0:         auto height
             *                  0.0 < height < 1.0: relation value to other rows having a relation value
             *                  height >= 1.0:      fixed row height
             */
            void SetRowHeight(UInt8 row, Float height) { if (row < m_rows) { m_rowData[row].m_param = height; } }

            /**
             *  Gets the height of the specified row.
             *  @param  row    The row index.
             *  @return        The rows height
             *                 height < 0:         auto height
             *                 0.0 < height < 1.0: relation value to other rows having a relation value
             *                 height >= 1.0:      fixed row height
             */
            Float GetRowHeight(UInt8 row) const { return m_rowData[row].m_param; }

            /**
             *  Sets the width of the specified row using the rules below.
             *  @param  column  The column index.
             *  @param  width   width < 0:         auto width
             *                  0.0 < width < 1.0: relation value to other columns having a relation value
             *                  width >= 1.0:      fixed column width
             */
            void SetColumnWidth(UInt8 column, Float width) { if (column < m_columns) { m_columnData[column].m_param = width; } }

            /**
             *  Gets the width of the specified row.
             *  @param  column The column index.
             *  @return        The columns width
             *                 width < 0:         auto width
             *                 0.0 < width < 1.0: relation value to other columns having a relation value
             *                 width >= 1.0:      fixed column height
             */
            Float GetColumnWidth(UInt8 column) const { return m_columnData[column].m_param; }

            /**
             *  Retrieves the destination grid row of a grid element.
             *  @param object The element, whose destination grid row is sought.
             *  @return The destination grid row of a grid element.
             */
            static UInt8 GetRow(const DynamicPropertyHost &object);

            /**
             *  Sets the destination grid row of a grid element.
             *  @param object The element, whose destination grid row is to be set.
             *  @param row The new grid row of object.
             */
            static void SetRow(DynamicPropertyHost &object, UInt8 row);

            /**
             *  Retrieves the destination grid column of a grid element.
             *  @param object The element, whose destination grid column is sought.
             *  @return The destination grid column of a grid element.
             */
            static UInt8 GetColumn(const DynamicPropertyHost &object);

            /**
             *  Sets the destination grid column of a grid element.
             *  @param object The element, whose destination grid column is to be set.
             *  @param column The new grid column of object.
             */
            static void SetColumn(DynamicPropertyHost &object, UInt8 column);


            /**
             *  Retrieves the grid row span of a grid element.
             *  @param object The element, whose grid row span is sought.
             *  @return The grid row span of a grid element.
             */
            static UInt8 GetRowSpan(const DynamicPropertyHost& object);

            /**
            *  Sets the grid row span the grid element shall use.
            *  @param object The element, whose grid row span is to be set.
            *  @param rowSpan
            */
            static void SetRowSpan(DynamicPropertyHost& object, UInt8 rowSpan);

            /**
            *  Retrieves the grid column span of a grid element.
            *  @param object The element, whose grid column span is sought.
            *  @return The grid column span of a grid element.
            */
            static UInt8 GetColumnSpan(const DynamicPropertyHost& object);

            /**
            *  Sets the grid column span the grid element shall use.
            *  @param object The element, whose grid column span is to be set.
            *  @param columnSpan
            */
            static void SetColumnSpan(DynamicPropertyHost& object, UInt8 columnSpan);

            /**
             *  Provides the parent of a dynamic property host for dynamic property value propagation.
             *  This function always returns 0, since layouters are no nodes and as such are
             *  not directly integrated in the child/parent relationships of scene graphs.
             *  This function is still required for general dynamic property support.
             *  @param host The dynamic property host, whose parent (considering property value
             *              propagation) is sought. This parameter is ignored.
             *  @return Always 0.
             */
            FEATSTD_LINT_NEXT_EXPRESSION(1511, "base class method made inaccessible with using statement")
            static const Candera::DynamicProperties::DynamicPropertyHost* ParentProvider(const Candera::DynamicProperties::DynamicPropertyHost * host) {
                FEATSTD_UNUSED(host);
                return 0;
            }

            // overrides Layouter::Clone
            virtual Layouter* Clone() const override;
             /**
             * Sets the grid autoArrangement
             *      GridAutoArrangementDisabled - the row and column of each child node will be used for layout
             *      GridAutoArrangementHorizontal - the child nodes will be arrayed horizontally, left to right, top to bottom
             *      GridAutoArrangementVertical - the child nodes will be arrayed vertically, top to bottom, left to right
             *  @param node The node to set the automatic arrangement for.
             *  @param arrangement The automatic arrangement setting for this node.
             */
            static void SetGridAutoArrangement(CanderaObject& node, GridAutoArrangement arrangement);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                static void SetGridAutoArrangement(Node2D& node, GridAutoArrangement arrangement));
#endif

            /**
             *  Retrieves the autoArrangement setting of a given node, set by SetGridAutoArrangement
             *  @param node The node to retrieve the autoArrangement for.
             *  @return The autoArrangement of the node.
             */
            static GridAutoArrangement GetGridAutoArrangement(CanderaObject& node);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                static GridAutoArrangement GetGridAutoArrangement(Node2D& node));
#endif

            /**
             * Provides the effective position and size of the given row.
             * NOTE: A correct position and size is only available after the layout has been performed.
             *  @param row The row.
             *  @param pos The effective position of the given row will be stored here.
             *  @param size The effective size of the given row will be stored here.
             */
            void GetEffectiveRowData(UInt8 row, Float& pos, Float& size) const
            {
                GetEffectiveData(row, m_rows, m_rowData, pos, size);
            }

            /**
             * Provides the effective position and size of the given column.
             * NOTE: A correct position and size is only available after the layout has been performed.
             *  @param column The column.
             *  @param pos The effective position of the given column will be stored here.
             *  @param size The effective size of the given column will be stored here.
             */
            void GetEffectiveColumnData(UInt8 column, Float& pos, Float& size) const
            {
                GetEffectiveData(column, m_columns, m_columnData, pos, size);
            }

        protected:
            GridLayouter();
            explicit GridLayouter(const GridLayouter& rhs);
            virtual ~GridLayouter() override;

            /**
            *  Implements virtual pure method from Layouter. @see Layouter::OnMeasure.
            *  @param node The node which child nodes shall be layouted.
            *  @param clientArea Area where layout shall be applied.
            *  @return Vector describing height and width of whole layouted rectangle.
            */
            virtual Vector2 OnMeasure(const AbstractNodePointer& node, const Vector2& clientArea) override;

            /**
            *  Implements virtual pure method from Layouter. @see Layouter::OnArrange.
            *  @param node Node which children shall be arranged.
            *  @param clientArea Rectangle where layout shall be applied.
            */
            virtual void OnArrange(const AbstractNodePointer& node, const Rectangle& clientArea) override;

        private:
            friend class GridLayouterDynamicProperties;
            UInt8 m_rows;
            UInt8 m_columns;

            struct CellData {
                CellData() :
                    m_node(),
                    m_next(0),
                    m_measured(false)
                {
                }

                AbstractNodePointer m_node;
                CellData* m_next;
                bool m_measured;
            };
            struct Data {
                Float m_param;      // the parametrized width (SetColumnWidth) / the parametrized height (SetRowHeight)
                Float m_pos;        // the calculated column x-position        / the calculated row y-position
                Float m_size;       // the calculated column width             / the calculated row height
                Float m_tempSize;   // the incomplete calculated column width  / the incomplete calculated row height

                Data() :
                    m_param(-1.0F),   // auto size
                    m_pos(0.0F),
                    m_size(-1.0F),
                    m_tempSize(-1.0F)
                {
                }
            };

            Data* m_rowData;
            Data* m_columnData;
            CellData* m_cellData;
            CellData* m_cellPoolHead;

            using Layouter::ParentProvider;

            static void GetEffectiveData(UInt8 index, UInt8 count, const Data* data, Float& pos, Float& size)
            {
                if (index < count) {
                    pos = data[index].m_pos;
                    size = data[index].m_size;
                }
                else {
                    pos = 0.0F;
                    size = 0.0F;
                }
            }

            void ClearCache();

            //OnMeasure helpers.
            void PreprocessCells(const AbstractNodePointer& node);
            static void PreprocessData(Data* data, UInt8 dataLength);
            static void UpdateAutoData(Data* data, UInt8 dataLength);
            static bool IsDataConfigurationComplete(const Data* data, UInt8 dataLength);
            static bool UpdateFractionData(GridLayouter::Data* data, UInt8 dataLength, Float availableSize);
            static void UpdateFractionDataForInfiniteSize(Data* data, UInt8 dataLength);
            static bool TryGetDataSize(const Data* data, UInt8 index, Float& size);
            static bool TryGetDataSpanSize(const Data* data, UInt8 dataLength, UInt8 index, UInt8 span, Float& size);
            static Float GetFixedDataSpanSize(const Data* data, UInt8 dataLength, UInt8 index, UInt8 span);
            static UInt8 CountAutoDataSpan(const Data* data, UInt8 dataLength, UInt8 index, UInt8 span);
            static void UpdateAutoDataMaxSize(Data* data, UInt8 index, Float size);
            static void UpdateSpanAutoDataMaxSize(Data* data, UInt8 dataLength, UInt8 index, UInt8 span, Float size);
            static Float GetCompleteDataSize(const Data* data, UInt8 dataLength);

            static bool IsAutoData(const Data* data, UInt8 index) { return IsAutoParam(data[index].m_param); }
            static bool IsAutoParam(Float param) { return param <= 0.0F; }

            static bool IsFixedData(const Data* data, UInt8 index) { return IsFixedParam(data[index].m_param); }
            static bool IsFixedParam(Float param) { return param >= 1.0F; }

            static bool IsFractionData(const Data* data, UInt8 index) { return IsFractionParam(data[index].m_param); }
            static bool IsFractionParam(Float param) { return !(IsAutoParam(param) || IsFixedParam(param)); }

            static void OnCellChanged(DynamicPropertyHost* obj, const DynamicProperties::ValueChangedArgs<UInt8>& /*args*/) { InvalidateLayout(obj); }
            static void OnAutoArrangementChanged(DynamicPropertyHost* obj, const DynamicProperties::ValueChangedArgs<GridAutoArrangement>& /*args*/) { InvalidateLayout(obj); }

            UInt8 GetLayoutRowIndex(const CanderaObject& node, UInt32 iterationIndex, GridAutoArrangement arrangement) const;
            UInt8 GetLayoutColumnIndex(const CanderaObject& node, UInt32 iterationIndex, GridAutoArrangement arrangement) const;

            CdaDynamicPropertiesDeclaration();
    };

 /** @} */ // end of Layout

}   // namespace Candera

#endif  // CANDERA_GRIDLAYOUTER_H
