//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "DockPanelLayouter.h"

#include "Candera/EngineBase/Layout/ArabicLayouterPatch.h"

namespace Candera {
    class DockPanelLayouterDynamicProperties
    {
    private:
        friend class DockPanelLayouter;

        static bool SetDockSide(CanderaObject& node, DockSide dockSide)
        {
            return node.SetValue(CdaDynamicPropertyInstance(DockSide), dockSide);
        }

        static DockSide GetDockSide(const CanderaObject& node)
        {
            return node.GetValue(CdaDynamicPropertyInstance(DockSide));
        }

        CdaDynamicPropertiesDefinition(DockPanelLayouter, Candera::Layouter);
            CdaDynamicProperty(DockSide, Candera::DockSide);
                CdaDynamicPropertyValueChangedCb(&DockPanelLayouter::OnDockSideChanged);
                CdaDynamicPropertyDefaultValue(DockPanelLayouter::DockSideDefault());
                CdaDynamicPropertyFlags(Candera::DynamicProperties::BrowsableForDescendants);
                CdaDynamicPropertyDescription("This value indicates the position of a child element within a parent DockPanel.")
                CdaDynamicPropertyCategory("Layout")
            CdaDynamicPropertyEnd();
        CdaDynamicPropertiesEnd();
    };

    const Candera::DynamicProperties::PropertyHierarchyNode&  DockPanelLayouter::GetPropertyHierarchy() const
    {
        return DockPanelLayouterDynamicProperties::GetPropertyHierarchy();
    }


#ifdef CANDERA_2D_ENABLED
    bool DockPanelLayouter::SetDockSide(Node2D& node, DockSide dockSide)
    {
        return DockPanelLayouterDynamicProperties::SetDockSide(node, dockSide);
    }
#endif

    bool DockPanelLayouter::SetDockSide(CanderaObject& node, DockSide dockSide)
    {
        return DockPanelLayouterDynamicProperties::SetDockSide(node, dockSide);
    }

#ifdef CANDERA_2D_ENABLED
    DockSide DockPanelLayouter::GetDockSide(const Node2D& node)
    {
        return DockPanelLayouterDynamicProperties::GetDockSide(node);
    }
#endif

    DockSide DockPanelLayouter::GetDockSide(const CanderaObject& node)
    {
        return DockPanelLayouterDynamicProperties::GetDockSide(node);
    }

    /******************************************************************************
     *  DockPanelLayouter
     ******************************************************************************/
    DockPanelLayouter::DockPanelLayouter()
    {
    }

    /******************************************************************************
     *  GetInstance
     ******************************************************************************/
    DockPanelLayouter* DockPanelLayouter::GetInstance()
    {
        static DockPanelLayouter s_overlayLayouter;
        return &s_overlayLayouter;
    }

    /************************************************************************
    * GetDockSide
    ************************************************************************/
#ifdef CANDERA_2D_ENABLED
    DockSide DockPanelLayouter::GetLanguageSensitiveDockSide(const Node2D& node)
    {
        AbstractNodePointer abstractNode(const_cast<Node2D*>(&node));
        return GetLanguageSensitiveDockSide(abstractNode);
    }
#endif

    DockSide DockPanelLayouter::GetLanguageSensitiveDockSide(const AbstractNodePointer& node)
    {
        DockSide dockSide = (node.IsValid()) ? DockPanelLayouter::GetDockSide(*node.ToCanderaObject()) : DockSideDefault();

        // swap left /right dock side in case of right-to-left text direction
        if (GetLanguageSensitiveDirection(node) == LayoutAlignment::LayoutDirection::RightToLeftDirection) {
            if (dockSide == DockLeft) {
                dockSide = DockRight;
            }
            else if (dockSide == DockRight) {
                dockSide = DockLeft;
            }
            else {
                /* do nothing */
            }
        }

        return dockSide;
    }

    /******************************************************************************
     *  OnMeasure
     ******************************************************************************/
    Vector2 DockPanelLayouter::OnMeasure(const AbstractNodePointer& node, const Vector2& clientArea)
    {
        Vector2 usedArea;
        AbstractNodePointer child = node.GetFirstChild();
        MeasureChildren(child, clientArea, usedArea);
        return usedArea;
    }

    /******************************************************************************
     *  MeasureChildren
     ******************************************************************************/
#ifdef CANDERA_2D_ENABLED
    void DockPanelLayouter::MeasureChildren(Node2D* firstChild, const Vector2& clientArea, Vector2& usedArea) const
    {
        AbstractNodePointer abstractNode(firstChild);
        MeasureChildren(abstractNode, clientArea, usedArea);
    }
#endif

    void DockPanelLayouter::MeasureChildren(const AbstractNodePointer& firstChild, const Vector2& clientArea, Vector2& usedArea) const
    {
        Vector2 childArea(clientArea);
        usedArea.SetZero();
        AbstractNodePointer child = firstChild;
        Float y = 0.0F;
        Float x = 0.0F;
        Float height= 0.0F;
        Float width = 0.0F;
        while (child.IsValid()) {
            Layouter* childLayouter = child.GetLayouter();
            AbstractNodePointer next = child.GetNextSibling();
            if (0 != childLayouter) {
                childLayouter->Measure(child, childArea);
                Vector2 preferredSize = GetClientSize(child);
                if (next.IsValid()) {
                    DockSide dockSide = GetDockSide(*child.ToCanderaObject());
                    switch (dockSide) {
                    case DockTop:
                    case DockBottom:
                        y += preferredSize.GetY();
                        if ((x + preferredSize.GetX()) > width) {
                            width = x + preferredSize.GetX();
                        }
                        childArea.SetY((childArea.GetY() - preferredSize.GetY()));
                        if (childArea.GetY() < 0.0F) {
                            childArea.SetY(0.0F);
                        }
                        break;
                    case DockLeft:
                    case DockRight:
                        x += preferredSize.GetX();
                        if ((y + preferredSize.GetY()) > height) {
                            height = y + preferredSize.GetY();
                        }
                        childArea.SetX(childArea.GetX() - preferredSize.GetX());
                        if (childArea.GetX() < 0.0F) {
                            childArea.SetX(0.0F);
                        }
                        break;
                    default:
                        break;
                    }
                }
                else {
                    if ((x + preferredSize.GetX()) > width) {
                        width = x + preferredSize.GetX();
                    }
                    if ((y + preferredSize.GetY()) > height) {
                        height = y + preferredSize.GetY();
                    }
                }
            }
            child = next;
        }
        usedArea.SetX(width);
        usedArea.SetY(height);
    }

    /******************************************************************************
     *  OnArrange
     ******************************************************************************/
    void DockPanelLayouter::OnArrange(const AbstractNodePointer& node, const Candera::Rectangle& clientArea)
    {
        Node2D* node2D = node.ToNode2D();
        if (0 == node2D || ArabicLayouterPatch::IsSceneEnabled(*node2D)){
            /** ---- Patched Layout ---- **/
            LayoutAlignment::LayoutDirection::Enum layoutDirection = GetLanguageSensitiveDirection(node);
            AbstractNodePointer child = node.GetFirstChild();
            ArrangeChildren(layoutDirection, child, Vector2(), clientArea.GetSize());
            /** ---- Patched Layout end ---- **/
        }
        else {
            /** ---- Old Layout ---- **/
            LayoutAlignment::LayoutDirection::Enum layoutDirection = GetLanguageSensitiveDirection(node);
            ArrangeChildren(layoutDirection, node.GetFirstChild(), Vector2(), clientArea.GetSize());
            SetNodePosition(node, clientArea.GetPosition());
            /** ---- Old Layout end ---- **/
        }
    }

    /******************************************************************************
     *  ArrangeChildren
     ******************************************************************************/
#ifdef CANDERA_2D_ENABLED
    void DockPanelLayouter::ArrangeChildren(LayoutAlignment::LayoutDirection::Enum layoutDirection, Node2D* firstChild, const Candera::Vector2& contentAreaPosition, const Candera::Vector2& contentAreaSize) const
    {
        AbstractNodePointer abstractNode(firstChild);
        ArrangeChildren(layoutDirection, abstractNode, contentAreaPosition, contentAreaSize);
    }
#endif

    void DockPanelLayouter::ArrangeChildren(LayoutAlignment::LayoutDirection::Enum layoutDirection, const AbstractNodePointer& firstChild, const Candera::Vector2& contentAreaPosition, const Candera::Vector2& contentAreaSize) const
    {
        AbstractNodePointer child = firstChild;
        Rectangle remainingArea(contentAreaPosition, contentAreaSize);
        Float actualWidth = 0.0F;
        Float actualHeight = 0.0F;
        Node2D* firstChild2D = firstChild.ToNode2D();
        if (0 == firstChild2D || ArabicLayouterPatch::IsSceneEnabled(*firstChild2D)){
            /** ---- Patched Layout ---- **/
            while (child.IsValid()) {
                Layouter* childLayouter = child.GetLayouter();
                AbstractNodePointer next = child.GetNextSibling();
                if (childLayouter != 0) {
                    Rectangle childArea(remainingArea);
                    Vector2 preferredSize = GetClientSize(child);
                    if (next.IsValid()) {
                        DockSide dockSide = GetDockSide(*child.ToCanderaObject());
                        if (layoutDirection == LayoutAlignment::LayoutDirection::RightToLeftDirection) {
                            switch (dockSide)
                            {
                            case DockRight:
                                dockSide = DockLeft;
                                break;
                            case DockLeft:
                                dockSide = DockRight;
                                break;
                            default:
                                break;
                            }
                        }
                        const Float preferredWidth = preferredSize.GetX();
                        const Float preferredHeight = preferredSize.GetY();
                        switch (dockSide) {
                        case DockBottom: {
                                             const Float remainingHeight = remainingArea.GetHeight();
                                             childArea.SetTop((remainingArea.GetTop() + remainingHeight) - preferredHeight);
                                             childArea.SetHeight(preferredHeight);
                                             remainingArea.SetHeight(remainingHeight - preferredHeight);
                                             break;
                        }

                        case DockRight: {
                                            const Float remainingWidth = remainingArea.GetWidth();
                                            childArea.SetLeft((remainingArea.GetLeft() + remainingWidth) - preferredWidth);
                                            childArea.SetWidth(preferredWidth);
                                            remainingArea.SetWidth(remainingWidth - preferredWidth);
                                            break;
                        }

                        case DockTop:
                            childArea.SetHeight(preferredHeight);
                            remainingArea.SetTop(remainingArea.GetTop() + preferredHeight);
                            remainingArea.SetHeight(remainingArea.GetHeight() - preferredHeight);
                            break;

                        case DockLeft:
                            childArea.SetWidth(preferredWidth);
                            remainingArea.SetLeft(remainingArea.GetLeft() + preferredWidth);
                            remainingArea.SetWidth(remainingArea.GetWidth() - preferredWidth);
                            break;

                        default:
                            break;
                        }
                    }
                    childLayouter->Arrange(child, childArea);
                    Vector2 childSize = GetArrangeActualSize();
                    if (actualWidth < childSize.GetX()) {
                        actualWidth = childSize.GetX();
                    }
                    if (actualHeight < childSize.GetY()) {
                        actualHeight = childSize.GetY();
                    }
                }
                child = next;
            }
            SetArrangeActualSize(Vector2(actualWidth, actualHeight));
            /** ---- Patched Layout end ---- **/
        }
        else {
            /** ---- Old Layout ---- **/
            Node2D* child = firstChild2D;
            Rectangle remainingArea(contentAreaPosition, contentAreaSize);
            while (child != 0) {
                Layouter* childLayouter = child->GetLayouter();
                if (childLayouter != 0) {
                    Rectangle childArea(remainingArea);
                    Vector2 preferredSize = GetClientSize(*child);
                    if (child->GetNextSibling() != 0) {
                        DockSide dockSide = GetDockSide(*child);
                        if (layoutDirection == LayoutAlignment::LayoutDirection::RightToLeftDirection) {
                            switch (dockSide) {
                            case DockRight:
                                dockSide = DockLeft;
                                break;
                            case DockLeft:
                                dockSide = DockRight;
                                break;
                            default:
                                break;
                            }
                        }
                        const Float preferredWidth = preferredSize.GetX();
                        const Float preferredHeight = preferredSize.GetY();
                        switch (dockSide) {
                        case DockBottom: {
                                             const Float remainingHeight = remainingArea.GetHeight();
                                             childArea.SetTop((remainingArea.GetTop() + remainingHeight) - preferredHeight);
                                             childArea.SetHeight(preferredHeight);
                                             remainingArea.SetHeight(remainingHeight - preferredHeight);
                                             break;
                        }

                        case DockRight: {
                                            const Float remainingWidth = remainingArea.GetWidth();
                                            childArea.SetLeft((remainingArea.GetLeft() + remainingWidth) - preferredWidth);
                                            childArea.SetWidth(preferredWidth);
                                            remainingArea.SetWidth(remainingWidth - preferredWidth);
                                            break;
                        }

                        case DockTop:
                            childArea.SetHeight(preferredHeight);
                            remainingArea.SetTop(remainingArea.GetTop() + preferredHeight);
                            remainingArea.SetHeight(remainingArea.GetHeight() - preferredHeight);
                            break;

                        case DockLeft:
                            childArea.SetWidth(preferredWidth);
                            remainingArea.SetLeft(remainingArea.GetLeft() + preferredWidth);
                            remainingArea.SetWidth(remainingArea.GetWidth() - preferredWidth);
                            break;

                        default:
                            break;
                        }
                    }
                    childLayouter->Arrange(*child, childArea);
                }
                child = child->GetNextSibling();
            }
            /** ---- Old Layout end ---- **/
        }
    }
}
