//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include <FeatStd/Util/StaticObject.h>
#include "DefaultLayouter.h"
#include <Candera/System/MemoryManagement/MemoryManagement.h>

#ifdef CANDERA_2D_ENABLED
#include <CanderaPlatform/Device/Common/Effects/BitmapBrush.h>
#include <Candera/Engine2D/Core/RenderNode.h>
#include <Candera/Engine2D/Core/Group2D.h>
#include <Candera/Engine2D/Effects/Effect2D.h>
#endif

#ifdef CANDERA_3D_ENABLED
#include <Candera/Engine3D/Core/Mesh.h>
#ifdef CANDERA_3D_CANVAS_ENABLED
#include <Candera/Engine3D/Canvas/CanvasText.h>
#include <Candera/Engine3D/Canvas/CanvasSprite.h>
#endif
#include <Candera/Engine3D/Core/Billboard.h>
#endif

#include "Candera/EngineBase/Layout/ArabicLayouterPatch.h"

namespace Candera {
    class InternalDefaultLayouterProperties {
    public:
        static bool SetInternalPreferredSize(CanderaObject& node, Vector2 dockSide) {
            return node.SetValue(CdaDynamicPropertyInstance(InternalPreferredSize), dockSide);
        }

        static Vector2 GetInternalPreferredSize(const CanderaObject& node) {
            return node.GetValue(CdaDynamicPropertyInstance(InternalPreferredSize));
        }

        static const Candera::DynamicProperties::DynamicPropertyHost* ParentProvider(const Candera::DynamicProperties::DynamicPropertyHost* host) {
            return Layouter::ParentProvider(host);
        }

    private:
        static const Vector2& DefaultInternalPreferredSize() {
            FEATSTD_UNSYNCED_STATIC_OBJECT(Vector2, s_defaultPreferredSize);
            return s_defaultPreferredSize;
        }

        CdaDynamicProperties(InternalDefaultLayouterProperties, Candera::DynamicProperties::DynamicPropertyHost);
            CdaDynamicProperty(InternalPreferredSize, Candera::Vector2);
                CdaDynamicPropertyDefaultValue(DefaultInternalPreferredSize());
            CdaDynamicPropertyEnd();
        CdaDynamicPropertiesEnd();
    };

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    DefaultLayouter::DefaultLayouter()
    {
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    DefaultLayouter::~DefaultLayouter()
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    DefaultLayouter* DefaultLayouter::GetInstance()
    {
        FEATSTD_UNSYNCED_STATIC_OBJECT(DefaultLayouter, s_defaultLayouter);
        return &s_defaultLayouter;
    }

    /******************************************************************************
     *  OnMeasure
     ******************************************************************************/
    Vector2 DefaultLayouter::OnMeasure(const AbstractNodePointer& node, const Vector2& clientArea)
    {
        if (node.IsValid()) {
            Node2D* node2D = node.ToNode2D();
            if (0 == node2D || ArabicLayouterPatch::IsSceneEnabled(*node2D)){
                /** ---- Patched Layout ---- **/
                AbstractNodePointer child = node.GetFirstChild();
                FeatStd::Float width = 0.0F;
                FeatStd::Float height = 0.0F;
                // collect and merge the preferred size of all children (not yet rotated and without stretch)
                while (child.IsValid()) {
                    Layouter* layouter = child.GetLayouter();
                    if (layouter != 0) {
                        layouter->Measure(child, Vector2(Math::MaxFloat(), Math::MaxFloat()));
                        Vector2 size = (GetPreferredSize(child) + child.GetPosition());
                        if (size.GetX() > width) {
                            width = size.GetX();
                        }
                        if (size.GetY() > height) {
                            height = size.GetY();
                        }
                    }
                    child = child.GetNextSibling();
                }
                // collect and merge the preferred size of this RenderNode (if it is a RenderNode) (not yet rotated and without stretch)
                switch (node.GetType()) {
#ifdef CANDERA_2D_ENABLED
                case AbstractNodePointer::Candera2D:
                    if (node.IsTypeOf(RenderNode::GetTypeId())) {
                        Rectangle boundingRect;
                        node.ToNode2D()->GetComputedLayoutRectangle(boundingRect);
                        Vector2 size = boundingRect.GetSize();
                        if (size.GetX() > width) {
                            width = size.GetX();
                        }
                        if (size.GetY() > height) {
                            height = size.GetY();
                        }
                    }
                    break;
#endif
#ifdef CANDERA_3D_ENABLED
                case AbstractNodePointer::Candera3D:
                    // TODO: handle further 3D primitive nodes like the mesh...
                    if (
                        (node.IsTypeOf(Mesh::GetTypeId())) ||
                        (node.IsTypeOf(Billboard::GetTypeId()))
#ifdef CANDERA_3D_CANVAS_ENABLED
                        || (node.IsTypeOf(CanvasSprite::GetTypeId())) ||
                        (node.IsTypeOf(CanvasText::GetTypeId()))
#endif
                        ) {
                        Node* nodePtr = node.ToNode();
                        FEATSTD_DEBUG_ASSERT(nodePtr);
                        if (nodePtr->IsBoundingBoxValid() || nodePtr->ComputeBoundingBox()) {
                            Vector3 min;
                            Vector3 max;
                            nodePtr->GetBoundingBox(min, max);
                            Vector2 size(max.GetX() - min.GetX(), max.GetY() - min.GetY());
                            if (size.GetX() > width) {
                                width = size.GetX();
                            }
                            if (size.GetY() > height) {
                                height = size.GetY();
                            }
                        }
                    }
                    break;
#endif
                case AbstractNodePointer::Undefined:
                    break;

                default:
                    break;
                }
                // width and height contains now the preferred size of the sub tree before considering any size setting
                FeatStd::Float clientWidth = clientArea.GetX();
                FeatStd::Float clientHeight = clientArea.GetY();
                if (Layouter::IsSizeSet(*node.ToCanderaObject())) {
                    Vector2 size = Layouter::GetSize(*node.ToCanderaObject());
                    if ((size.GetX() > 0.0F) && (size.GetX() > clientWidth)) {
                        clientWidth = size.GetX();
                    }
                    if ((size.GetY() > 0.0F) && (size.GetY() > clientHeight)) {
                        clientHeight = size.GetY();
                    }
                }
                else {
                    // clientWidth and clientHeight are already set to clientArea
                }
                Vector2 preferredScale(1.0F, 1.0F);
                // obtain the scale (due to configuration or a stretch setting)
                if (Layouter::GetStretchBehavior(*node.ToCanderaObject()) != LayoutAlignment::StretchBehavior::None) {
                    // the width and height is needed for further processing in the OnArrange to calculate the final scale factor
                    static_cast<void>(InternalDefaultLayouterProperties::SetInternalPreferredSize(*node.ToCanderaObject(), Vector2(width, height)));
                    // for now we can set the preliminary scale factor. it may be required by other layouters.
                    preferredScale = GetScale(node, Vector2(clientWidth, clientHeight), Vector2(width, height));
                    if (Layouter::IsActualSizeAvailable(node)) {
                        SetActualSize(node, Vector2(-1.0F, -1.0F));
                    }
                    else {
                        SetNodeScale(node, preferredScale);
                    }
                    preferredScale = Vector2(1.0F, 1.0F);
                }
                else {
                    preferredScale = node.GetScale();
                }
                // if any fix size is configured return this a preferred size
                bool isWidthConfigured = false;
                bool isHeightConfigured = false;
                if (Layouter::IsSizeSet(*node.ToCanderaObject())) {
                    Vector2 size = Layouter::GetSize(*node.ToCanderaObject());
                    if (size.GetX() > 0.0F) {
                        width = size.GetX();
                        isWidthConfigured = true;
                    }
                    if (size.GetY() > 0.0F) {
                        height = size.GetY();
                        isHeightConfigured = true;
                    }
                }
                if ((!isWidthConfigured) || (!isHeightConfigured)) {
                    switch (node.GetType()) {
#ifdef CANDERA_2D_ENABLED
                    case AbstractNodePointer::Candera2D:
                        if (node.ToNode2D()->IsBoundingRectangleSet()) {
                            Vector2 size = node.ToNode2D()->GetBoundingRectangle().GetSize();
                            if ((!isWidthConfigured) && (size.GetX() > 0.0F)) {
                                width = size.GetX();
                            }
                            if ((!isHeightConfigured) && (size.GetY() > 0.0F)) {
                                height = size.GetY();
                            }
                        }
                        break;
#endif
                    case AbstractNodePointer::Undefined:
                        break;

                    default:
                        break;
                    }
                }

                Vector2 scaledSize = Vector2(width, height) * preferredScale;
                if (scaledSize.GetX() < 0.0F) {
                    scaledSize.SetX(-scaledSize.GetX());
                }

                if (scaledSize.GetY() < 0.0F) {
                    scaledSize.SetY(-scaledSize.GetY());
                }
                return scaledSize;
                /** ---- Patched Layout end ---- **/
            }
            else {
                /** ---- Old Layout ---- **/
                Node2D* child = (*node2D).GetFirstChild();
                FeatStd::Float width = 0.0F;
                FeatStd::Float height = 0.0F;
                // the configured scaling has to be considered (if stretch behavior is set none and a scale is set on a node)
                Vector2 measureScale = (Layouter::GetStretchBehavior((*node2D)) == LayoutAlignment::StretchBehavior::None) ? node.GetScale() : Vector2(1.0F, 1.0F);
                if (measureScale.GetX() < 0.0F) {
                    measureScale.SetX(-measureScale.GetX());
                }

                if (measureScale.GetY() < 0.0F) {
                    measureScale.SetY(-measureScale.GetY());
                }

                // collect and merge the preferred size of all children (not yet rotated and without stretch)
                while (child != 0) {
                    Layouter* layouter = child->GetLayouter();
                    if (layouter != 0) {
                        layouter->Measure(*child, Vector2(Math::MaxFloat(), Math::MaxFloat()));
                        Vector2 size = (GetPreferredSize(AbstractNodePointer(child)) + child->GetPosition()) * measureScale;
                        if (size.GetX() > width) {
                            width = size.GetX();
                        }
                        if (size.GetY() > height) {
                            height = size.GetY();
                        }
                    }
                    child = child->GetNextSibling();
                }
                // collect and merge the preferred size of this RenderNode (if it is a RenderNode) (not yet rotated and without stretch)
                if (node2D->IsTypeOf(RenderNode::GetTypeId())) {
                    Rectangle boundingRect;
                    node2D->GetComputedLayoutRectangle(boundingRect);
                    Vector2 size = boundingRect.GetSize();
                    if (size.GetX() > width) {
                        width = size.GetX();
                    }
                    if (size.GetY() > height) {
                        height = size.GetY();
                    }
                }
                // width and height contains now the preferred size of the sub tree before considering any size setting
                FeatStd::Float clientWidth = clientArea.GetX();
                FeatStd::Float clientHeight = clientArea.GetY();
                if (Layouter::IsSizeSet(*node2D)) {
                    Vector2 size = Layouter::GetSize(*node2D);
                    if ((size.GetX() > 0.0F) && (size.GetX() > clientWidth)) {
                        clientWidth = size.GetX();
                    }
                    if ((size.GetY() > 0.0F) && (size.GetY() > clientHeight)) {
                        clientHeight = size.GetY();
                    }
                }
                else {
                    // clientWidth and clientHeight are already set to clientArea
                }
                // obtain the scale (due to configuration or a stretch setting)
                if (Layouter::GetStretchBehavior(*node2D) != LayoutAlignment::StretchBehavior::None) {
                    // the width and height is needed for further processing in the OnArrange to calculate the final scale factor
                    InternalDefaultLayouterProperties::SetInternalPreferredSize(*node2D, Vector2(width, height));
                    // for now we can set the preliminary scale factor. it may be required by other layouters.
                    Vector2 preferredScale(1.0F, 1.0F);
                    preferredScale = GetScale(node, Vector2(clientWidth, clientHeight), Vector2(width, height));
                    SetNodeScale(node, preferredScale);
                }
                // if any fix size is configured return this a preferred size
                bool isWidthConfigured = false;
                bool isHeightConfigured = false;
                if (Layouter::IsSizeSet(*node2D)) {
                    Vector2 size = Layouter::GetSize(*node2D);
                    if (size.GetX() > 0.0F) {
                        width = size.GetX();
                        isWidthConfigured = true;
                    }
                    if (size.GetY() > 0.0F) {
                        height = size.GetY();
                        isHeightConfigured = true;
                    }
                }
                if (((!isWidthConfigured) || (!isHeightConfigured)) && node2D->IsBoundingRectangleSet()) {
                    Vector2 size = node2D->GetBoundingRectangle().GetSize();
                    if ((!isWidthConfigured) && (size.GetX() > 0.0F)) {
                        width = size.GetX();
                    }
                    if ((!isHeightConfigured) && (size.GetY() > 0.0F)) {
                        height = size.GetY();
                    }
                }
                if (Layouter::GetStretchBehavior(*node2D) != LayoutAlignment::StretchBehavior::None) {
                    return GetAxisAlignedDimension(Vector2(width, height), node.GetRotation());
                }
                Vector2 scaledSize = Vector2(width, height) * node.GetScale();
                if (scaledSize.GetX() < 0.0F) {
                    scaledSize.SetX(-scaledSize.GetX());
                }

                if (scaledSize.GetY() < 0.0F) {
                    scaledSize.SetY(-scaledSize.GetY());
                }

                return GetAxisAlignedDimension(scaledSize, node.GetRotation());
                /** ---- Old Layout end ---- **/
            }
        }
        return Vector2();
    }

    /******************************************************************************
     *  OnArrange
     ******************************************************************************/
    void DefaultLayouter::OnArrange(const AbstractNodePointer& node, const Rectangle& clientArea)
    {
        if (node.IsValid()) {
            Node2D* node2D = node.ToNode2D();
            if (0 == node2D || ArabicLayouterPatch::IsSceneEnabled(*node2D)){
                /** ---- Patched Layout ---- **/
                Vector2 actualSize = GetPreferredSize(node);
                if (0.0F != node.GetRotation()) {
                    actualSize = Layouter::GetOriginalPreferredSize(*node.ToCanderaObject());
                }
                if (Layouter::GetStretchBehavior(*node.ToCanderaObject()) != LayoutAlignment::StretchBehavior::None) {
                    // actualSize = clientArea.GetSize();
                    actualSize = InternalDefaultLayouterProperties::GetInternalPreferredSize(*node.ToCanderaObject());
                    Vector2 preferredScale = GetScale(node, clientArea.GetSize(), actualSize);
                    bool isActualSizeAvailable = Layouter::IsActualSizeAvailable(node);
                    if (!isActualSizeAvailable) {
                        SetNodeScale(node, preferredScale);
                    }
                    actualSize = actualSize * preferredScale;
                    if (isActualSizeAvailable) {
                        SetActualSize(node, actualSize);
                    }
                }
                AbstractNodePointer child = node.GetFirstChild();
                while (child.IsValid()) {
                    Layouter* layouter = child.GetLayouter();
                    if (layouter != 0) {
                        const Margin& margin = Layouter::GetMargin(*child.ToCanderaObject());
                        const Int16 hMargin = margin.GetLeft() + margin.GetRight();
                        const Int16 vMargin = margin.GetTop() + margin.GetBottom();
                        layouter->Arrange(child, Rectangle(child.GetPosition(), layouter->GetPreferredSize(child) + Vector2(static_cast<Float>(hMargin), static_cast<Float>(vMargin))));
                    }
                    child = child.GetNextSibling();
                }
                SetArrangeActualSize(actualSize);
                /** ---- Patched Layout end ---- **/
            }
            else {
                /** ---- Old Layout ---- **/
                if (Layouter::GetStretchBehavior(*node2D) != LayoutAlignment::StretchBehavior::None) {
                    Vector2 preferredScale = GetScale(node, clientArea.GetSize(), InternalDefaultLayouterProperties::GetInternalPreferredSize(*node2D));
                    SetNodeScale(node, preferredScale);
                }
                Node2D* child = node2D->GetFirstChild();
                while (child != 0) {
                    Layouter* layouter = child->GetLayouter();
                    if (layouter != 0) {
                        layouter->Arrange(*child, Rectangle(child->GetPosition(), layouter->GetPreferredSize(AbstractNodePointer(child))));
                    }
                    child = child->GetNextSibling();
                }

                Rectangle rect(clientArea);

                Rectangle boundingRect;
                GetParentSpaceAxisAlignedBoundingRect(node, boundingRect);

                Vector2 offset = node.GetPosition() - boundingRect.GetPosition();

                SetNodePosition(node, rect.GetPosition() + offset);

#if defined(CANDERA_LAYOUT_CLIPPING_ENABLED)
                RenderNode* renderNode = Dynamic_Cast<RenderNode*>(node2D);

                if (renderNode != 0) {
                    if ((boundingRect.GetWidth() > clientArea.GetWidth()) || (boundingRect.GetHeight() > clientArea.GetHeight())) {
                        renderNode->SetClippingRect(clientArea);
                    }
                    else {
                        renderNode->SetClippingRect();
                    }
                }
#endif
                /** ---- Old Layout end ---- **/
            }
        }
    }

    /******************************************************************************
     *  OnLimitPreferredSize
     ******************************************************************************/
    void DefaultLayouter::OnLimitPreferredSize(CanderaObject& node, Vector2& preferredSize, const Vector2& nodeSize, const Vector2& nodeMinSize, const Vector2& nodeMaxSize)
    {
        if (Layouter::GetStretchBehavior(node) == LayoutAlignment::StretchBehavior::Uniform) {
            Vector2 originalPreferredSize(preferredSize);
            Layouter::OnLimitPreferredSize(node, preferredSize, nodeSize, nodeMinSize, nodeMaxSize);
            if (((preferredSize.GetX() < originalPreferredSize.GetX()) || (preferredSize.GetY() < originalPreferredSize.GetY())) && (0.0F != originalPreferredSize.GetY()) && (0.0F != originalPreferredSize.GetX())) {
                FeatStd::Float valueX = originalPreferredSize.GetX() * preferredSize.GetY();
                FeatStd::Float valueY = originalPreferredSize.GetY() * preferredSize.GetX();
                // division is slower that multiplication and the check which axis has to be adapted can be done without the division.
                // therefore an additional multiplication is performed instead of a potential unnecessary division.
                if (valueX < valueY) {
                    preferredSize.SetX(valueX / originalPreferredSize.GetY());
                }
                else {
                    preferredSize.SetY(valueY / originalPreferredSize.GetX());
                }
            }
        }
        else {
            Layouter::OnLimitPreferredSize(node, preferredSize, nodeSize, nodeMinSize, nodeMaxSize);
        }
    }

    /******************************************************************************
     *  GetScale
     ******************************************************************************/
    Vector2 DefaultLayouter::GetScale(const AbstractNodePointer& node, const Vector2& clientSize, const Vector2& preferredSize) const
    {
        const LayoutAlignment::StretchBehavior::Enum stretchBehavior = Layouter::GetStretchBehavior(*node.ToCanderaObject());
        if (stretchBehavior == LayoutAlignment::StretchBehavior::None) {
            return node.GetScale();
        }
        if (stretchBehavior == LayoutAlignment::StretchBehavior::Fill) {
            FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
            return Vector2((Math::FloatAlmostEqual(clientSize.GetX(), Math::MaxFloat(), 0.001F) || (preferredSize.GetX() == 0.0F)) ? Float(1.0F) : (clientSize.GetX() / (preferredSize.GetX() + node.GetPivotOffset().GetX())),
                           (Math::FloatAlmostEqual(clientSize.GetY(), Math::MaxFloat(), 0.001F) || (preferredSize.GetY() == 0.0F)) ? Float(1.0F) : (clientSize.GetY() / (preferredSize.GetY() + node.GetPivotOffset().GetY())));
            FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()
        }
        FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
        const Float imageWidth = preferredSize.GetX() + node.GetPivotOffset().GetX();
        const Float imageHeight = preferredSize.GetY() + node.GetPivotOffset().GetY();
        FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()

        const Float scaleX = (Math::FloatAlmostEqual(clientSize.GetX(), Math::MaxFloat(), 0.001F)) ? Float(1.0F) : ((imageWidth == Float(0.0F)) ? Float(0.0F) : (clientSize.GetX() / imageWidth));
        const Float scaleY = (Math::FloatAlmostEqual(clientSize.GetY(), Math::MaxFloat(), 0.001F)) ? Float(1.0F) : ((imageHeight == Float(0.0F)) ? Float(0.0F) : (clientSize.GetY() / imageHeight));
        const Float scaleXY = (stretchBehavior == LayoutAlignment::StretchBehavior::Uniform) ? Math::Minimum(scaleX, scaleY) : Math::Maximum(scaleX, scaleY);
        return Vector2(scaleXY, scaleXY);
    }


}   // namespace Candera
