//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_BASELINELAYOUTER_H)
    #define CANDERA_BASELINELAYOUTER_H

#include <Candera/EngineBase/Layout/Layouter.h>
#include <Candera/EngineBase/DynamicProperties/DynamicProperty.h>

namespace Candera {

/** @addtogroup Layout
 *  @{
 */

    /**
     *  @brief A BaseLineLayouter behaves similar to a Horizontal StackLayouter but
     *  instead of aligning the objects to the client area of the Layouter, it aligns
     *  them to a line.
     *
     *  Horizontal stretching is disabled to avoid greedy items.
     *  Vertical stretching is disabled as result of DefaultLayouter::GetScale.
     *  The Vertical Alignment is interpreted as follows:
     *      - VTop: the top of the child area is aligned to the base line.
     *      - VBottom: the bottom of the child area is aligned to the base line.
     *      - VCenter: the point defined by the child as its base point is aligned to the base line.
     *          this is typically in the center of an image, or on the base line of a text.
     *      - VStretch: same as VCenter, as stretching is disabled.
     *  The baseline can be either fixed or automatic. The automatic baseline is configured
     *  such that all the children fit from the top of the client area.
     */
    class BaseLineLayouter : public Layouter {
        FEATSTD_TYPEDEF_BASE(Layouter);

        public:
            /**
             *  Creates an instance of this class.
             *  @return A pointer to the created object.
             */
            static BaseLineLayouter* Create();

            /**
             *  Sets the baseline offset of this layout. A negative value means
             *  automatic selection of a base line offset.
             *  The value defines the distance for the top of the client area to the line
             *  @param node The node on which to set the property.
             *  @param offset The base line offset.
             */
            static void SetBaseLineOffset(CanderaObject& node, Float offset);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                static void SetBaseLineOffset(Node2D& node, Float offset));
#endif

            /**
             *  Retrieves the baseline offset of this layout. See SetBaseLineOffset for more details.
             *  @param node The node from which to get the property.
             *  @return The base line offset.
             */
            static Float GetBaseLineOffset(CanderaObject& node);
#ifdef CANDERA_2D_ENABLED
            CANDERA_LAYOUTER_DEPRECATED_3_4_2("The 2d limited version has been replaced with the abstract node version. Please use the abstract version.",
                static Float GetBaseLineOffset(Node2D& node));
#endif

            /**
             *  The base line layouter does not provide means of inheritance from parent nodes.
             *  Overrides method from Layouter. Implements method needed by DynamicPropertyHost.
             *  @param host Unused.
             *  @return Null pointer.
             */
            FEATSTD_LINT_NEXT_EXPRESSION(1511, "base class method made inaccessible with using statement")
            static const Candera::DynamicProperties::DynamicPropertyHost* ParentProvider(const Candera::DynamicProperties::DynamicPropertyHost * host);

            virtual ~BaseLineLayouter() override;

            // overrides Layouter::Clone
            virtual Layouter* Clone() const override;

            /**
            * Disposes the instance of this class.
            */
            virtual void Dispose() { }

        protected:
            /**
             *  Implements virtual pure method from Layouter. @see Layouter::OnMeasure.
             *  @param node The node which child nodes shall be layouted.
             *  @param clientArea Area where layout shall be applied.
             *  @return Vector describing height and width of whole layouted rectangle.
             */
            virtual Vector2 OnMeasure(const AbstractNodePointer& node, const Vector2& clientArea) override;

            /**
             *  Implements virtual pure method from Layouter. @see Layouter::OnArrange.
             *  @param node Node which children shall be arranged.
             *  @param clientArea Rectangle where layout shall be applied.
             */
            virtual void OnArrange(const AbstractNodePointer& node, const Rectangle& clientArea) override;

        private:
            friend class BaseLineLayouterDynamicProperties;

            BaseLineLayouter();
            BaseLineLayouter(const BaseLineLayouter& rhs);

            static void OnBaseLineOffsetChanged(DynamicPropertyHost* obj, const DynamicProperties::ValueChangedArgs<Float>& /*args*/) { InvalidateLayout(obj); }

            using Layouter::ParentProvider;

            CdaDynamicPropertiesDeclaration();
     };

 /** @} */ // end of Layout

inline const Candera::DynamicProperties::DynamicPropertyHost* BaseLineLayouter::ParentProvider(const Candera::DynamicProperties::DynamicPropertyHost * /*host*/) {
    return 0;
}

}   // namespace Candera

#endif  // CANDERA_BASELINELAYOUTER_H
