//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "BaseLineLayouter.h"
#ifdef CANDERA_2D_ENABLED
#include <Candera/Engine2D/Core/Node2D.h>
#endif
#include <Candera/System/MemoryManagement/MemoryManagement.h>
#include <FeatStd/Util/StaticObject.h>

#include "Candera/EngineBase/Layout/ArabicLayouterPatch.h"

namespace Candera {
    class BaseLineLayouterDynamicProperties
    {
    private:
        static void SetBaseLineOffset(CanderaObject& node, Float offset) { static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(BaseLineOffset), offset)); }

        static Float GetBaseLineOffset(CanderaObject& node) { return node.GetValue(CdaDynamicPropertyInstance(BaseLineOffset)); }

        friend class BaseLineLayouter;
        CdaDynamicPropertiesDefinition(Candera::BaseLineLayouter, Candera::Layouter);
            CdaDynamicProperty(BaseLineOffset, Float);
                CdaDynamicPropertyFlags(Candera::DynamicProperties::BrowsableForNodeAndDescendants);
                CdaDynamicPropertyValueChangedCb(&BaseLineLayouter::OnBaseLineOffsetChanged);
                CdaDynamicPropertyDefaultValue(-1.F);
                CdaDynamicPropertyDescription(
                    "The offset of the baseline. Negative values make the baseline "
                    "automatic such that all objects fit from the top. Positive values "
                    "fixate the baseline at the given distance from the top.")
                CdaDynamicPropertyCategory("Layout")
            CdaDynamicPropertyEnd();
        CdaDynamicPropertiesEnd();
    };

    class BaseLineLayouterPrivateProperties : public Candera::DynamicProperties::DynamicPropertyHost
    {
    public:
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1764, node, CANDERA_LINT_REASON_NONCONST)
        static Float GetBaseLine(CanderaObject& node) { return node.GetValue(CdaDynamicPropertyInstance(BaseLine)); }

        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1764, node, CANDERA_LINT_REASON_NONCONST)
        static bool IsBaseLineSet(CanderaObject& node) { return node.IsValueSet(CdaDynamicPropertyInstance(BaseLine)); }

        static void SetBaseLine(CanderaObject& node, Float baseLine) { static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(BaseLine), baseLine)); }

        static void ClearBaseLine(CanderaObject& node) { static_cast<void>(node.ClearValue(CdaDynamicPropertyInstance(BaseLine))); }

        static const Candera::DynamicProperties::DynamicPropertyHost* ParentProvider(const Candera::DynamicProperties::DynamicPropertyHost* host) {
            return Layouter::ParentProvider(host);
        }

    private:
        CdaDynamicProperties(Candera::BaseLineLayouterPrivateProperties, Candera::DynamicProperties::DynamicPropertyHost);
            CdaDynamicProperty(BaseLine, Float);
                CdaDynamicPropertyDefaultValue(0.0F);
            CdaDynamicPropertyEnd();
        CdaDynamicPropertiesEnd();
    };

    const Candera::DynamicProperties::PropertyHierarchyNode&  BaseLineLayouter::GetPropertyHierarchy() const
    {
        return BaseLineLayouterDynamicProperties::GetPropertyHierarchy();
    }

    void BaseLineLayouter::SetBaseLineOffset(CanderaObject& node, Float offset)
    {
        BaseLineLayouterDynamicProperties::SetBaseLineOffset(node, offset);
    }

#ifdef CANDERA_2D_ENABLED
    void BaseLineLayouter::SetBaseLineOffset(Node2D& node, Float offset)
    {
        BaseLineLayouterDynamicProperties::SetBaseLineOffset(node, offset);
    }
#endif

    Float BaseLineLayouter::GetBaseLineOffset(CanderaObject& node)
    {
        return BaseLineLayouterDynamicProperties::GetBaseLineOffset(node);
    }

#ifdef CANDERA_2D_ENABLED
    Float BaseLineLayouter::GetBaseLineOffset(Node2D& node)
    {
        return BaseLineLayouterDynamicProperties::GetBaseLineOffset(node);
    }
#endif

    /******************************************************************************
     *  Create
     ******************************************************************************/
    BaseLineLayouter* BaseLineLayouter::Create()
    {
        FEATSTD_UNSYNCED_STATIC_OBJECT(BaseLineLayouter, s_baseLineLayouter);
        return &s_baseLineLayouter;
    }

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    BaseLineLayouter::BaseLineLayouter()
    {
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    BaseLineLayouter::~BaseLineLayouter()
    {
    }

    static inline Float NormalizeRotation(Float rotation)
    {
        if (rotation < 0.0F) {
            return 360.0F - NormalizeRotation(-rotation);
        }
        if (rotation < 360.0F) {
            return rotation;
        }
        Float normalizedRotation = FeatStd::Float(FeatStd::UInt(rotation) % FeatStd::UInt(360)) + (rotation - FeatStd::Float(FeatStd::UInt(rotation)));
        if (normalizedRotation > 360.0F) {
            return normalizedRotation - 360.0F;
        }
        return normalizedRotation;
    }

    static FeatStd::Float GetBaseLine(const AbstractNodePointer& node, const Vector2& childSize, const Vector2& originalPreferredSize)
    {
        Float basePointY = 0.0F;
        if (node.IsValid()) {
            switch (Layouter::GetVerticalAlignment(*node.ToCanderaObject())) {
            case VBottom:   // The whole node is above the base line.
                basePointY = childSize.GetY();
                break;
            FEATSTD_LINT_NEXT_EXPRESSION(1960, "MISRA C++ 2008 Required Rule 6-4-5, missing unconditional break from switch case")
            case VStretch:  // Stretch behaves the same as Center.
            case VCenter: { // The base line passes through the base point.
                    basePointY = BaseLineLayouter::GetBaseLineOffset(*node.ToCanderaObject());
                    if (basePointY < 0.0F) {
                        if (BaseLineLayouterPrivateProperties::IsBaseLineSet(*node.ToCanderaObject())) {
                            basePointY = BaseLineLayouterPrivateProperties::GetBaseLine(*node.ToCanderaObject());
                        }
                        else {
                            Candera::Vector2 basePoint;
                            node.GetBasePoint(basePoint);
                            basePointY = basePoint.GetY() * node.GetScale().GetY();
                        }
                    }
                    break;
                }
            FEATSTD_LINT_NEXT_EXPRESSION(1960, "MISRA C++ 2008 Required Rule 6-4-5, missing unconditional break from switch case")
            case VTop:      // The whole node is under the base line.
            default:
                break;
            }
            Float rotation = NormalizeRotation(node.GetRotation());
            if (0.0F != rotation) {
                const Float alpha = Math::DegreeToRadian(rotation);
                const Float cosAlpha = Math::Cosine(alpha);
                const Float sinAlpha = Math::Sine(alpha);
                Float basePointX = 0.0F;
                HorizontalAlignment horizontalAlignment = Layouter::GetHorizontalAlignment(*node.ToCanderaObject());
                switch (horizontalAlignment)
                {
                case Candera::HLeft:
                    basePointX = 0.0F;
                    break;
                case Candera::HRight:
                    basePointX = originalPreferredSize.GetX();
                    break;
                    FEATSTD_LINT_NEXT_EXPRESSION(1960, "MISRA C++ 2008 Required Rule 6-4-5, missing unconditional break from switch case")
                case Candera::HCenter:
                case Candera::HStretch:
                    basePointX = originalPreferredSize.GetX() * 0.5F;
                    break;
                default:
                    break;
                }
                if (rotation > 270.0F) {
                    basePointY = - basePointX * sinAlpha + basePointY * cosAlpha;
                }
                else if ((rotation > 180.0F) && (rotation < 270.0F)) {
                    basePointY = childSize.GetY() + basePointX * sinAlpha + basePointY * cosAlpha;
                }
                else if ((rotation > 90.0F) && (rotation < 180.0F)) {
                    basePointY = (childSize.GetY() - basePointX * sinAlpha) + (basePointY * cosAlpha);
                }
                else {
                    basePointY = basePointX * sinAlpha + basePointY * cosAlpha;
                }
            }
        }
        return basePointY;
    }

    /******************************************************************************
     *  OnMeasure
     ******************************************************************************/
    Vector2 BaseLineLayouter::OnMeasure(const AbstractNodePointer& node, const Vector2& clientArea)
    {
        Vector2 childArea(clientArea);
        AbstractNodePointer child = node.GetFirstChild();
        Float width = 0.0F;
        Float height = 0.0F;
        Float baseLine = 0.0F;
        Float descent = 0.0F;
        while (child.IsValid()) {
            Layouter* childLayouter = child.GetLayouter();
            if (childLayouter != 0) {
                childLayouter->Measure(child, childArea);
                Vector2 childSize = childLayouter->GetClientSize(child);
                Float childBaseLine = GetBaseLine(child, childSize, Layouter::GetOriginalPreferredSize(*child.ToCanderaObject()));
                Float childDescent = childSize.GetY() - childBaseLine;
                if (childBaseLine > baseLine) {
                    baseLine = childBaseLine;
                }
                if ((childDescent > 0.0F) && (childDescent > descent)) {
                    descent = childDescent;
                }
                width += childSize.GetX();
                if (height < childSize.GetY()) {
                    height = childSize.GetY();
                }
            }
            child = child.GetNextSibling();
        }
        Float baseLineOffset = BaseLineLayouter::GetBaseLineOffset(*node.ToCanderaObject());
        if (BaseLineLayouter::GetBaseLineOffset(*node.ToCanderaObject()) >= 0.0F) {
            baseLine = baseLineOffset;
        }
        BaseLineLayouterPrivateProperties::SetBaseLine(*node.ToCanderaObject(), baseLine);
        return Vector2(width, baseLine + descent);
    }

    /******************************************************************************
     *  OnArrange
     ******************************************************************************/
    void BaseLineLayouter::OnArrange(const AbstractNodePointer& node, const Rectangle& clientArea)
    {
        if (node.IsValid()) {
            Node2D* node2D = node.ToNode2D();
            if (0 == node2D || ArabicLayouterPatch::IsSceneEnabled(*node2D)){
                /** ---- Patched Layout ---- **/
                const Float clientWidth = clientArea.GetWidth();
                const Float clientHeight = clientArea.GetHeight();
                const LayoutAlignment::LayoutDirection::Enum orientation = GetLanguageSensitiveDirection(node);
                Float cursor = (orientation == LayoutAlignment::LayoutDirection::RightToLeftDirection) ? clientWidth : Float(0);
                AbstractNodePointer child = node.GetFirstChild();
                Float baseLine = BaseLineLayouter::GetBaseLineOffset(*node.ToCanderaObject());
                Float sizeCorrection = 0.0F;
                if (baseLine < 0.0F) {
                    baseLine = BaseLineLayouterPrivateProperties::GetBaseLine(*node.ToCanderaObject());
                    if (baseLine > clientHeight) {
                        sizeCorrection = baseLine - clientHeight;
                    }
                }
                while (child.IsValid()) {
                    Layouter* childLayouter = child.GetLayouter();
                    if (childLayouter != 0) {
                        Vector2 childSize = childLayouter->GetClientSize(child);
                        Float childLeft = 0.0F;
                        if (orientation == LayoutAlignment::LayoutDirection::RightToLeftDirection) {
                            cursor -= childSize.GetX();
                            childLeft = cursor;
                        }
                        else{
                            childLeft = cursor;
                            cursor += childSize.GetX();
                        }
                        Float childTop = baseLine - (GetBaseLine(child, childSize, Layouter::GetOriginalPreferredSize(*child.ToCanderaObject())) + sizeCorrection);
                        Rectangle childArea(childLeft, childTop, childSize.GetX(), childSize.GetY());
                        childLayouter->Arrange(child, childArea);
                    }
                    child = child.GetNextSibling();
                }
                BaseLineLayouterPrivateProperties::ClearBaseLine(*node.ToCanderaObject());
                Vector2 actualSize = GetPreferredSize(node);
                if (GetHorizontalAlignment(*node.ToCanderaObject()) == HStretch) {
                    actualSize.SetX(clientWidth);
                }
                SetArrangeActualSize(actualSize);
                /** ---- Patched Layout end ---- **/
            }
            else {
                /** ---- Old Layout ---- **/
                const Float clientWidth = clientArea.GetWidth();
                const Float clientHeight = clientArea.GetHeight();
                const LayoutAlignment::LayoutDirection::Enum orientation = GetLanguageSensitiveDirection(node);
                Float cursor = (orientation == LayoutAlignment::LayoutDirection::RightToLeftDirection) ? clientWidth : Float(0);
                Node2D* child = (*node2D).GetFirstChild();
                Float baseLine = BaseLineLayouter::GetBaseLineOffset(*node2D);
                Float sizeCorrection = 0.0F;
                if (baseLine < 0.0F) {
                    baseLine = BaseLineLayouterPrivateProperties::GetBaseLine(*node2D);
                    if (baseLine > clientHeight) {
                        sizeCorrection = baseLine - clientHeight;
                    }
                }
                while (child != 0) {
                    Layouter* childLayouter = child->GetLayouter();
                    if (childLayouter != 0) {
                        Vector2 childSize = childLayouter->GetClientSize(*child);
                        Float childLeft = 0.0F;
                        if (orientation == LayoutAlignment::LayoutDirection::RightToLeftDirection) {
                            cursor -= childSize.GetX();
                            childLeft = cursor;
                        }
                        else{
                            childLeft = cursor;
                            cursor += childSize.GetX();
                        }
                        Float childTop = baseLine - (GetBaseLine(AbstractNodePointer(child), childSize, Layouter::GetOriginalPreferredSize(*child)) + sizeCorrection);
                        Rectangle childArea(childLeft, childTop, childSize.GetX(), childSize.GetY());
                        childLayouter->Arrange(*child, childArea);
                    }
                    child = child->GetNextSibling();
                }
                BaseLineLayouterPrivateProperties::ClearBaseLine(*node2D);
                SetNodePosition(node, clientArea.GetPosition());
                /** ---- Old Layout end ---- **/
            }
        }
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Layouter* BaseLineLayouter::Clone() const
    {
        return Create();
    }
}   // namespace Candera
