//########################################################################
// (C)  Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_ARABIC_LAYOUTER_PATCH_H)
#define CANDERA_ARABIC_LAYOUTER_PATCH_H
#include <Candera/EngineBase/Common/CanderaObject.h>
#include <Candera/Engine2D/Core/Scene2D.h>
#include <CanderaWidget/Widget2D/Widget2D.h>
#include <FeatStd/Util/Optional.h>

namespace Candera {
    class ArabicLayouterPatch : public DynamicProperties::DynamicPropertyHost {
    public:
        friend class Candera::DynamicProperties::DynamicPropertyHost;

        static void SetDefaultEnabled(bool enabled)
        {
            DefaultEnabled() = enabled;
        }

        static bool IsDefaultEnabled()
        {
            return DefaultEnabled();
        }

        static void SetPartialEnabled(bool enabled)
        {
            DefaultPartial() = enabled;
        }

        static bool IsPartialEnabled()
        {
            return DefaultPartial();
        }

        static void SetOnlyForRightToLeft(bool enabled)
        {
            DefaultOnlyForRightToLeft() = enabled;
        }

        static bool IsOnlyForRightToLeft()
        {
            return DefaultOnlyForRightToLeft();
        }

        static void SetSceneEnabled(const Node2D& node, bool enabled)
        {
            Scene2D* scene = node.GetScene();
            if (0 != scene) {
                static_cast<void>(scene->SetValue(CdaDynamicPropertyInstance(ArabicLayouterPatchEnabled), static_cast<Int8>(enabled ? Enabled : Disabled)));
            }
        }

        static bool IsSceneEnabled(const Node2D& node)
        {
            bool enabled = DefaultEnabled();
            Scene2D* scene = node.GetScene();
            if ((0 != scene) && scene->IsValueSet(CdaDynamicPropertyInstance(ArabicLayouterPatchEnabled))) {
                enabled = scene->GetValue(CdaDynamicPropertyInstance(ArabicLayouterPatchEnabled)) == static_cast<Int8>(Enabled);
            }
            if (enabled && IsSceneOnlyForRightToLeft(node)) {
                enabled = LayouterHelper::IsRightToLeftDirection(node);
            }
            return enabled;
        }

        static void ClearSceneEnabled(const Node2D& node)
        {
            Scene2D* scene = node.GetScene();
            if (0 != scene) {
                static_cast<void>(scene->SetValue(CdaDynamicPropertyInstance(ArabicLayouterPatchEnabled), static_cast<Int8>(UseDefault)));
            }
        }

        static void SetScenePartial(const Node2D& node, bool partial)
        {
            Scene2D* scene = node.GetScene();
            if (0 != scene) {
                static_cast<void>(scene->SetValue(CdaDynamicPropertyInstance(ArabicLayouterPatchPartial), static_cast<Int8>(partial ? Partial : Complete)));
            }
        }

        static bool IsScenePartial(const Node2D& node)
        {
            Scene2D* scene = node.GetScene();
            if ((0 != scene) && scene->IsValueSet(CdaDynamicPropertyInstance(ArabicLayouterPatchPartial))) {
                return scene->GetValue(CdaDynamicPropertyInstance(ArabicLayouterPatchPartial)) == static_cast<Int8>(Partial);
            }
            return DefaultPartial();
        }

        static void ClearScenePartial(const Node2D& node)
        {
            Scene2D* scene = node.GetScene();
            if (0 != scene) {
                static_cast<void>(scene->SetValue(CdaDynamicPropertyInstance(ArabicLayouterPatchPartial), static_cast<Int8>(UseDefault)));
            }
        }

        static void SetSceneOnlyForRightToLeft(const Node2D& node, bool onlyForRightToLeft)
        {
            Scene2D* scene = node.GetScene();
            if (0 != scene) {
                static_cast<void>(scene->SetValue(CdaDynamicPropertyInstance(ArabicLayouterPatchOnlyForRightToLeft), static_cast<Int8>(onlyForRightToLeft ? OnlyForRightToLeft : BothDirections)));
            }
        }

        static bool IsSceneOnlyForRightToLeft(const Node2D& node)
        {
            Scene2D* scene = node.GetScene();
            if ((0 != scene) && scene->IsValueSet(CdaDynamicPropertyInstance(ArabicLayouterPatchOnlyForRightToLeft))) {
                return scene->GetValue(CdaDynamicPropertyInstance(ArabicLayouterPatchOnlyForRightToLeft)) == static_cast<Int8>(OnlyForRightToLeft);
            }
            return DefaultOnlyForRightToLeft();
        }

        static void ClearSceneOnlyForRightToLeft(const Node2D& node)
        {
            Scene2D* scene = node.GetScene();
            if (0 != scene) {
                static_cast<void>(scene->SetValue(CdaDynamicPropertyInstance(ArabicLayouterPatchOnlyForRightToLeft), static_cast<Int8>(UseDefault)));
            }
        }

    private:
        class LayouterHelper : public Layouter {
        public:
            friend class ArabicLayouterPatch;
            static bool IsRightToLeftDirection(const Node2D& node)
            {
                return LayouterHelper::GetLanguageSensitiveDirection(node) == LayoutAlignment::LayoutDirection::RightToLeftDirection;
            }
        };

        enum {
            UseDefault = 0,
            Disabled = 1,
            Enabled = 2,
            Partial = 3,
            Complete = 4,
            OnlyForRightToLeft = 5,
            BothDirections = 4
        };

        static const Candera::DynamicProperties::DynamicPropertyHost* ParentProvider(const Candera::DynamicProperties::DynamicPropertyHost* host) {
            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1774, type check above ensures cast is safe)
                return static_cast<const Node2D*>(host)->GetParent();
        }

        static bool& DefaultEnabled()
        {
            static bool s_defaultEnabled = false;
            return s_defaultEnabled;
        }

        static bool& DefaultPartial()
        {
            static bool s_defaultPartial = false;
            return s_defaultPartial;
        }

        static bool& DefaultOnlyForRightToLeft()
        {
            static bool s_defaultOnlyForRightToLeft = false;
            return s_defaultOnlyForRightToLeft;
        }

        static void InvalidateLayout(const Node2D* node)
        {
            Scene2D* scene = node->GetScene();
            if (scene != 0) {
                Layouter::InvalidateLayout(node);
            }
        }

        static void InvalidateLayout(DynamicPropertyHost* obj)
        {
            Node2D* node = FeatStd::Internal::PointerToPointer<Node2D*>(obj);
            InvalidateLayout(node);
        }

        static void OnArabicLayouterPatchEnabledChanged(DynamicPropertyHost* obj, const DynamicProperties::ValueChangedArgs<Int8>& /*args*/)
        {
            InvalidateLayout(obj);
        }

        CdaDynamicProperties(Candera::ArabicLayouterPatch, Candera::DynamicProperties::DynamicPropertyHost);
            CdaDynamicProperty(ArabicLayouterPatchEnabled, Int8);
                CdaDynamicPropertyValueChangedCb(&ArabicLayouterPatch::OnArabicLayouterPatchEnabledChanged);
                CdaDynamicPropertyDescription("Activate or deactivate the patched behavior for a scene.")
            CdaDynamicPropertyEnd();
            CdaDynamicProperty(ArabicLayouterPatchPartial, Int8);
                CdaDynamicPropertyValueChangedCb(&ArabicLayouterPatch::OnArabicLayouterPatchEnabledChanged);
                CdaDynamicPropertyDescription("The arabic patch is used partial (only in horizontal direction) or complete.")
            CdaDynamicPropertyEnd();
            CdaDynamicProperty(ArabicLayouterPatchOnlyForRightToLeft, Int8);
                CdaDynamicPropertyValueChangedCb(&ArabicLayouterPatch::OnArabicLayouterPatchEnabledChanged);
                CdaDynamicPropertyDescription("The arabic patch is only used for right to left layout direction.")
            CdaDynamicPropertyEnd();
        CdaDynamicPropertiesEnd();
    };

    class ArabicLayouterPatchWidget : public Candera::Widget2D
    {
    public:
        ArabicLayouterPatchWidget() :
            m_enabled(true),
            m_partial(false),
            m_onlyForRightToLeft(false)
        {
        }

        virtual ~ArabicLayouterPatchWidget()
        {
        }

        virtual void Update()
        {
            if (!m_valid) {
                Node2D* node = GetNode();
                if (0 != node) {
                    ArabicLayouterPatch::SetSceneEnabled(*node, m_enabled);
                    ArabicLayouterPatch::SetScenePartial(*node, m_partial);
                    ArabicLayouterPatch::SetSceneOnlyForRightToLeft(*node, m_onlyForRightToLeft);
                }
                m_valid = true;
            }
        }

        bool IsEnabled() const 
        {
            return m_enabled;
        }

        void SetEnabled(bool enabled)
        {
            if (m_enabled != enabled) {
                m_enabled = enabled;
                m_valid = false;
            }
        }

        bool IsPartial() const
        {
            return m_partial;
        }

        void SetPartial(bool partial)
        {
            if (m_partial != partial) {
                m_partial = partial;
                m_valid = false;
            }
        }

        bool IsOnlyForRightToLeft() const
        {
            return m_onlyForRightToLeft;
        }

        void SetOnlyForRightToLeft(bool onlyForRightToLeft)
        {
            if (m_onlyForRightToLeft != onlyForRightToLeft) {
                m_onlyForRightToLeft = onlyForRightToLeft;
                m_valid = false;
            }
        }

        CGI_WIDGET_RTTI_DECLARATION(ArabicLayouterPatchWidget, Candera::Widget2D);

    private:
        CdaWidget2DDef(ArabicLayouterPatchWidget, Candera::Widget2D)
            CdaDescription("ArabicLayouterPatchWidget")
            CdaCategory("Layout")
            CdaProperties()
                CdaProperty(Enabled, bool, IsEnabled, SetEnabled)
                    CdaDescription("The ArabicLayouterPatch is enabled or disabled for this scene. Last widget in the scene will set the final value.")
                CdaPropertyEnd()
              //  CdaProperty(Partial, bool, IsPartial, SetPartial)
              //      CdaDescription("The ArabicLayouterPatch is complete or partial for this scene if enabled. Last widget in the scene will set the final value.")
              //      CdaVisibilityTest(IsEnabled)
              //  CdaPropertyEnd()
              //  CdaProperty(OnlyForRightToLeft, bool, IsOnlyForRightToLeft, SetOnlyForRightToLeft)
              //      CdaDescription("The ArabicLayouterPatch is only used for right to left layout direction. Last widget in the scene will set the final value.")
              //      CdaVisibilityTest(IsEnabled)
              //  CdaPropertyEnd()
            CdaPropertiesEnd()
        CdaWidgetDefEnd()

        FeatStd::Initialized<bool> m_valid;
        FeatStd::Initialized<bool> m_enabled;
        FeatStd::Initialized<bool> m_partial;
        FeatStd::Initialized<bool> m_onlyForRightToLeft;
    };

}

#endif
