//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_DynamicProperties_ValueNode_H)
#define CANDERA_DynamicProperties_ValueNode_H

#include <Candera/EngineBase/DynamicProperties/ValueNodeBase.h>
#include <Candera/EngineBase/DynamicProperties/TypeInfo.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>

namespace Candera {
    namespace DynamicProperties {

        /// @addtogroup DynamicPropertiesBase
        /// @{

        /** @brief
            *  Value node for the value of a property attached to a DynamicPropertyHost.
            *  @param ValueType The type of the value contained in this node.
            */
        template<typename ValueType> 
        class ValueNode : public ValueNodeBase {
            public:

                /**
                    *  Retrieves the value stored in this node.
                    *  @return The value stored in this node.
                    */
                typename TypeInfo<ValueType>::ConstRefType Value() const;

                /**
                    *  Sets the value of this node.
                    *  @param value The new value of this node.
                    */
                void SetValue(typename TypeInfo<ValueType>::ConstRefType value);

                /**
                    *  Creates and initializes a new ValueNode.
                    *  @param owningProperty The property associated with value.
                    *  @param value The value to store in the new ValueNode.
                    *  @return A new ValueNode with its value set, if successful, null if no memory could be allocated.
                    */
                static ValueNode* New(DynamicPropertyBase* owningProperty, typename TypeInfo<ValueType>::ConstRefType value);

                /**
                    *  Frees the memory for a given ValueNode.
                    *  @param val The ValueNode, whose memory should be freed.
                    */
                static void Free(ValueNode* val);

            private:
                ValueType m_value;

                /**
                    *  Construction.
                    * @param owningProperty The property associated with this ValueNode.
                    * @param value The value to store.
                    */
                ValueNode(DynamicPropertyBase* owningProperty, const ValueType& value);
        };

        template<typename ValueType>
        inline typename TypeInfo<ValueType>::ConstRefType ValueNode<ValueType>::Value() const
        {
            return m_value;
        }

        template<typename ValueType>
        inline void ValueNode<ValueType>::SetValue(typename TypeInfo<ValueType>::ConstRefType value)
        {
            m_value = value;
        }

        template<typename ValueType>
        ValueNode<ValueType>* ValueNode<ValueType>::New(DynamicPropertyBase* owningProperty, typename TypeInfo<ValueType>::ConstRefType value)
        {
            ValueNode* val = FEATSTD_NEW(ValueNode)(owningProperty, value);
            return val;
        }

        template<typename ValueType>
        inline void ValueNode<ValueType>::Free(ValueNode* val)
        {
            FEATSTD_DELETE(val);
        }

        template<typename ValueType>
        ValueNode<ValueType>::ValueNode(DynamicPropertyBase* owningProperty, const ValueType& value) :
            ValueNodeBase(owningProperty), 
            m_value(value)
        {
        }

        /// @}
    }
}
#endif    // !defined(CANDERA_DynamicProperties_ValueNode_H)
