//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_DynamicProperties_DynamicPropertySpecification_H)
#define CANDERA_DynamicProperties_DynamicPropertySpecification_H

#include <Candera/Environment.h>
#include <Candera/EngineBase/DynamicProperties/DynamicProperty.h>

namespace Candera {
    namespace DynamicProperties {

        class DynamicPropertyHost;

        /**
            * @brief Specifies a dynamic property host.
            */
        template<typename HostType, typename HostBaseClassType> class HostSpecification {
            public:
                // The type (class) of the dynamic property host.
                typedef HostType Host;

                // The base type (class) of the dynamic property host.
                typedef HostBaseClassType BaseClass;
        };

        /**
            * @brief Specifies a dynamic property.
            */
        template<typename HostSpec, typename ValType, typename ConvType> 
        class DynamicPropertySpecification {
            public:
                typedef typename HostSpec::Host HostType;
                typedef typename HostSpec::BaseClass HostBaseClassType;
                typedef ValType ValueType;
                typedef DynamicPropertySpecification<HostSpec, ValType, ConvType> ThisType;
                typedef DynamicProperty<ValType, ThisType> DynamicPropertyType;
                typedef typename ValueChangedArgs<ValType>::Signature ValueChangedSignature;
                typedef typename CoerceValueArgs<ValType>::Signature CoerceValueSignature;
                typedef typename ValidateValue<ValType>::Signature ValidateValueSignature;
                typedef typename TypeInfo<ValType>::ConstRefType ValueConstRefType;
                typedef typename TypeInfo<ValType>::RefType ValueRefType;
                typedef ConvType ConvertorType;

                virtual ~DynamicPropertySpecification();

                /**
                    *  Retrieves the flags associated with this dynamic property spec.
                    *  @return The flags associated with this dynamic property spec.
                    */
                static UInt16 GetFlags();

                /**
                    *  Retrieves a function pointer to the value changed callback function.
                    *  This function will be called every time, the value of a property is changed.
                    *  @return The function pointer to the value changed callback function.
                    */
                static ValueChangedSignature ValueChangedCallback();

                /**
                    *  Retrieves a function pointer to the coerce value callback function.
                    *  This function will be called every time, a value of a property is changed, that is
                    *  not equal to the default value.
                    *  @return The function pointer to the coerce value callback function.
                    */
                static CoerceValueSignature CoerceValueCallback();

                /**
                    *  Retrieves a function pointer to the validate value callback function.
                    *  This function will be called every time, a value of a property is changed, after the
                    *  coerce value callback function and before the value changed callback function.
                    *  @return The function pointer to the validate value callback function.
                    */
                static ValidateValueSignature ValidateValueCallback();

                /**
                    *  Retrieves the default value of the property represented by this specification.
                    *  @return The default value of the property represented by this specification.
                    */
                static ValueType GetDefaultValue();

                /**
                    *  Retrieves a description of this property. This description will
                    *  be used as meta-information e.g. in Scene Composer.
                    *  @return The description of this property.
                    */
                static const Char* GetDescription();

                /**
                    *  Retrieves the category of this property. The category will be used
                    *  as meta-information e.g. in Scene Composer to group properties in
                    *  categories.
                    *  @return The category of this property.
                    */
                static const Char* GetCategory();

                /**
                    *  Add a property to the corresponding DynamicPropertyRegistry.
                    *  @param property The property added to the registry.
                    */
                static void Register(DynamicPropertyBase* property);

            private:
                friend class DynamicPropertyHost;
        };

        template<typename HostSpec, typename ValType, typename ConvType>
        inline UInt16 DynamicPropertySpecification<HostSpec, ValType, ConvType>::GetFlags()
        {
            return LowFreqAccess;
        }

        template<typename HostSpec, typename ValType, typename ConvType>
        inline typename DynamicPropertySpecification<HostSpec, ValType, ConvType>::ValueChangedSignature DynamicPropertySpecification<HostSpec, ValType, ConvType>::ValueChangedCallback()
        {
            return &ValueChangedArgs<ValType>::DefaultCallback;
        }

        template<typename HostSpec, typename ValType, typename ConvType>
        inline typename DynamicPropertySpecification<HostSpec, ValType, ConvType>::CoerceValueSignature DynamicPropertySpecification<HostSpec, ValType, ConvType>::CoerceValueCallback()
        {
            return &CoerceValueArgs<ValType>::DefaultCallback;
        }

        template<typename HostSpec, typename ValType, typename ConvType>
        inline typename DynamicPropertySpecification<HostSpec, ValType, ConvType>::ValidateValueSignature DynamicPropertySpecification<HostSpec, ValType, ConvType>::ValidateValueCallback()
        {
            return &ValidateValue<ValType>::DefaultCallback;
        }

        template<typename HostSpec, typename ValType, typename ConvType>
        inline ValType DynamicPropertySpecification<HostSpec, ValType, ConvType>::GetDefaultValue()
        {
            return TypeInfo<ValueType>::GetDefaultValue();
        }

        template<typename HostSpec, typename ValType, typename ConvType>
        inline const Char* DynamicPropertySpecification<HostSpec, ValType, ConvType>::GetDescription()
        {
            return "";
        }
        
        template<typename HostSpec, typename ValType, typename ConvType>
        inline const Char* DynamicPropertySpecification<HostSpec, ValType, ConvType>::GetCategory()
        {
            return "Other";
        }

        template<typename HostSpec, typename ValType, typename ConvType>
        inline void DynamicPropertySpecification<HostSpec, ValType, ConvType>::Register(DynamicPropertyBase* property)
        {
            DynamicPropertyRegistry<HostType>::GetPropertyHierarchy().Register(property);
        }

        /**
            This class represents one dynamic property.
            */
        template<typename SpecType> struct Instance {
            typedef typename SpecType::ValueType ValueType;
            typedef DynamicProperty<ValueType, SpecType> PropertyType;

            /**
                *  Retrieves the singleton instance of this property.
                *  @return The singleton instance of this property.
                */
            static PropertyType& Property()
            {
                return prop;
            }

            static PropertyType prop;
        };

        template<typename SpecType> typename Instance<SpecType>::PropertyType Instance<SpecType>::prop;
    }
}
#endif    // CANDERA_DynamicProperties_DynamicPropertySpecification_H
