//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_DynamicProperties_PropertyRegistryBase_H)
#define CANDERA_DynamicProperties_PropertyRegistryBase_H

#include <Candera/Environment.h>

namespace Candera { namespace DynamicProperties {
class DynamicPropertyBase;
class DynamicPropertyRegistryBase;

/**
    * @brief A property hierarchy node represents one property for a specific type. All
    *        properties for that type are chained in a single linked list.
    */
class PropertyHierarchyNode {
    public:
        /**
            *  Initialization.
            */
        PropertyHierarchyNode();

        /**
            *  Retrieves the list of properties associated with this property hierarchy node.
            *  @return The list of properties associated with this property hierarchy node.
            */
        inline DynamicPropertyBase* GetProperties() const {
            return m_properties;
        }

        /**
            *  Registers a dynamic property with this property hierarchy node.
            *  @param prop The property to register.
            */
        void Register(DynamicPropertyBase *prop);

        /**
            *  Retrieves the parent property hierarchy node.
            *  @return The parent property hierarchy node.
            */
        inline PropertyHierarchyNode* GetParent() const {
            return m_parent;
        };

        /**
            *  Sets the parent property hierarchy node.
            *  @param parent The parent property hierarchy node.
            */
        inline void SetParent(PropertyHierarchyNode* parent) {
            m_parent = parent;
        }

        /**
            *  Retrieves the name of the property host type.
            *  @return The name of the property host type.
            */
        const Char* GetHostTypeName() const;

    private:
        friend class DynamicPropertyRegistryBase;

        PropertyHierarchyNode *m_parent;
        DynamicPropertyBase *m_properties;
        PropertyHierarchyNode* m_nextRegistered;

        PropertyHierarchyNode* GetNextRegistered() { return m_nextRegistered; }
};

class DynamicPropertyRegistryBase {
public:

    /**
        *  Retrieves a dynamic property instance for a given FULLY QUALIFIED type name
        *  and property name.
        *  @param hostTypeName The fully qualified type name of the dynamic property host.
        *  @param propertyName The name of the sought property.
        *  @return The dynamic property instance, if found, 0 otherwise.
        */
    static DynamicPropertyBase* FindProperty(const Char *hostTypeName, const Char *propertyName);

private:
    static PropertyHierarchyNode *m_registeredHierarchyNodes;
    static PropertyHierarchyNode *m_lastFoundHierarchyNode;

    static void SetRegisteredHierarchyNodes(PropertyHierarchyNode *newHead) { m_registeredHierarchyNodes = newHead; }
    static PropertyHierarchyNode *GetRegisteredHierarchyNodes() { return m_registeredHierarchyNodes; }

    /**
        *  Retrieves a hierarchy node for a given FULLY QUALIFIED name.
        *  NOTE: This function is only to be used by function FindProperty.
        *        Only hierarchy nodes whose host types have at
        *        least one dynamic property declared are found.
        *  @param hostTypeName the fully qualified host type name.
        *  @return The corresponding hierarchy node, if found, 0 otherwise.
        */
    static PropertyHierarchyNode *FindHierarchyNode(const Char* hostTypeName);

    friend class PropertyHierarchyNode;
};
}}
#endif    // !defined(CANDERA_DynamicProperties_PropertyRegistryBase_H)
