//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "DynamicPropertyHost.h"

namespace Candera { namespace DynamicProperties {
DynamicPropertyHost::DynamicPropertyHost() :
    m_values(0)
{
}

DynamicPropertyHost::DynamicPropertyHost(const DynamicPropertyHost &obj) :
    m_values(0)
{
    CloneValues(obj);
}

DynamicPropertyHost::~DynamicPropertyHost()
{
    // m_values is set to zero in FreeValues()
    CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1579, Candera::DynamicProperties::DynamicPropertyHost::m_values, CANDERA_LINT_REASON_DESTRUCTORMISINTERPRETATION)

    FreeValues();
}

DynamicPropertyHost& DynamicPropertyHost::operator=(const DynamicPropertyHost &obj)
{
    if (this == &obj) {
        return *this;
    }

    FreeValues();
    CloneValues(obj);

    return *this;
}

void DynamicPropertyHost::FreeValues()
{
    while (m_values != 0) {
        ValueNodeBase *val = m_values->Next();
        if (m_values->OwningProperty() != 0) {
            m_values->OwningProperty()->Free(m_values);
        }
        m_values = val;
    }
    m_values = 0;
}

void DynamicPropertyHost::CloneValues(const DynamicPropertyHost &srcObj)
{
    ValueNodeBase *val = srcObj.m_values;
    ValueNodeBase *prev = 0;
    m_values = 0;
    while (val != 0) {
        if (val->OwningProperty() != 0) {
            ValueNodeBase *clone = val->OwningProperty()->Clone(val);
            if (clone != 0) {
                if (0 == prev) {
                    m_values = clone;
                }
                else {
                    prev->SetNext(clone);
                    clone->SetNext(0);
                }
                prev = clone;
            }
        }
        val = val->Next();
    }
}

ValueNodeBase* DynamicPropertyHost::FindValue(const DynamicPropertyBase &prop, ParentProviderFn ParentProvider, bool& searchedInAncestor) const
{
    ValueNodeBase *val = 0;
    const DynamicPropertyHost *host = this;
    searchedInAncestor = false;

    while (host != 0) {
        ValueNodeBase *prev;
        val = prop.FindValue(host->m_values, prev);

        if ((val != 0) || (!prop.IsValuePropagated())) {
            break;
        }
        host = ParentProvider(host);
        searchedInAncestor = true;
    }
    return val;
}

bool DynamicPropertyHost::IsValueSet(const DynamicPropertyBase& prop) const
{
    ValueNodeBase* prev;
    ValueNodeBase* val = prop.FindValue(m_values, prev);
    return val != 0;
}

void DynamicPropertyHost::AddValue(ValueNodeBase *newVal)
{
    UInt16 prio = newVal->OwningProperty()->GetAccessPriority();
    ValueNodeBase *val = m_values;
    ValueNodeBase *prev = 0;
    while ((val != 0) && ((val->OwningProperty() == 0) || (prio < val->OwningProperty()->GetAccessPriority()))) {
        prev = val;
        val = val->Next();
    }
    if (prev == 0) {
        newVal->SetNext(m_values);
        m_values = newVal;
    }
    else {
        newVal->SetNext(val);
        prev->SetNext(newVal);
    }
}

const PropertyHierarchyNode& DynamicPropertyHost::GetPropertyHierarchy() const
{
    return DynamicPropertyRegistry<DynamicPropertyHost>::GetPropertyHierarchy();
}
}}
