//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_DynamicProperties_DynamicPropertyBase_H)
    #define CANDERA_DynamicProperties_DynamicPropertyBase_H

#include <Candera/Environment.h>
#include <Candera/EngineBase/DynamicProperties/ValueNodeBase.h>

namespace Candera { namespace DynamicProperties {
class DynamicPropertyHost;

/**
 *  Flags specifying usage of a dynamic property such as access frequency, inheritance, etc.
 */
enum PropertyFlags {
    LowFreqAccess = static_cast<UInt8>(1U << 0U),           ///< Low access priority
    MidFreqAccess = static_cast<UInt8>(1U << 1U),           ///< Medium access priority
    HighFreqAccess = static_cast<UInt8>(1U << 2U),          ///< High access priority

    PropagatesValue = static_cast<UInt8>(1U << 3U),         ///< Set if the value is inherited by children (propagated down the scene graph)
    AffectsMeasure = static_cast<UInt8>(1U << 4U),          ///< Set if the property affects measuring for layout.
    AffectsRender = static_cast<UInt8>(1U << 5U),           ///< Set if the property affects rendering.
    BrowsableForDescendants = static_cast<UInt8>(1U << 6U), ///< Set if this property is browsable for descendants.
    BrowsableForNodeAndDescendants = static_cast<UInt8>(1U << 7U), // Set if this property is browsable for both node and descendants

    BrowseableForDescendants = BrowsableForDescendants  ///< Candera Deprecated 3.0.4, will be removed in future version. Use BrowsableForDescendants instead.
};

/**
 *  The type of dynamic property, either generic or bool type.
 */
enum PropertyType {
    GenericType, ///< Generic dynamic property
    BoolType     ///< Boolean dynamic property
};

/**
    * @brief Base class for all dynamic properties.
    */
class DynamicPropertyBase {
    public:
        /**
            *  Initialization.
            *  @param flags The flags further specifying usage (access frequency, inheritance, ...).
            *               See PropertyFlags.
            *  @param type The property type (can be boolean or generic).
            */
        DynamicPropertyBase(UInt16 flags, PropertyType type)
            : m_flags(flags), m_type(type), m_next(0)
        {
        }

        /**
            *  Determines, whether values are inherited (propagated) down the scene graph.
            *  @return true, if inherited, false otherwise.
            */
        bool IsValuePropagated() const {
            return (m_flags & UInt16(PropagatesValue)) != 0;
        }

        /**
            *  Determines, whether values are browsable for descendants or not.
            *  @return true, if browsable, false otherwise.
            */
        bool IsBrowsableForDescendants() const {
            return (m_flags & UInt16(BrowsableForDescendants)) != 0;
        }

        /**
            *  Determines, whether values are browsable for node and descendants or not.
            *  @return true, if browsable, false otherwise.
            */
        bool IsBrowsableForNodeAndDescendants() const {
            return (m_flags & UInt16(BrowsableForNodeAndDescendants)) != 0;
        }

        /**
            *  Determines the access priority of this property. According to the set
            *  access priority, property values are ordered so that they can be fetched
            *  quicker if the priority is higher.
            *  @return The access priority of this property.
            */
        UInt16 GetAccessPriority() const {
            return m_flags & (static_cast<UInt16>(HighFreqAccess) | static_cast<UInt16>(MidFreqAccess) | static_cast<UInt16>(LowFreqAccess));
        }

        /**
            *  Determines the flags set for this dynamic property.
            *  Flags set determine:
            *    + Access priority
            *    + Value inheritance (propagation down the scene graph)
            *    + If the property affects rendering
            *    + If the property affects measuring for layout.
            *   @return The flags set for this dynamic property.
            */
        UInt16 GetFlags() const {
            return m_flags;
        }

        /**
            *  Determines the next dynamic property in the list of properties of a given type.
            *  @return The next dynamic property in the list of properties of a given type.
            */
        DynamicPropertyBase* GetNext() const {
            return m_next;
        }

        /**
            *  Sets the next dynamic property in the list of properties of a given type.
            *  @param next    The next dynamic property in the list of properties of a given type.
            */
        void SetNext(DynamicPropertyBase *next) {
            m_next = next;
        }

        /**
            *  Determines the type of this dynamic property. Valid types are boolean or generic.
            *  All types other than boolean properties are generic properties.
            *  @return The type of this dynamic property
            */
        PropertyType GetType() const {
            return m_type;
        }

        /**
            *  Retrieves the name of this dynamic property.
            *  @return The name of this dynamic property.
            */
        virtual const Char* GetPropertyName() const = 0;

        /**
            *  Retrieves the name of the host type where this dynamic property was defined.
            *  @return The name of the host type where this dynamic property was defined.
            */
        virtual const Char* GetHostTypeName() const = 0;

        /**
            *  Conversion function for converting a value of this property. To be used for de-serialization
            *  in conjunction with e.g. Scene Composer.
            *  @param host The host where the property value should be set.
            *  @param valString The string representing the value.
            *  @return true, if the conversion was successful, false otherwise.
            */
        virtual bool ConvertFromString(DynamicPropertyHost &host, const Char *valString) = 0;

        /**
            *  Conversion function for converting a value of this property. To be used for serialization
            *  in conjunction with e.g. Scene Composer.
            *  @param host The host from where the property value should be retrieved.
            *  @param valString The string to hold the value.
            *  @param nChars The number of characters available in valString to hold the value.
            *  @return true, if the conversion was successful, false otherwise.
            */
        virtual bool ToString(const DynamicPropertyHost &host, Char *valString, UInt32 nChars) const = 0;

        /**
            *  Clears the value of this dynamic property on a given property host.
            *  @param host The host of which the dynamic property shall be cleared.
            */
        virtual void Clear(DynamicPropertyHost& host) = 0;

        /**
            *  Determines, if a value of this dynamic property was set on a given
            *  property host.
            *  @param host The dynamic property host to check for a set value.
            *  @return True if a value of this dynamic property was set on a given
            *  property host, False otherwise.
            */
        virtual bool IsSet(DynamicPropertyHost& host) const = 0;

#if defined(CANDERA_META_DESCRIPTION)
        /**
            *  Retrieves the editor to use for this dynamic property. To be used for editing in e.g.
            *  Scene Composer.
            *  @return The editor to use for this dynamic property.
            */
        virtual const Char* GetEditor() const = 0;

        /**
            *  The description of this property. To be used as meta-information, e.g. as
            *  tooltip in Scene Composer.
            *  @return The description of this property.
            */
        virtual const Char* GetDescription() const = 0;

        /**
            *  The category of this property. To be used as meta-information, e.g. in conjunction with Scene Composer.
            *  Categories can be used to group properties.
            *  @return The category of this property.
            */
        virtual const Char* GetCategory() const = 0;
#endif    // CANDERA_META_DESCRIPTION

    protected:
        typedef ValueNodeBase* ValueNodeBasePtr;

        /**
            *  Clones a value node for this property.
            *  @param val the value to clone.
            *  @return The clone.
            */
        virtual ValueNodeBase* Clone(const ValueNodeBase *val) = 0;

        /**
            *  Frees the memory associated with a value node.
            *  Note: Currently, the memory is not freed but returned to the allocator.
            *  @param val The value node to free.
            */
        virtual void Free(ValueNodeBase *val) const = 0;

        /**
            *  Searches for a value of this property.
            *  @param valueNodeListHead The head of the value node list to search.
            *  @param prev Passes out a pointer to the previous value node.
            *  @return A pointer to the value of this property, if found, 0 otherwise.
            */
        virtual ValueNodeBase* FindValue(ValueNodeBase *valueNodeListHead, ValueNodeBasePtr &prev) const = 0;

    private:
        UInt16 m_flags;
        PropertyType m_type;
        DynamicPropertyBase *m_next;

        friend class DynamicPropertyHost;
};
}}
#endif    // !defined(CANDERA_DynamicProperties_DynamicProperty_H)
