//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_DynamicProperties_DynamicProperty_H)
#define CANDERA_DynamicProperties_DynamicProperty_H

#include <Candera/EngineBase/DynamicProperties/DynamicPropertyBase.h>
#include <Candera/EngineBase/DynamicProperties/TypeInfo.h>
#include <Candera/EngineBase/DynamicProperties/ValueNode.h>
#include <Candera/EngineBase/DynamicProperties/DynamicPropertyRegistry.h>
#include <Candera/EngineBase/DynamicProperties/DynamicPropertyHost.h>
#include <Candera/System/MetaInfo/DataType.h>

namespace Candera {
    namespace DynamicProperties {
        /// @addtogroup DynamicPropertiesBase
        /// @{
        class ValueNodeBase;

        /**
            *  @brief Generic class for all properties whose values are not of boolean type.
            *  @param ValueType The type of the property values.
            *  @param SpecType The property specification type.
            */
        template<typename ValueType, typename SpecType> 
        class DynamicProperty : public DynamicPropertyBase {
            public:
                typedef SpecType Specification;
                typedef ValueNode<ValueType> Value;

                /**
                    *  Initialization.
                    *  Constructs a dynamic property instance according to its specification.
                    */
                DynamicProperty();

                /**
                    *  Retrieves the default value of this property.
                    *  @return The default value of this property.
                    */
                typename TypeInfo<ValueType>::ConstRefType GetDefaultValue() const;

                /**
                    *  Retrieves the name of this property.
                    *  @return The name of this property.
                    */
                virtual const Char* GetPropertyName() const override;

                /**
                    *  Retrieves the name of the host type where this property was defined.
                    *  @return The name of the host type where this property was defined.
                    */
                virtual const Char* GetHostTypeName() const override;

#if defined(CANDERA_META_DESCRIPTION)
                /**
                    *  The description of this property. To be used as meta-information, e.g. as
                    *  tooltip in Scene Composer.
                    *  @return The description of this property.
                    */
                virtual const Char* GetDescription() const override;

                /**
                    *  The category of this property. To be used as meta-information, e.g. in conjunction with Scene Composer.
                    *  Categories can be used to group properties.
                    *  @return The category of this property.
                    */
                virtual const Char* GetCategory() const override;

                /**
                    *  Retrieves the editor to use by Scene Composer.
                    *  @return The editor to use by Scene Composer.
                    */
                virtual const Char* GetEditor() const override;
#endif    // defined(CANDERA_META_DESCRIPTION)

                /**
                    *  Conversion function for converting a value of this property. To be used for de-serialization
                    *  in conjunction with e.g. Scene Composer.
                    *  @param host The host where the property value should be set.
                    *  @param valString The string representing the value.
                    *  @return true, if the conversion was successful, false otherwise.
                    */
                bool ConvertFromString(DynamicPropertyHost& host, const Char* valString);

                /**
                    *  Conversion function for converting a value of this property. To be used for serialization
                    *  in conjunction with e.g. Scene Composer.
                    *  @param host The host from where the property value should be retrieved.
                    *  @param valString The string to hold the value.
                    *  @param nChars The number of characters available in valString to hold the value.
                    *  @return true, if the conversion was successful, false otherwise.
                    */
                bool ToString(const DynamicPropertyHost& host, Char* valString, UInt32 nChars) const;

                /**
                    *  Clears the value of this dynamic property on a given property host.
                    *  @param host The host of which the dynamic property shall be cleared.
                    */
                virtual void Clear(DynamicPropertyHost& host) override;

                virtual bool IsSet(DynamicPropertyHost& host) const override;

            private:
                /**
                    *  Clones a value node for this property.
                    *  @param val the value to clone.
                    *  @return The clone.
                    */
                ValueNodeBase* Clone(const ValueNodeBase* val);

                /**
                    *  Allocates memory for a new value node.
                    *  @param value The value to store in the node.
                    *  @return a new value node containing value, if successful, 0 otherwise.
                    */
                inline Value* New(typename TypeInfo<ValueType>::ConstRefType value);

                /**
                    *  Frees the memory associated with a value node.
                    *  Note: Currently, the memory is not freed but returned to the allocator.
                    *  @param val The value node to free.
                    */
                void Free(ValueNodeBase* val) const;

                /**
                    *  Searches for a value of this property.
                    *  @param cur The head of the value node list to search.
                    *  @param prev Passes out a pointer to the previous value node.
                    *  @return A pointer to the value of this property, if found, 0 otherwise.
                    */
                ValueNodeBase* FindValue(ValueNodeBase* cur, ValueNodeBasePtr& prev) const;

                friend class DynamicPropertyHost;
        };
        /// @}

        template<typename ValueType, typename SpecType>
        DynamicProperty<ValueType, SpecType>::DynamicProperty() : 
            DynamicPropertyBase(Specification::GetFlags(), 
            GenericType)
        {
            Specification::Register(this);
        }
        template<typename ValueType, typename SpecType>
        typename TypeInfo<ValueType>::ConstRefType DynamicProperty<ValueType, SpecType>::GetDefaultValue() const
        {
            return Specification::GetDefaultValue();
        }

        template<typename ValueType, typename SpecType>
        const Char* DynamicProperty<ValueType, SpecType>::GetPropertyName() const
        {
            return Specification::GetPropertyName();
        }

        template<typename ValueType, typename SpecType>
        const Char* DynamicProperty<ValueType, SpecType>::GetHostTypeName() const
        {
            return Specification::GetHostTypeName();
        }

#if defined(CANDERA_META_DESCRIPTION)

        template<typename ValueType, typename SpecType>
        const Char* DynamicProperty<ValueType, SpecType>::GetDescription() const
        {
            return Specification::GetDescription();
        }

        template<typename ValueType, typename SpecType>
        const Char* DynamicProperty<ValueType, SpecType>::GetCategory() const
        {
            return Specification::GetCategory();
        }

        template<typename ValueType, typename SpecType>
        const Char* DynamicProperty<ValueType, SpecType>::GetEditor() const
        {
            return Specification::ConvertorType::GetEditor();
        }

#endif

        template<typename ValueType, typename SpecType>
        bool DynamicProperty<ValueType, SpecType>::ConvertFromString(DynamicPropertyHost& host, const Char* valString)
        {
            ValueType value = host.GetValue(*this);
            bool ok = Specification::ConvertorType::ConvertFromString(value, valString);
            if (ok) {
                ok = host.SetValue(*this, value);
            }
            return ok;
        }

        template<typename ValueType, typename SpecType>
        bool DynamicProperty<ValueType, SpecType>::ToString(const DynamicPropertyHost& host, Char* valString, UInt32 nChars) const
        {
            return Specification::ConvertorType::ConvertToString(host.GetValue(*this), valString, nChars);
        }

        template<typename ValueType, typename SpecType>
        void DynamicProperty<ValueType, SpecType>::Clear(DynamicPropertyHost& host)
        {
            host.ClearValue(*this);
        }

        template<typename ValueType, typename SpecType>
        bool DynamicProperty<ValueType, SpecType>::IsSet(DynamicPropertyHost& host) const
        {
            return host.IsValueSet(*this);
        }

        template<typename ValueType, typename SpecType>
        inline ValueNodeBase* DynamicProperty<ValueType, SpecType>::Clone(const ValueNodeBase* val)
        {
            return Value::New(this, static_cast<const Value*>(val)->Value());
        }

        template<typename ValueType, typename SpecType>
        inline typename DynamicProperty<ValueType, SpecType>::Value* DynamicProperty<ValueType, SpecType>::New(typename TypeInfo<ValueType>::ConstRefType value)
        {
            return Value::New(this, value);
        }

        template<typename ValueType, typename SpecType>
        inline void DynamicProperty<ValueType, SpecType>::Free(ValueNodeBase* val) const
        {
            Value::Free(static_cast<Value*>(val));
        }

        template<typename ValueType, typename SpecType>
        ValueNodeBase* DynamicProperty<ValueType, SpecType>::FindValue(ValueNodeBase* cur, ValueNodeBasePtr& prev) const
        {
            prev = 0;
            while ((cur != 0) && (cur->OwningProperty() != this) && ((cur->OwningProperty() == 0) || (GetAccessPriority() <= cur->OwningProperty()->GetAccessPriority()))) {
                prev = cur;
                cur = cur->Next();
            }
            return (cur != 0 && cur->OwningProperty() == this) ? cur : 0;
        }

        namespace Internal {
            template<typename T> struct EmptyConvertor {
                static inline bool ConvertToString(T /*val*/, Char* /*buf*/, UInt /*size*/)
                {
                    return false;
                }
                static inline bool ConvertFromString(T& /*val*/, const Char* /*buf*/)
                {
                    return false;
                }
                static inline const Char* GetEditor()
                {
                    return "";
                }
            };
        }
    }
}

#include <Candera/EngineBase/DynamicProperties/DynamicPropertySpecification.h>
/*  ----------------------------------------------------------------------------
Public macros
---------------------------------------------------------------------------- */
/// @addtogroup DynamicPropertiesBase
/// @{
/**
    *  Macro for opening the declaration of a classes dynamic properties in the header file. Use macro
    *  CdaDynamicPropertiesEnd() to close the property definition.
    *  Example:
    *
    *    CdaDynamicProperties(Candera::Camera, Candera::Node);
    *      CdaDynamicProperty("RenderBenchmark", Float);
    *      ...
    *      CdaDynamicPropertyEnd();
    *    CdaDynamicPropertiesEnd();
    *
    *  @param hostType The FULLY QUALIFIED type of the host declaring the dynamic property.
    *  @param hostBaseClass The FULLY QUALIFIED base class of the host defining the property.
    */
#define CdaDynamicProperties(hostType, hostBaseClass) \
    /*lint -save -e1511 -e578*/ \
    public: \
    virtual const Candera::DynamicProperties::PropertyHierarchyNode& GetPropertyHierarchy() const {    \
        Candera::DynamicProperties::PropertyHierarchyNode& hNode = Candera::DynamicProperties::DynamicPropertyRegistry<hostType>::GetPropertyHierarchy(); \
        if (hNode.GetParent() == 0) { \
            hNode.SetParent(&(Candera::DynamicProperties::DynamicPropertyRegistry<hostBaseClass>::GetPropertyHierarchy())); \
        } \
        return hNode;    \
    }    \
    private: \
    struct DynamicPropertiesSpec {    \
        typedef Candera::DynamicProperties::HostSpecification<hostType, hostBaseClass> HostSpec;    \
        static inline const Candera::Char* HostName() {    \
            return FEATSTD_STRINGIZE(hostType);    \
        }

/**
*  Macro for declaration that the class provides dynamic properties in die source file
*/
#define CdaDynamicPropertiesDeclaration() \
    /*lint -save -e1511 -e578*/ \
    public: \
    virtual const Candera::DynamicProperties::PropertyHierarchyNode& GetPropertyHierarchy() const

/**
*  Macro for opening the declaration of a classes dynamic properties in the source file. Use macro
*  CdaDynamicPropertiesEnd() to close the property definition.
*  Example:
*
*    CdaDynamicPropertiesDefinition(Candera::Camera, Candera::Node);
*      CdaDynamicProperty("RenderBenchmark", Float);
*      ...
*      CdaDynamicPropertyEnd();
*    CdaDynamicPropertiesEnd();
*
*  @param hostType The FULLY QUALIFIED type of the host declaring the dynamic property.
*  @param hostBaseClass The FULLY QUALIFIED base class of the host defining the property.
*/
#define CdaDynamicPropertiesDefinition(hostType, hostBaseClass) \
    /*lint -save -e1511 -e578*/ \
    static const Candera::DynamicProperties::PropertyHierarchyNode& GetPropertyHierarchy() { \
        Candera::DynamicProperties::PropertyHierarchyNode& hNode = Candera::DynamicProperties::DynamicPropertyRegistry<hostType>::GetPropertyHierarchy(); \
        if (hNode.GetParent() == 0) { \
                hNode.SetParent(&(Candera::DynamicProperties::DynamicPropertyRegistry<hostBaseClass>::GetPropertyHierarchy())); \
        } \
        return hNode; \
    } \
    struct DynamicPropertiesSpec { \
        typedef Candera::DynamicProperties::HostSpecification<hostType, hostBaseClass> HostSpec; \
        static inline const Candera::Char* HostName() { \
                return FEATSTD_STRINGIZE(hostType); \
        }

/**
 *  Macro for closing a classes dynamic properties declaration. Use macro CdaDynamicProperties(...) to open
 *  the properties definition.
 */
#define CdaDynamicPropertiesEnd()    \
    /*lint -restore*/ \
    }

/**
 *  Macro for open the definition of one dynamic property. Use macro CdaDynamicPropertyEnd() to
 *  close a dynamic property definition.
 *  Example:
 *    ...
 *    CdaDynamicProperty("RenderBenchmark", Float);
 *      CdaDynamicPropertyDefaultValue(BenchmarkDefaultValue());
 *      CdaDynamicPropertyValueChangedCb(BenchmarkChanged());
 *      CdaDynamicPropertyCoerceValueCb(CoerceBenchmark());
 *    CdaDynamicPropertyEnd();
 *    ...
 *
 *  @param name The name of the dynamic property.
 *  @param type The type of the dynamic property.
 */
#define CdaDynamicProperty(name, type)    \
    struct name##PropSpec : public Candera::DynamicProperties::DynamicPropertySpecification<HostSpec, type, Candera::MetaInfo::Internal::DataType<type> > {    \
    static inline const Candera::Char* GetPropertyName() { \
        return FEATSTD_STRINGIZE(name);    \
    }    \
    static inline const Candera::Char* GetHostTypeName() { \
        return HostName();    \
    }

/**
 *  Macro for declaring unregistered dynamic properties. Unregistered dynamic properties behave
 *  the same as regular dynamic properties except they don't show in property lists of their host,
 *  and can't be serialized.
 *
 *  @param name The name of the dynamic property.
 *  @param type The type of the dynamic property.
 */
#define CdaDynamicPropertyUnregistered(name, type)    \
    struct name##PropSpec : public Candera::DynamicProperties::DynamicPropertySpecification<HostSpec, type, Candera::DynamicProperties::Internal::EmptyConvertor<type> > {    \
    static inline const Candera::Char* GetPropertyName() { \
        return FEATSTD_STRINGIZE(name);    \
    }    \
    static inline const Candera::Char* GetHostTypeName() { \
        return HostName();    \
    }    \
    static inline void Register(Candera::DynamicProperties::DynamicPropertyBase* /*property*/) {}

/**
 *  Macro for closing one dynamic property definition. Use macro CdaDynamicProperty(...) to open
 *  a property definition.
 */
#define CdaDynamicPropertyEnd()    \
    }

/**
 *  Macro defining the default value of a dynamic property. A reference to this macro has to be enclosed
 *  between a CdaDynamicProperty(...) and a CdaDynamicPropertyEnd() macro.
 *  @param defValue The default value.
 */
#define CdaDynamicPropertyDefaultValue(defValue)    \
    static inline ValueConstRefType GetDefaultValue() {    \
        return defValue;    \
    }

/**
 *  Macro defining the flags to use with a dynamic property. A reference to this macro has to be enclosed
 *  between a CdaDynamicProperty(...) and a CdaDynamicPropertyEnd() macro.
 *  @param flags The flags to use. See Candera::DynamicProperties::PropertyFlags for available values.
 */
#define CdaDynamicPropertyFlags(flags)    \
    static inline Candera::UInt16 GetFlags() {    \
        return flags;    \
    }

/**
 *  Macro for defining the value changed callback of a dynamic property. A reference to this macro in code
 *  has to be enclosed between a CdaDynamicProperty(...) macro and a CdaDynamicPropertyEnd() macro.
 *  @param callback The address of the static callback function.
 */
#define CdaDynamicPropertyValueChangedCb(callback) \
    static inline ValueChangedSignature ValueChangedCallback() {    \
        return callback;    \
    }

/**
 *  Macro for defining the coerce value callback of a dynamic property. A reference to this macro in code
 *  has to be enclosed between a CdaDynamicProperty(...) macro and a CdaDynamicPropertyEnd() macro.
 *  @param callback The address of the static callback function.
 */
#define CdaDynamicPropertyCoerceValueCb(callback)    \
    static inline CoerceValueSignature CoerceValueCallback() {    \
        return callback;    \
    }

/**
 *  Macro for defining the validate value callback of a dynamic property. A reference to this macro in code
 *  has to be enclosed between a CdaDynamicProperty(...) macro and a CdaDynamicPropertyEnd() macro.
 *  @param callback The address of the static callback function.
 */
#define CdaDynamicPropertyValidateValueCb(callback)    \
    static inline ValidateValueSignature ValidateValueCallback() {    \
        return callback;    \
    }

/**
 *  Macro for the access to an actual dynamic property instance. A reference to this macro in code
 *  has to be enclosed between a CdaDynamicProperty(...) macro and a CdaDynamicPropertyEnd() macro. Use
 *  this macro for writing getter/setter/clear convenience functions for dynamic properties.
 *  Example:
 *
 *    const Float& GetRenderBenchmark() const {
 *      return GetValue(CdaDynamicPropertyInstance(RenderBenchmark));
 *    }
 *
 *    void SetRenderBenchmark(const Float& benchmark) {
 *      SetValue(CdaDynamicPropertyInstance(RenderBenchmark), benchmark);
 *    }
 *
 *  @param propName The name of the property to access.
 */
#define CdaDynamicPropertyInstance(propName)    \
    Candera::DynamicProperties::Instance<DynamicPropertiesSpec::propName##PropSpec>::Property()

/**
*  Macro for the access to an actual dynamic property instance from an extern class that contains
*  the property specification. A reference to this macro in code has to be enclosed between a
*  CdaDynamicProperty(...) macro and a CdaDynamicPropertyEnd() macro. Use this macro for writing
*  getter/setter/clear convenience functions for dynamic properties.
*  Example:
*
*    const Float& Node2D::GetRenderBenchmark() const {
*      return GetValue(CdaDynamicPropertyExternInstance(Node2DDynamicProperties, RenderBenchmark));
*    }
*
*    void Node2D::SetRenderBenchmark(const Float& benchmark) {
*      SetValue(CdaDynamicPropertyExternInstance(Node2DDynamicProperties, RenderBenchmark), benchmark);
*    }
*
*  @param propClass
*  @param propName The name of the property to access.
*/
#define CdaDynamicPropertyExternInstance(propClass, propName) \
    Candera::DynamicProperties::Instance<propClass::DynamicPropertiesSpec::propName##PropSpec>::Property()

#if defined(CANDERA_META_DESCRIPTION)
/**
 *  Macro for defining the description of a dynamic property. To be used as meta-information, e.g. as
 *  tooltip in Scene Composer. A reference to this macro in code has to be enclosed between a
 *  CdaDynamicProperty(...) macro and a CdaDynamicPropertyEnd() macro.
 *  @param description A description of this dynamic property.
 */
#define CdaDynamicPropertyDescription(description)    \
        static inline const Candera::Char* GetDescription() {    \
            return description;    \
        }

/**
 *  Macro for defining the category of a dynamic property. To be used as meta-information, e.g. to group
 *  dynamic properties in Scene Composer. A reference to this macro in code has to be enclosed between a
 *  CdaDynamicProperty(...) macro and a CdaDynamicPropertyEnd() macro.
 *  @param description A description of this dynamic property.
 */
#define CdaDynamicPropertyCategory(category)    \
        static inline const Candera::Char* GetCategory() {    \
            return category;    \
        }
#else    // defined(CANDERA_META_DESCRIPTION)
#define CdaDynamicPropertyDescription(description)
#define CdaDynamicPropertyCategory(category)
#endif    // CANDERA_META_DESCRIPTION
/// @}
#endif    // !defined(CANDERA_DynamicProperties_DynamicProperty_H)
