//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_CANDERAOBJECTDATATYPE_H)
#define CANDERA_CANDERAOBJECTDATATYPE_H

#include <Candera/System/Rtti/Rtti.h>
#include <Candera/System/MemoryManagement/SharedPointer.h>
#include <Candera/System/MemoryManagement/TypeTraits.h>
#include <Candera/EngineBase/Common/CanderaObject.h>
#include <Candera/EngineBase/DataTypes/CanderaObjectDataTypeBase.h>

namespace Candera {
    class AssetProvider;
    namespace MetaInfo {

        /**
         * @brief Converter to string for asset object references.
         *
         * The ConvertToString method of this structure should be used by DataTypes of asset reference types.
         * Since the string representation of an asset object reference is an item path, most of the time it is
         * impossible to recreate the original path from the object. For this reason, the ConvertToString method
         * will fail to convert to string the asset reference, but still try to return the name of the object in
         * the string.
         * @tparam Type of object to be converted.
         */
        template<typename T>
        struct CanderaObjectConverterToString {
        };

        namespace Internal {
            /**
             * @brief Internal converter to string for asset object references.
             *
             * It has two versions:
             * - one for CanderaObject reference types, in which case the name of the CanderaObject is returned.
             * - one for non CanderaObject reference types, in which case an empty string is returned.
             *
             * @tparam IsCanderaObject Flag to specify if the input pointer points to a CanderaObject object.
             */
            template<bool IsCanderaObject>
            struct CanderaObjectConverterToString;

            /**
             * @brief CanderaObjectConverterToString specialization for objects that are not CanderaObjects.
             */
            template<> struct CanderaObjectConverterToString<false>: public NonCanderaObjectBaseConverterToString {
                template<typename T> friend struct Candera::MetaInfo::CanderaObjectConverterToString;
            };

            /**
             * @brief CanderaObjectConverterToString specialization for CanderaObjects.
             */
            template<> struct CanderaObjectConverterToString<true>: public CanderaObjectBaseConverterToString {
                template<typename T> friend struct Candera::MetaInfo::CanderaObjectConverterToString;
            };
        }


/** @addtogroup DataTypesBase
 *  @{
 */

        /**
         * @brief CanderaObjectConverterToString specialization for pointers to asset objects.
         */
        template<typename T>
        struct CanderaObjectConverterToString<T*>{
            /**
             * Retrieve the object string representation.
             *
             * @param       ptr Pointer to the object.
             * @param [out] buf The string representation.
             * @param       bufLen Buffer length.
             * @return true if ptr points to 0, in which case the empty string representation is correct.
             */
            static inline bool ConvertToString(const T* ptr, Char *buf, UInt bufLen)
            {
                return Internal::CanderaObjectConverterToString<TypeTraits::Internal::IsBaseOf<CanderaObject, T>::Value>::ConvertToString(ptr, buf, bufLen);
            }
        };

        /**
         * @brief CanderaObjectConverterToString specialization for shared pointers to asset objects.
         */
        template<typename T>
        struct CanderaObjectConverterToString<MemoryManagement::SharedPointer<T> >{
            /**
             * Retrieve the object string representation.
             *
             * @param       ptr Shared pointer to the object.
             * @param [out] buf The string representation.
             * @param       bufLen Buffer length.
             * @return true if ptr points to 0, in which case the empty string representation is correct, false otherwise.
             */
            static inline bool ConvertToString(const MemoryManagement::SharedPointer<T>& ptr, Char *buf, UInt bufLen)
            {
                return ConvertToString(ptr.GetPointerToSharedInstance(), buf, bufLen);
            }

            /**
             * Retrieve the object string representation.
             *
             * @param       ptr Pointer to the object.
             * @param [out] buf The string representation.
             * @param       bufLen Buffer length.
             * @return true if ptr points to 0, in which case the empty string representation is correct, false otherwise.
             */
            static inline bool ConvertToString(const T* ptr, Char *buf, UInt bufLen)
            {
                return Internal::CanderaObjectConverterToString<TypeTraits::Internal::IsBaseOf<CanderaObject, T>::Value>::ConvertToString(ptr, buf, bufLen);
            }
        };

        /**
         * Function used for converting from one type to another
         * within CanderaObjectInheritedDataType.
         * By default it does the conversion using Dynamic_Cast.
         * Instantiate the template to provide different casting method.
         *
         * @tparam Destination The type of the output object.
         * @tparam Source The type of the input object.
         *
         * @param source The input object.
         * @return An object resulted from casting the source object.
         */
        template <typename Destination, typename Source>
        struct CanderaObjectDataTypeCast {};

        FEATSTD_LINT_SYMBOL(1577, Candera::MetaInfo::CanderaObjectDataTypeCast<*>, "CanderaObjectDataTypeCast is specialized where it's used")
        template <typename Destination, typename Source>
        struct CanderaObjectDataTypeCast<Destination *, Source *>
        {
            static Destination* Cast(Source* source) { return Dynamic_Cast<Destination*, Source*>(source); }
        };

        template <typename Destination, typename Source>
        struct CanderaObjectDataTypeCast<FeatStd::MemoryManagement::SharedPointer<Destination>, FeatStd::MemoryManagement::SharedPointer<Source> >
        {
            static FeatStd::MemoryManagement::SharedPointer<Destination> Cast(FeatStd::MemoryManagement::SharedPointer<Source> source) { 
                return Dynamic_Cast<FeatStd::MemoryManagement::SharedPointer<Destination>, FeatStd::MemoryManagement::SharedPointer<Source> >(source);
            }
        };

        template <typename Destination, typename Source>
        struct CanderaObjectInheritedDataType {
            static inline bool ConvertFromString(Destination &destination, const Char *buf, AssetProvider* provider) {
                Source source(0);
                bool result = Candera::MetaInfo::Internal::DataType<Source>::ConvertFromString(source, buf, provider);
                if (result && (source == 0)) {
                    destination = Destination(0);
                    return true;
                }
                else if (source == 0) {
                    return false;
                }
                else {
                    destination = CanderaObjectDataTypeCast<Destination, Source>::Cast(source);
                    return (destination != 0);
                }
            }
        };

    }        
}


/**
 * DataType specialization for asset referenced objects.
 */
#define CanderaObjectInheritedDataTypeDecl(DestinationType, SourceType) \
    bool DataType<DestinationType>::ConvertFromString(DestinationType &destination, const Char *buf, AssetProvider* provider) { \
    return CanderaObjectInheritedDataType<DestinationType, SourceType>::ConvertFromString(destination, buf, provider); \
    }


/**
 * DataType specialization for asset referenced objects.
 */
#define CanderaObjectInheritedDataTypeEditorDecl(DestinationType, SourceType) \
    CanderaObjectInheritedDataTypeDecl(DestinationType, SourceType) \
    const Char* DataType<DestinationType>::GetEditor() { \
        return Candera::MetaInfo::Internal::DataType<SourceType>::GetEditor(); \
    }

#endif // CANDERA_CANDERAOBJECTDATATYPE_H
