//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "ResourceData.h"

#include <CanderaPlatform/OS/MemoryPlatform.h>
#include <Candera/EngineBase/Common/ResourceProvider.h>
#include <Candera/EngineBase/Common/ResourceDataHandle.h>

namespace Candera {
    namespace Internal {

        ResourceData::ResourceData(ResourceDataHandle& handle)
            :m_data(handle.m_accessor.m_memory.m_data),
            m_isPersistent(handle.m_isFromAsset == 0),
            m_isMutable(handle.m_isMutable)
        {
            if (m_isPersistent == 0) {
                if (ResourceProvider::Instance() != 0) {
                    bool isPersistent = false;
                    m_data = ResourceProvider::Instance()->Lock(handle, isPersistent);
                    m_isPersistent = isPersistent || (m_data == 0);
                }
            }
        }

        ResourceData::ResourceData(const ResourceDataHandle& handle)
            :m_data(handle.m_accessor.m_memory.m_data),
            m_isPersistent(handle.m_isFromAsset == 0),
            m_isMutable(false)
        {
            if (m_isPersistent == 0) {
                if (ResourceProvider::Instance() != 0) {
                    bool isPersistent = false;
                    m_data = ResourceProvider::Instance()->Lock(handle, isPersistent);
                    m_isPersistent = isPersistent || (m_data == 0);
                }
            }
        }

        ResourceData::ResourceData(const ResourceData& other):
            m_data(other.m_data),
            m_isPersistent(other.m_isPersistent),
            m_isMutable(other.m_isMutable)
        {
            if ((0 == m_isPersistent) && (0 != m_data)) {
                if (ResourceProvider::Instance() != 0) {
                    ResourceProvider::Instance()->Retain(m_data);
                }
            }
        }

        ResourceData::~ResourceData()
        {
            if ((0 == m_isPersistent) && (0 != m_data)) {
                if (ResourceProvider::Instance() != 0) {
                    ResourceProvider::Instance()->Release(m_data);
                }
            }
        }

        SizeType ResourceData::CopyData(void* buffer, const ResourceDataHandle& handle, OffsetType offset, SizeType count)
        {
            if (buffer == 0) {
                return 0;
            }

            if (handle.m_isFromAsset == 0) {
                if (handle.m_accessor.m_memory.m_data != 0) {
                    MemoryPlatform::Copy(buffer, FeatStd::Internal::PointerAdd(handle.m_accessor.m_memory.m_data, offset), count);
                    return count;
                }
            }
            else {
                return ResourceProvider::Instance()->ReadData(buffer, handle, offset, count);
            }

            return 0;
        }

        const void* ResourceData::GetAddress(const ResourceDataHandle& handle)
        {
            if (handle.m_isFromAsset == 0) {
                return handle.m_accessor.m_memory.m_data;
            }

            bool isPersistent = true;
            return ResourceProvider::Instance()->Lock(handle, isPersistent);
        }

        ResourceData& ResourceData::operator = (const ResourceData& other)
        {
            if ((0 == m_isPersistent) && (0 != m_data)) {
                if (ResourceProvider::Instance() != 0) {
                    ResourceProvider::Instance()->Release(m_data);
                }
            }

            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1555, Candera::Internal::ResourceData::m_data, It is direct copy of pointer but it points to const.);
            m_data = other.m_data;
            m_isMutable = other.m_isMutable;
            m_isPersistent = other.m_isPersistent;

            if ((0 == m_isPersistent) && (0 != m_data)) {
                if (ResourceProvider::Instance() != 0) {
                    ResourceProvider::Instance()->Retain(m_data);
                }
            }

            return *this;
        }
    }
} // namespace Candera
