//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################



#if !defined(Candera_Base_PixelBakery_h)
#define Candera_Base_PixelBakery_h

#include <Candera/EngineBase/Common/Color.h>
#include <Candera/EngineBase/Common/Bitmap.h>
#include <Candera/System/MemoryManagement/Endianness.h>
#include <Candera/System/Mathematics/Math.h>
#include <CanderaPlatform/OS/MemoryPlatform.h>

namespace Candera {

namespace Internal {

/** @addtogroup CommonBase
*  @{
*/


class PixelBakery {
public:

    /**
     *  Defines the pixel scalar base type.
     */
    template<UInt BitsPerPx, typename Endian> struct PixelBaseType {
        // select next larger base type to host a pixel
        typedef typename PixelBaseType<BitsPerPx + 1, Endian>::Type Type;
        typedef Endian EndianType;
    };

    /** 32BPP type. */
    template<typename Endian> struct PixelBaseType<32, Endian> {
        typedef UInt32 Type;
        typedef Endian EndianType;
    };

    /** 16BPP type. */
    template<typename Endian> struct PixelBaseType<16, Endian> {
        typedef UInt16 Type;
        typedef Endian EndianType;
    };

    /** 8BPP type. */
    template<typename Endian> struct PixelBaseType<8, Endian> {
        typedef UInt8 Type;
        typedef Endian EndianType;
    };


    /**
     *  FloatToBakingInteger converts Floating color channels to
     *  Integer channels used by the Bakery.
     */
    template <typename BaseType> struct FloatToBakeryInteger {
        static BaseType Convert(Float val)
        {
            return static_cast<BaseType>(
                Math::Minimum(static_cast<Float>(255), 
                    Math::Maximum(static_cast<Float>(0), 
                        val * static_cast<Float>(255))));
        }
    };

    /**
     *  ChannelDef defines the offset and width of a color channel in a pixel.
     */
    template<UInt size, UInt offset> struct ChannelDef {
        enum {
            Size = size,    ///< Size/Width
            Offset = offset ///< Offset
        };
    };

    /**
     *  Defines if a channel definition is valid or not.
     *  Invalid channels have zero width.
     */
    template<typename Channel, bool valid = (Channel::Size > 0)> struct IsValidChannel {
        static const bool value = valid;
    };

    /**
     *  @brief Defines if the color channels of a pixel are byte addressable.
     *  Byte addressable means that the offset of a channel can be
     *  divided by 8 and the width is either 0 or 8.
     */
    template<typename Channels> class IsByteAddressable {
        private:
            template<Int size> struct SizeConstraint {
                static const bool value = (size == 0 || size == 8);
            };

            template<Int offset> struct OffsetConstraint {
                static const bool value = (offset % 8) == 0;
            };

        public:
            static const bool value =
                SizeConstraint<Channels::Red::Size>::value &&
                SizeConstraint<Channels::Green::Size>::value &&
                SizeConstraint<Channels::Blue::Size>::value &&
                SizeConstraint<Channels::Alpha::Size>::value &&
                OffsetConstraint<Channels::Red::Offset>::value &&
                OffsetConstraint<Channels::Green::Offset>::value &&
                OffsetConstraint<Channels::Blue::Offset>::value &&
                OffsetConstraint<Channels::Alpha::Offset>::value;

    };

    /**
     *  @brief Base class of Pixel.
     *  Pixels are defined by their red, green, blue, and alpha channels.
     */
    template<typename Channels, typename Endian = MemoryManagement::Internal::HostEndianness> class PixelBase {
        public:
            typedef typename Channels::Red RedChannel;
            typedef typename Channels::Green GreenChannel;
            typedef typename Channels::Blue BlueChannel;
            typedef typename Channels::Alpha AlphaChannel;

            /**
             *  BitsPerPx    - number of bits per pixel.
             *  BytesPerPx   - number of bytes per pixel.
             *  HasRedFlag   - color channels valid or not.
             *  HasGreenFlag - color channels valid or not.
             *  HasBlueFlag  - color channels valid or not.
             *  HasAlphaFlag - alpha channel valid or not.
             */
            enum {
                // Number of bits per pixel.
                BitsPerPx = RedChannel::Size + GreenChannel::Size + BlueChannel::Size + AlphaChannel::Size,

                // Number of bytes per pixel.
                BytesPerPx = (BitsPerPx + 7) / 8,

                // Color channels valid or not.
                HasRedFlag = RedChannel::Size != 0,
                HasGreenFlag = GreenChannel::Size != 0,
                HasBlueFlag = BlueChannel::Size != 0,

                // Alpha channel valid or not.
                HasAlphaFlag = AlphaChannel::Size != 0
            };

            typedef Endian Endianness;

            /**
             *  Returns if the pixel has an alpha channel.
             *  @return True if the pixel has an alpha channel.
             */
            static inline bool HasAlphaChannel() { return AlphaChannel::Size != 0; }

    };

    /**
     *  Generic blender function for pixels with alpha channel.
     */
    template<typename Px, bool hasAlpha = Px::HasAlphaFlag> struct Blender {
        /**
         *  Blend the given background pixel with the given foreground pixel and alpha.
         *  @param bg    Background pixel to blend.
         *  @param fg    Foreground pixel to blend.
         *  @param alpha Alpha value to blend.
         *  @return      The blended pixel.
         */
        static typename Px::BaseType& Blend(typename Px::BaseType &bg, const typename Px::BaseType &fg, UInt8 alpha) {
            UInt8 fgAlpha = Px::GetAlpha(fg);

            alpha = UInt8((UInt32(alpha) * UInt32(fgAlpha)) / 255U);

            if (alpha == 255) {
                Px::Assign(bg, fg);
            }
            else if (alpha != 0) {
                UInt8 iAlpha = 255 - alpha;

                UInt32 addedAlpha = UInt32(alpha) + UInt32(Px::GetAlpha(bg));

                Px::Compose(bg,
                            UInt8(((UInt32(Px::GetRed(fg)) * alpha) + (UInt32(Px::GetRed(bg)) * iAlpha)) / 255),
                            UInt8(((UInt32(Px::GetGreen(fg)) * alpha) + (UInt32(Px::GetGreen(bg)) * iAlpha)) / 255),
                            UInt8(((UInt32(Px::GetBlue(fg)) * alpha) + (UInt32(Px::GetBlue(bg)) * iAlpha)) / 255),
                            addedAlpha < 255 ? UInt8(addedAlpha) : 255U);
            }

            return bg;
        }
    };

    /**
     *  Generic blender function for pixels without alpha channel.
     */
    template<typename Px> struct Blender<Px, false> {
        /**
         *  Blend the given background pixel with the given foreground pixel and alpha.
         *  @param bg    Background pixel to blend.
         *  @param fg    Foreground pixel to blend.
         *  @param alpha Alpha value to blend.
         *  @return      The blended pixel.
         */
        static typename Px::BaseType& Blend(typename Px::BaseType &bg, const typename Px::BaseType &fg, UInt8 alpha) {
            if (alpha == 255) {
                Px::Assign(bg, fg);
            }
            else if (alpha != 0) {
                UInt8 iAlpha = 255 - alpha;
                Px::Compose(bg,
                            UInt8(((UInt32(Px::GetRed(fg)) * alpha) + (UInt32(Px::GetRed(bg)) * iAlpha)) / 255),
                            UInt8(((UInt32(Px::GetGreen(fg)) * alpha) + (UInt32(Px::GetGreen(bg)) * iAlpha)) / 255),
                            UInt8(((UInt32(Px::GetBlue(fg)) * alpha) + (UInt32(Px::GetBlue(bg)) * iAlpha)) / 255),
                            0);
            }
            return bg;
        }
    };

    //TODO: endianness handling is currently not implemented. OpenGL does not comment endianness handling at all.

    /**
     *  @brief Pixel class for non byte addressable color channels (i.e. RGB565).
     */
    template<typename Channels, typename Endian = MemoryManagement::Internal::HostEndianness> class BitChannelPixel : public PixelBase<Channels, Endian> {
        typedef PixelBase<Channels, Endian> Base;
        public:
            using Base::HasAlphaFlag;

            typedef typename PixelBaseType<Base::BitsPerPx, Endian>::Type BaseType;

            typedef typename Channels::Red RedChannel;
            typedef typename Channels::Green GreenChannel;
            typedef typename Channels::Blue BlueChannel;
            typedef typename Channels::Alpha AlphaChannel;

            static inline UInt8 GetRed(const BaseType &px) { return Channel<RedChannel>::Get(px); }
            static inline void SetRed(BaseType &px, UInt8 val) { Channel<RedChannel>::Set(px, val); }
            static inline UInt8 GetGreen(const BaseType &px) { return Channel<GreenChannel>::Get(px); }
            static inline void SetGreen(BaseType &px, UInt8 val) { Channel<GreenChannel>::Set(px, val); }
            static inline UInt8 GetBlue(const BaseType &px) { return Channel<BlueChannel>::Get(px); }
            static inline void SetBlue(BaseType &px, UInt8 val) { Channel<BlueChannel>::Set(px, val); }
            static inline UInt8 GetAlpha(const BaseType &px) { return Channel<AlphaChannel>::Get(px); }
            static inline void SetAlpha(BaseType &px, UInt8 val) { Channel<AlphaChannel>::Set(px, val); }

            /**
             *  Compose a pixel from RGBA.
             *  @param px Pixel to compose values into.
             *  @param r  Red color channel.
             *  @param g  Green color channel.
             *  @param b  Blue color channel.
             *  @param a  Alpha channel.
             *  @return   The composed pixel.
             */
            static inline BaseType& Compose(BaseType &px, UInt8 r, UInt8 g, UInt8 b, UInt8 a) {
                px = Channel<RedChannel>::ShiftedValue(r) | Channel<GreenChannel>::ShiftedValue(g) |
                        Channel<BlueChannel>::ShiftedValue(b) | Channel<AlphaChannel>::ShiftedValue(a);
                return px;
            }

            /**
             *  Compose a pixel from RGBA.
             *  @param px Pixel to compose values into.
             *  @param color Color object containing RGBA values.
             *  @return   The composed pixel.
             */
            static inline BaseType& Compose(BaseType &px, const Color &color) {
                px = Channel<RedChannel>::ShiftedValue(FloatToBakeryInteger<UInt8>::Convert(color.GetRed())) |
                        Channel<GreenChannel>::ShiftedValue(FloatToBakeryInteger<UInt8>::Convert(color.GetGreen())) |
                        Channel<BlueChannel>::ShiftedValue(FloatToBakeryInteger<UInt8>::Convert(color.GetBlue())) |
                        Channel<AlphaChannel>::ShiftedValue(FloatToBakeryInteger<UInt8>::Convert(color.GetAlpha()));
                return px;
            }

            /**
             *  Blend background with foreground pixel with given alpha.
             *  @param bg    Background pixel to blend.
             *  @param fg    Foreground pixel to blend.
             *  @param alpha Alpha value to blend.
             *  @return      The blended pixel.
             */
            static inline BaseType& Blend(BaseType &bg, BaseType fg, UInt8 alpha) {
                return Blender<BitChannelPixel<Channels, Endian> >::Blend(bg, fg, alpha);
            }

            /**
             *  Pixel assignment.
             *  @param bg Background color to assign foreground color to.
             *  @param fg Foreground color to be assigned.
             */
            static inline void Assign(BaseType &bg, BaseType fg) {
                bg = fg;
            }

            /**
             *  Convert pixel to transparent UInt32 value.
             *  @param src  Pixel to be converted.
             *  @param dest UInt32 to convert pixel into.
             */
            static inline void ToUInt32(BaseType src, UInt32 &dest) {
                dest = src;
            }

            /**
             *  Convert from transparent UInt32 value to pixel
             *  @param src  UInt32 to be converted.
             *  @param dest Pixel to convert UInt32 into.
             */
            static inline void ToBaseType(UInt32 src, BaseType &dest) {
                dest = static_cast<BaseType>(src);
            }

        private:
            template<typename T, bool valid = IsValidChannel<T>::value> struct Channel {
                enum { Mask = ((1U << T::Size) - 1U) << T::Offset };

                static inline UInt8 Get(const BaseType &px) {
                    return UInt8((px & Mask) >> T::Offset);
                }

                static inline void Set(BaseType &px, UInt8 val) {
                    px = (px & ~Mask) | (BaseType(val) << T::Offset);
                }

                static inline BaseType ShiftedValue(UInt8 val) {
                    return BaseType(val) << T::Offset;
                }
            };

            template<typename T> struct Channel<T, false> {
                enum { Mask = 0 };
                static inline UInt8 Get(const BaseType&) { return 0; }
                static inline void Set(BaseType &, UInt8) { }
                static inline BaseType ShiftedValue(UInt8) { return 0; }
            };
    };

    /**
     *  @brief Pixel class color formats with byte address color channels (e.g. RGB888)..
     */
    template<typename Channels, typename Endian = MemoryManagement::Internal::HostEndianness> class ByteChannelPixel  : public PixelBase<Channels, Endian> {
        typedef PixelBase<Channels, Endian> Base;

        public:
            using Base::HasAlphaFlag;

            typedef Char BaseType[Base::BytesPerPx];

            typedef typename Channels::Red RedChannel;
            typedef typename Channels::Green GreenChannel;
            typedef typename Channels::Blue BlueChannel;
            typedef typename Channels::Alpha AlphaChannel;

            static inline UInt8 GetRed(const BaseType &px) { return Channel<RedChannel>::Get(px); }
            static inline void SetRed(BaseType &px, UInt8 val) { Channel<RedChannel>::Set(px, val); }
            static inline UInt8 GetGreen(const BaseType &px) { return Channel<GreenChannel>::Get(px); }
            static inline void SetGreen(BaseType &px, UInt8 val) { Channel<GreenChannel>::Set(px, val); }
            static inline UInt8 GetBlue(const BaseType &px) { return Channel<BlueChannel>::Get(px); }
            static inline void SetBlue(BaseType &px, UInt8 val) { Channel<BlueChannel>::Set(px, val); }
            static inline UInt8 GetAlpha(const BaseType &px) { return Channel<AlphaChannel>::Get(px); }
            static inline void SetAlpha(BaseType &px, UInt8 val) { Channel<AlphaChannel>::Set(px, val); }

            /**
             *  Compose a pixel from RGBA.
             *  @param px Pixel to compose values into.
             *  @param r  Red color channel.
             *  @param g  Green color channel.
             *  @param b  Blue color channel.
             *  @param a  Alpha channel.
             *  @return   The composed pixel.
             */
            static inline BaseType& Compose(BaseType &px, UInt8 r, UInt8 g, UInt8 b, UInt8 a) {
                Channel<RedChannel>::Set(px, r);
                Channel<GreenChannel>::Set(px, g);
                Channel<BlueChannel>::Set(px, b);
                Channel<AlphaChannel>::Set(px, a);
                return px;
            }

            /**
             *  Compose pixel from RGBA.
             *  @param px Pixel to compose values into.
             *  @param color Color object containing RGBA values.
             *  @return   The composed pixel.
             */
            static inline BaseType& Compose(BaseType &px, const Color &color) {
                Channel<RedChannel>::Set(px, FloatToBakeryInteger<UInt8>::Convert(color.GetRed()));
                Channel<GreenChannel>::Set(px, FloatToBakeryInteger<UInt8>::Convert(color.GetGreen()));
                Channel<BlueChannel>::Set(px, FloatToBakeryInteger<UInt8>::Convert(color.GetBlue()));
                Channel<AlphaChannel>::Set(px, FloatToBakeryInteger<UInt8>::Convert(color.GetAlpha()));
                return px;
            }

            /**
             *  Blend background with foreground pixel with given alpha.
             *  @param bg    Background pixel to blend.
             *  @param fg    Foreground pixel to blend.
             *  @param alpha Alpha value to blend.
             *  @return      The blended pixel.
             */
            static inline BaseType& Blend(BaseType &bg, const BaseType &fg, UInt8 alpha) {
                return Blender<ByteChannelPixel<Channels, Endian> >::Blend(bg, fg, alpha);
            }

            /**
             *  Pixel assignment.
             *  @param bg Background color to assign foreground color to.
             *  @param fg Foreground color to be assigned.
             */
            static inline void Assign(BaseType &bg, const BaseType &fg) {
                MemoryPlatform::Copy(bg, fg, sizeof(fg));
            }

            /**
             *  Convert pixel to transparent UInt32 value.
             *  @param src  Pixel to be converted.
             *  @param dest UInt32 to convert pixel into.
             */
            static inline void ToUInt32(const BaseType &src, UInt32 &dest) {
                MemoryPlatform::Copy(&dest, src, sizeof(BaseType));
            }

            /**
             *  Convert from transparent UInt32 value to pixel
             *  @param src  UInt32 to be converted.
             *  @param dest Pixel to convert UInt32 into.
             */
            static inline void ToBaseType(UInt32 src, BaseType &dest) {
                MemoryPlatform::Copy(dest, &src, sizeof(BaseType));
            }

        private:
            template<typename T, bool valid = (T::Size > 0)> struct Channel {
                enum { ArrayIndex = sizeof(BaseType) - (T::Offset / 8) - 1 };

                static inline UInt8 Get(const BaseType &px) { return px[ArrayIndex]; }
                static inline void Set(BaseType &px, UInt8 val) { px[ArrayIndex] = val; }
            };

            template<typename T> struct Channel<T, false> {
                static inline UInt8 Get(const BaseType &) { return 0; }
                static inline void Set(BaseType&, UInt8) { }
            };
    };

    /** RGB888 color format definition. */
    struct RgbChannels {
        typedef ChannelDef<8, 16> Red;
        typedef ChannelDef<8,  8> Green;
        typedef ChannelDef<8,  0> Blue;
        typedef ChannelDef<0,  0> Alpha;
    };

    /** RGBA8888 color format definition. */
    struct RgbaChannels {
        typedef ChannelDef<8, 24> Red;
        typedef ChannelDef<8, 16> Green;
        typedef ChannelDef<8,  8> Blue;
        typedef ChannelDef<8,  0> Alpha;
    };

    /** ARGB8888 color format definition. */
    struct ArgbChannels {
        typedef ChannelDef<8, 24> Alpha;
        typedef ChannelDef<8, 16> Red;
        typedef ChannelDef<8,  8> Green;
        typedef ChannelDef<8,  0> Blue;
    };

    /** ABGR8888 color format definition. */
    struct AbgrChannels {
        typedef ChannelDef<8, 24> Alpha;
        typedef ChannelDef<8, 16> Green;
        typedef ChannelDef<8,  8> Blue;
        typedef ChannelDef<8,  0> Red;
    };

    /** RGBA5551 color format definition. */
    struct Rgba5551Channels {
        typedef ChannelDef<5, 11> Red;
        typedef ChannelDef<5,  6> Green;
        typedef ChannelDef<5,  1> Blue;
        typedef ChannelDef<1,  0> Alpha;
    };

    /** RGB565 color format definition. */
    struct Rgb565Channels {
        typedef ChannelDef<5, 11> Red;
        typedef ChannelDef<6,  5> Green;
        typedef ChannelDef<5,  0> Blue;
        typedef ChannelDef<0,  0> Alpha;
    };

    /** RGBA4444 color format definition. */
    struct Rgba4444Channels {
        typedef ChannelDef<4, 12> Red;
        typedef ChannelDef<4,  8> Green;
        typedef ChannelDef<4,  4> Blue;
        typedef ChannelDef<4,  0> Alpha;
    };

    /** A8 (pure alpha mask) color format definition. */
    struct A8Channels {
        typedef ChannelDef<0,  0> Red;
        typedef ChannelDef<0,  0> Green;
        typedef ChannelDef<0,  0> Blue;
        typedef ChannelDef<8,  0> Alpha;
    };

    template<typename Channels, typename Endian = MemoryManagement::Internal::HostEndianness, bool ByteAddressable = Internal::PixelBakery::IsByteAddressable<Channels>::value> struct PixelClassSelector {
        typedef Internal::PixelBakery::ByteChannelPixel<Channels, Endian> Type;
    };

    template<typename Channels, typename Endian> struct PixelClassSelector<Channels, Endian, false> {
        typedef Internal::PixelBakery::BitChannelPixel<Channels, Endian> Type;
    };

}; // class PixelBackery
}    // namespace Internal


/** RGB888 pixel format. */
typedef Internal::PixelBakery::PixelClassSelector<Internal::PixelBakery::RgbChannels>::Type RgbPixelBakeryFormat;

/** RGBA888 pixel format. */
typedef Internal::PixelBakery::PixelClassSelector<Internal::PixelBakery::RgbaChannels>::Type RgbaPixelBakeryFormat;
/** ARGB888 pixel format. */
typedef Internal::PixelBakery::PixelClassSelector<Internal::PixelBakery::ArgbChannels>::Type ArgbPixelBakeryFormat;
/** AGBR888 pixel format. */
typedef Internal::PixelBakery::PixelClassSelector<Internal::PixelBakery::AbgrChannels>::Type AgbrPixelBakeryFormat;

/** RGBA5551 pixel format. */
typedef Internal::PixelBakery::PixelClassSelector<Internal::PixelBakery::Rgba5551Channels>::Type Rgba5551PixelBakeryFormat;
/** RGB565 pixel format. */
typedef Internal::PixelBakery::PixelClassSelector<Internal::PixelBakery::Rgb565Channels>::Type Rgb565PixelBakeryFormat;

/** RGBA4444 pixel format. */
typedef Internal::PixelBakery::PixelClassSelector<Internal::PixelBakery::Rgba4444Channels>::Type Rgba4444PixelBakeryFormat;

/** A8 (alpha mask) pixel format. */
typedef Internal::PixelBakery::PixelClassSelector<Internal::PixelBakery::A8Channels>::Type A8PixelBakeryFormat;


// Bitmap::PixelFormat to pixel format mappers.
template<Bitmap::PixelFormat format> struct BitmapToPixelBakeryFormatMapper {
};

template<> struct BitmapToPixelBakeryFormatMapper<Bitmap::RgbaUnsignedBytePixelFormat> {
    typedef RgbaPixelBakeryFormat PixelFormat;
};

template<> struct BitmapToPixelBakeryFormatMapper<Bitmap::RgbUnsignedBytePixelFormat> {
    typedef RgbPixelBakeryFormat PixelFormat;
};


template<> struct BitmapToPixelBakeryFormatMapper<Bitmap::RgbUnsignedShort565PixelFormat> {
    typedef Rgb565PixelBakeryFormat PixelFormat;
};

template<> struct BitmapToPixelBakeryFormatMapper<Bitmap::RgbaUnsignedShort4444PixelFormat> {
    typedef Rgba4444PixelBakeryFormat PixelFormat;
};


template<> struct BitmapToPixelBakeryFormatMapper<Bitmap::RgbaUnsignedShort5551PixelFormat> {
    typedef Rgba5551PixelBakeryFormat PixelFormat;
};


template<> struct BitmapToPixelBakeryFormatMapper<Bitmap::AlphaUnsignedBytePixelFormat> {
    typedef A8PixelBakeryFormat PixelFormat;
};

// Pixel format to Bitmap::Format / Bitmap::Type mappers.
template<typename PixelFormat> struct PixelBakeryFormatToBitmapMapper {
};

template<> struct PixelBakeryFormatToBitmapMapper<RgbPixelBakeryFormat> {
    enum {
        Format = Bitmap::RgbUnsignedBytePixelFormat
    };
};

template<> struct PixelBakeryFormatToBitmapMapper<RgbaPixelBakeryFormat> {
    enum {
        Format = Bitmap::RgbaUnsignedBytePixelFormat
    };
};

template<> struct PixelBakeryFormatToBitmapMapper<Rgba5551PixelBakeryFormat> {
    enum {
        Format = Bitmap::RgbaUnsignedShort5551PixelFormat
    };
};

template<> struct PixelBakeryFormatToBitmapMapper<Rgb565PixelBakeryFormat> {
    enum {
        Format = Bitmap::RgbUnsignedShort565PixelFormat
    };
};

template<> struct PixelBakeryFormatToBitmapMapper<Rgba4444PixelBakeryFormat> {
    enum {
        Format = Bitmap::RgbaUnsignedShort4444PixelFormat
    };
};

template<> struct PixelBakeryFormatToBitmapMapper<A8PixelBakeryFormat> {
    enum {
        Format = Bitmap::AlphaUnsignedBytePixelFormat
    };
};

template<typename T> static inline Bitmap::PixelFormat MapPixelBakeryFormatToBitmapPixelFormat() {
    return static_cast<Bitmap::PixelFormat>(PixelBakeryFormatToBitmapMapper<T>::Format);
}

    /** @} */ // end of CommonBase
}    // namespace Candera


#endif //Candera_Base_PixelBakery_h
