// ########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
// ########################################################################
#ifndef CANDERA_NotifyPropertyChangedInterface_h
#define CANDERA_NotifyPropertyChangedInterface_h

#include <FeatStd/Event/EventListener.h>
#include <Candera/EngineBase/Common/NotifyPropertyChangedEvent.h>

namespace Candera {

    /** @addtogroup CommonBase
    *  @{
    */

    /**
    * NotifyPropertyChangedInterface is an interface which gives a structure of how to support 
    * sending Candera::NotifyPropertyChangedEvent as soon as a property of the implementer has changed.
    * EventListener can be registered to receive this event. 
    * This structure is not required to support this event. It is just a guideline of what is the minimum requirement:
    * An external entity (class/struct/...) can register and unregister a listener to the object. 
    * As soon as it is registered it expects to receive Candera::NotifyPropertyChangedEvent.
    * Within this interface, Candera::NotifyPropertyChangedInterface::NotifyPropertyChanged will send the event.
    */
    class NotifyPropertyChangedInterface {
    public:
        virtual ~NotifyPropertyChangedInterface() {}

        /**
        * This function is used to attach an EventListener. The listener expects to receive Candera::NotifyPropertyChangedEvent until it
        * detaches itself again.
        * @param eventListener which will receive NotifyPropertyChangedEvents.
        * @return success state of adding the eventListener
        */
        virtual bool AddNotifyPropertyChangedEventListener(FeatStd::EventListener* eventListener) = 0;

        /**
        * This function is used to remove an EventListener. The listener expected to receive Candera::NotifyPropertyChangedEvent.
        * @param eventListener which shall be removed from the notification list.
        * @return success state of removing the eventListener
        */
        virtual bool RemoveNotifyPropertyChangedEventListener(FeatStd::EventListener* eventListener) = 0;
    protected:
        /**
        * Function which creates the Candera::NotifyPropertyChangedEvent and sends it to all its listeners.
        */
        virtual void NotifyPropertyChanged() = 0;

        NotifyPropertyChangedInterface() {}

    };
    /// @}
}



#endif // CANDERA_NotifyPropertyChangedInterface_h
