//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_GLYPHATLASGLYPHCACHEACCESS_H
#define CANDERA_GLYPHATLASGLYPHCACHEACCESS_H

#include <Candera/EngineBase/Common/GlyphAtlasCommon.h>
#include <Candera/System/Diagnostics/Log.h>
#include <Candera/System/MemoryManagement/CanderaHeap.h>
#include <Candera/TextEngine/GlyphCacheAccess.h>
#include <Candera/TextEngine/Internal/CacheAccessType.h>

namespace Candera
{

FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
class GlyphAtlasGlyphCacheAccess : public TextRendering::GlyphCacheAccess
{
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaEngine3D);

public:
    /**
     *  Get the GlyphAtlasGlyphCacheAccess.
     *  @return  A reference to the GlyphAtlasGlyphCacheAccess instance.
     */
    static GlyphAtlasGlyphCacheAccess& GetInstance();

    // overrides TextRendering::GlyphCacheAccess::Create
    virtual UInt8* Create(const TextRendering::GlyphBitmap& bitmap, UInt8* cacheItem) override;

    // overrides TextRendering::GlyphCacheAccess::CheckCacheItem
    virtual bool CheckCacheItem(TextRendering::PixelBuffer buffer) override;

    // overrides TextRendering::GlyphCacheAccess::DestroyCacheItem
    virtual void DestroyCacheItem(TextRendering::PixelBuffer buffer) override;

    virtual TextRendering::Internal::CacheAccessType::Enum GetCacheAccessType() const;


protected:
    virtual ~GlyphAtlasGlyphCacheAccess() override;

    template<typename GlyphAtlasType>
    UInt8* CreateCacheItem(const TextRendering::GlyphBitmap& bitmap, UInt8* cacheItem);

    template<typename GlyphAtlasType>
    bool CheckCacheItemT(TextRendering::PixelBuffer buffer);

    FEATSTD_MAKE_CLASS_STATIC(GlyphAtlasGlyphCacheAccess);

    struct GlyphAtlasGlyphCacheItem {
        GlyphAtlasBase::EntryHandle m_atlasEntry;
    };
    

private:
    friend struct FeatStd::MemoryManagement::Internal::Destructor<GlyphAtlasGlyphCacheAccess>;
    FEATSTD_MAKE_CLASS_UNCOPYABLE(GlyphAtlasGlyphCacheAccess);
};
FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()

template<typename GlyphAtlasType>
UInt8* GlyphAtlasGlyphCacheAccess::CreateCacheItem(const TextRendering::GlyphBitmap& bitmap, UInt8* cacheItem)
{
    GlyphAtlasGlyphCacheItem* glyphAtlasGlyphCacheItem = FeatStd::Internal::PointerToPointer<GlyphAtlasGlyphCacheItem*>(cacheItem);
    if (0 == glyphAtlasGlyphCacheItem) {
        glyphAtlasGlyphCacheItem = CANDERA_NEW(GlyphAtlasGlyphCacheItem);
        if (0 == glyphAtlasGlyphCacheItem) {
            FEATSTD_LOG_ERROR("Creating a glyph atlas cache item failed.");
            return 0;
        }

        bool isBitmapValid = (bitmap.pixels != 0) && (bitmap.width != 0) && (bitmap.height != 0);
        if (isBitmapValid) {
            GlyphAtlasType& glyphAtlas = GlyphAtlasType::GetInstance();
            GlyphAtlasBase::EntryHandle entry = glyphAtlas.Get(bitmap);
            if (entry.IsNull()) {
                entry = glyphAtlas.Add(bitmap);
                if (entry.IsNull()) {
                    FEATSTD_LOG_ERROR("Adding the glyph to the GlyphAtlas failed.");
                }
            }

            glyphAtlasGlyphCacheItem->m_atlasEntry = entry;
        }
    }

    return FeatStd::Internal::PointerToPointer<UInt8*>(glyphAtlasGlyphCacheItem);
}

template<typename GlyphAtlasType>
bool GlyphAtlasGlyphCacheAccess::CheckCacheItemT(TextRendering::PixelBuffer buffer)
{
    const GlyphAtlasGlyphCacheItem* glyphAtlasGlyphCacheItem = FeatStd::Internal::PointerToPointer<const GlyphAtlasGlyphCacheItem*>(buffer);
    if (0 != glyphAtlasGlyphCacheItem) {
        if (glyphAtlasGlyphCacheItem->m_atlasEntry.IsNull()) {
            // Null handle is valid, because it represents white space.
            return true;
        }

        bool isValid = GlyphAtlasType::GetInstance().IsValid(glyphAtlasGlyphCacheItem->m_atlasEntry);
        return isValid;
    }

    return false;
}

}

#endif
