//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_GLYPHATLASCOMMONUTILS_H
#define CANDERA_GLYPHATLASCOMMONUTILS_H

#include <Candera/System/Container/Map.h>
#include <FeatStd/MemoryManagement/SharedPointer.h>
#include <FeatStd/Util/ConvenienceMacros.h>

namespace Candera { namespace Internal {

using FeatStd::MemoryManagement::SharedPointer;


template<typename BitmapImage>
class GlyphAtlasBitmapUsageTracker
{
public:
    //
    GlyphAtlasBitmapUsageTracker():
        m_crntIter(0),
        m_filterOutUnused(m_crntIter) {}

    //
    ~GlyphAtlasBitmapUsageTracker()
    {
        UnloadAll();
        m_map.Clear();
    }

    GlyphAtlasBitmapUsageTracker &operator=(const GlyphAtlasBitmapUsageTracker &other)
    {
        m_map = other.m_map;
        m_crntIter = other.m_crntIter;

        return (*this);
    }

    // Call for each bitmap received from GlyphAtlas and required in rendering
    void Add(const SizeType bitmapIndex, const SharedPointer<BitmapImage> &bitmap)
    {
        TrackedBitmaps *item = m_map.Find(bitmapIndex);
        if (item != 0)
        {
            item->usedInIter = m_crntIter;
        }
        else
        {
            // Add and upload it
            TrackedBitmaps newItem;
            newItem.bitmap = bitmap;
            newItem.usedInIter = m_crntIter;
            if (m_map.Insert(bitmapIndex, newItem))
            {
                bitmap->Upload();
            }
        }
    }

    // Call after all required bitmaps were taken/received from the GlyphAtlas
    void FinishIteration()
    {
        // Unload and remove all that were not used this iteration
        m_map.Filter(m_filterOutUnused);

        // Next iteration
        ++m_crntIter;
        m_crntIter = m_crntIter & 0x0F;
    }

    //
    void UnloadAll() { m_map.Visit(m_unloadAll); }

    //
    void UploadAll() { m_map.Visit(m_uploadAll); }

    //
    SharedPointer<BitmapImage> &Get(SizeType bitmapIndex)
    {
        static SharedPointer<BitmapImage> empty;
        FEATSTD_DEBUG_ASSERT(empty.PointsToNull());

        TrackedBitmaps *found = m_map.Find(bitmapIndex);
        if (found != 0)
        {
            // Also mark it as used, to avoid unloading
            found->usedInIter = m_crntIter;
        }
        return (found != 0) ? (found->bitmap) : empty;
    }

private:
    struct TrackedBitmaps
    {
        SharedPointer<BitmapImage> bitmap;
        UInt8 usedInIter;

        TrackedBitmaps(): usedInIter(0) {}
    };

    typedef Internal::Map<SizeType, TrackedBitmaps> GlyphAtlasMap;
    GlyphAtlasMap m_map;

    UInt8 m_crntIter; // keep track if a bitmap was used this run

    //
    class UnloadFilter : public GlyphAtlasMap::Visitor
    {
    public:
        UnloadFilter(const UInt8 &iter):
            m_iter(iter)
        {}

        virtual bool Visit(const SizeType& key, TrackedBitmaps& value)
        {
            FEATSTD_UNUSED(key);
            if (value.usedInIter != m_iter)
            {
                value.bitmap->Unload();
                return false; // Remove entry from map, also
            }

            // Bitmap was used in iteration, keep it
            return true;
        }

    private:
        const UInt8 &m_iter;
        FEATSTD_MAKE_CLASS_UNCOPYABLE(UnloadFilter);
    };
    UnloadFilter m_filterOutUnused;

    //
    class UnloadAllVisitor: public GlyphAtlasMap::Visitor
    {
    public:
        UnloadAllVisitor() {}

        virtual bool Visit(const SizeType& key, TrackedBitmaps &value)
        {
            FEATSTD_UNUSED(key);
            value.bitmap->Unload();
            return true;
        }
    private:
        FEATSTD_MAKE_CLASS_UNCOPYABLE(UnloadAllVisitor);
    };
    UnloadAllVisitor m_unloadAll;

    //
    class UploadAllVisitor: public GlyphAtlasMap::Visitor
    {
    public:
        UploadAllVisitor() {}

        virtual bool Visit(const SizeType& key, TrackedBitmaps &value)
        {
            FEATSTD_UNUSED(key);
            value.bitmap->Upload();
            return true;
        }
    private:
        FEATSTD_MAKE_CLASS_UNCOPYABLE(UploadAllVisitor);
    };
    UnloadAllVisitor m_uploadAll;
};
}}
#endif

