//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_GLYPHATLASCOMMON_H
#define CANDERA_GLYPHATLASCOMMON_H

#include <Candera/EngineBase/Common/BitmapAtlas.h>
#include <Candera/System/Container/Vector.h>
#include <Candera/System/Mathematics/Vector2.h>
#include <Candera/TextEngine/GlyphBitmap.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice.h>
#include <FeatStd/Container/HashTable.h>
#include <FeatStd/Util/GenericHandle.h>
#include <FeatStd/Util/StaticObject.h>

// #define CANDERA_GLYPHATLAS_DEBUG // Maintains a copy of the atlas in system memory for debug purposes.

namespace Candera {
/** @addtogroup CommonBase
 *  @{
 */

class BitmapAtlas;

class GlyphAtlasBase
{
public:
    struct Entry {
        Entry() : AtlasIndex(0) {}

        struct Rect {
            Vector2 TopLeft;        /*!< The top-left UV coordinate of the glyph in the bitmap */
            Vector2 BottomRight;    /*!< The bottom-right UV coordinate of the glyph in the bitmap. */
        } UvRect;

        SizeType AtlasIndex;        /*!< The index identifying the bitmap of the atlas. */
        FeatStd::Internal::GenericHandleFactory<Entry>::Handle Handle;
    };

    typedef FeatStd::Internal::GenericHandleFactory<Entry> HandleFactory;
    typedef HandleFactory::Handle EntryHandle;
};

/**
 *  @brief GlyphAtlas stores glyphs in bitmaps together with lookup information for re-use.
 *
 * @tparam BitmapImage Type of the bitmap image (usually BitmapTextureImage for 3D, or BitmapImage2D) to wrap the
 *                     Bitmap of the GlyphAtlas holding the glyphs for rendering.
 */
template<typename BitmapImage>
class GlyphAtlasCommon : public GlyphAtlasBase
{
public:
    /**
     *  Get the GlyphAtlas.
     *  @return  A reference to the GlyphAtas instance.
     */
    static GlyphAtlasCommon& GetInstance();

    /**
     *  Add a glyph to the atlas.
     *  @param glyphBitmap     The glyph bitmap to add to the atlas.
     *  @return  The Entry structure that contains information about the glyph in the atlas.
     */
    EntryHandle Add(const TextRendering::GlyphBitmap& glyphBitmap);

    /**
     *  Check if the given EntryHandle is valid.
     *  @param entryHandle  The EntryHandle to check for validity.
     *  @return  True, if the given EntryHandle is valid. False, otherwise.
     */
    bool IsValid(EntryHandle entryHandle) { return (m_entryHandleFactory.Get(entryHandle) != 0); }

    /**
     *  Get the glyph entry handle from the atlas.
     *  @param glyphBitmap      The glyph to get the glyph entry information for.
     *  @return  The Entry structure that contains information about the glyph in the atlas, or 0 if 
     *           the glyph is not part of the atlas.
     */
    EntryHandle Get(const TextRendering::GlyphBitmap& glyphBitmap) const {
        Entry* entry = m_glyphMap.Find(GlyphKey(glyphBitmap.glyphIndex, glyphBitmap.faceIdentifier, glyphBitmap.fontHeight, glyphBitmap.fontWidth));
        if (0 != entry) {
            return entry->Handle;
        }
        return EntryHandle();
    }

    /**
     *  Get the glyph entry handle from the atlas.
     *  @param glyphIndex      The index of the glyph to get the glyph entry information for.
     *  @param faceIdentifier  The font identifier of the glyph to get the glyph entry information for.
     *  @param fontHeight  The font height of the glyph to get the glyph entry information for.
     *  @return  The Entry structure that contains information about the glyph in the atlas, or 0 if
     *           the glyph is not part of the atlas.
     */
    EntryHandle Get(TextRendering::GlyphIndex glyphIndex, TextRendering::FaceIdentifier faceIdentifier, Int fontHeight) const {
        return Get(glyphIndex, faceIdentifier, fontHeight, fontHeight);
    }

    /**
    *  Get the glyph entry handle from the atlas.
    *  @param glyphIndex      The index of the glyph to get the glyph entry information for.
    *  @param faceIdentifier  The font identifier of the glyph to get the glyph entry information for.
    *  @param fontHeight  The font height of the glyph to get the glyph entry information for.
    *  @param fontWidth  The font width of the glyph to get the glyph entry information for.
    *  @return  The Entry structure that contains information about the glyph in the atlas, or 0 if
    *           the glyph is not part of the atlas.
    */
    EntryHandle Get(TextRendering::GlyphIndex glyphIndex, TextRendering::FaceIdentifier faceIdentifier, Int fontHeight, Int fontWidth) const
    {
        Entry* entry = m_glyphMap.Find(GlyphKey(glyphIndex, faceIdentifier, fontHeight, fontWidth));
        if (0 != entry) {
            return entry->Handle;
        }
        return EntryHandle();
    }

    /**
     *  Get the glyph entry information from the atlas.
     *  @param entryHandle  The handle of the entry to get.
     *  @return  A pointer to the entry that is identified by the given handle, or 0 if the handle is invalid.
     */
    const Entry* Get(EntryHandle entryHandle) const { return m_entryHandleFactory.Get(entryHandle); }

    /**
     *  Get the bitmap image for an atlas index.
     *  @param atlasIndex  The index from GlyphAtlasBase::Entry of the atlas to get the bitmap image for.
     *  @return  The bitmap image of the atlas for the given index.
     */
    SharedPointer<BitmapImage> GetBitmapImage(SizeType atlasIndex) const;

    /**
     *  Return the total number of bitmap images used by the atlas.
     *  @return  The total number of bitmap images used by the atlas.
     */
    SizeType GetBitmapImageCount() const { return m_atlasEntries.Size(); }

    /**
     *  Get the current width used for creating bitmaps in the atlas.
     *  @return  The current width used for creating bitmaps in the atlas.
     */
    UInt GetAtlasWidth() const { return m_atlasWidth; }

    /**
     *  Set the current width used for creating bitmaps in the atlas.
     *  @param width  The current width to use for creating bitmaps in the atlas.
     */
    void SetAtlasWidth(UInt width) { m_atlasWidth = width; }

    /**
     *  Get the current height used for creating bitmaps in the atlas.
     *  @return  The current height used for creating bitmaps in the atlas.
     */
    UInt GetAtlasHeight() const { return m_atlasHeight; }

    /**
     *  Set the current height used for creating bitmaps in the atlas.
     *  @param height  The current height to use for creating bitmaps in the atlas.
     */
    void SetAtlasHeight(UInt height) { m_atlasHeight = height; }

    /**
     *  Remove all glyphs associated with the given face identifier from the atlas.
     *  @param faceId  The face identifier to remove the glyphs for.
     *  @return  The number of glyphs that have been removed from the atlas.
     */
    SizeType RemoveFont(TextRendering::FaceIdentifier faceId);

    /**
     *  Clears the GlyphAtlas, discarding all information including uploaded bitmaps.
     */
    void Clear();

    /**
     *  Constructor
     */
    GlyphAtlasCommon();

private:
    typedef FeatStd::Internal::GenericHandleFactory<Entry> HandleFactory;
    HandleFactory m_entryHandleFactory;

    FEATSTD_MAKE_CLASS_UNCOPYABLE(GlyphAtlasCommon);

    friend struct FeatStd::MemoryManagement::Internal::Destructor<GlyphAtlasCommon>;
    ~GlyphAtlasCommon();

    void RemoveGlyphAtlas(SizeType atlasIndex);

    bool TryAddingGlyphToAtlas(BitmapAtlas* atlas, const UInt8* glyphData, const BitmapImage* textureImage,
        UInt width, UInt height, Entry& entry, UInt padding);

    UInt m_atlasWidth;
    UInt m_atlasHeight;

    // Note: It is not allowed to change the hint of a font at runtime for the same font. For this to work, the key
    // needs to be extended with the hint. This has been neglected deliberately to not bloat the key by covering this
    // unlikely scenario.
    struct GlyphKey {
        GlyphKey(TextRendering::GlyphIndex glyphIndex, TextRendering::FaceIdentifier faceIdentifier, Int fontHeight, Int fontWidth) :
            GlyphIndex(glyphIndex), FaceIdentifier(faceIdentifier), FontHeight(fontHeight), FontWidth(fontWidth)
        {}
        GlyphKey() : GlyphIndex(), FaceIdentifier(), FontHeight(), FontWidth() {}
        TextRendering::GlyphIndex GlyphIndex;
        TextRendering::FaceIdentifier FaceIdentifier;
        Int FontHeight;
        Int FontWidth;

        bool operator== (const GlyphKey& rhs) const {
            return ((GlyphIndex == rhs.GlyphIndex) && (FaceIdentifier == rhs.FaceIdentifier)) && (FontHeight == rhs.FontHeight) && (FontWidth == rhs.FontWidth);
        }
        bool operator!= (const GlyphKey& rhs) const {
            return ((GlyphIndex != rhs.GlyphIndex) || (FaceIdentifier != rhs.FaceIdentifier)) || (FontHeight != rhs.FontHeight) || (FontWidth != rhs.FontWidth);
        }
        operator SizeType() const  { return static_cast<SizeType>(GlyphIndex); }
    };

    struct AtlasEntry {
        AtlasEntry(BitmapAtlas* atlas, const SharedPointer<BitmapImage>& image) : Atlas(atlas), Image(image) {}
        BitmapAtlas* Atlas;
        SharedPointer<BitmapImage> Image;

    private:
        AtlasEntry();
    };

    Candera::Internal::Vector<AtlasEntry> m_atlasEntries;
    FeatStd::Internal::HashTable<GlyphKey, Entry> m_glyphMap;
    UInt m_padding;
    UInt m_clearCount;

    bool ClearGlyphInAtlas(BitmapAtlas* atlas, const BitmapImage* textureImage,
        const Entry& entry, UInt padding);
};

template<typename BitmapImage>
SharedPointer<BitmapImage> GlyphAtlasCommon< BitmapImage>::GetBitmapImage(SizeType atlasIndex) const {
    return (atlasIndex < m_atlasEntries.Size()) ? m_atlasEntries[atlasIndex].Image : (typename BitmapImage::SharedPointer(0));
}

template<typename BitmapImage>
GlyphAtlasCommon<BitmapImage>::GlyphAtlasCommon()
    :
    m_atlasWidth(0),
    m_atlasHeight(0),
    m_padding(1),
    m_clearCount(0)
{
}

template<typename BitmapImage>
GlyphAtlasCommon<BitmapImage>::~GlyphAtlasCommon()
{
    Clear();
}

template<typename BitmapImage>
bool GlyphAtlasCommon<BitmapImage>::TryAddingGlyphToAtlas(BitmapAtlas* atlas, const UInt8* glyphData,
    const BitmapImage* textureImage, UInt width, UInt height, Entry& entry, UInt padding)
{
    const bool updateVram = (0 != textureImage) && (textureImage->IsUploaded());
#ifndef CANDERA_GLYPHATLAS_DEBUG
    const UInt8* systemRam = updateVram ? 0 : glyphData;
#else
    const UInt8* systemRam = glyphData;
#endif
    UInt xPosInAtlas = 0;
    UInt yPosInAtlas = 0;

    // A BitmapImage that is uploaded in VRAM will be updated. If it is not uploaded or none has been
    // passed, update the bitmap's buffer in system RAM instead.
    if (atlas->Add(width, height, xPosInAtlas, yPosInAtlas, systemRam)) {
        entry.UvRect.TopLeft = atlas->ConvertToUvFlipped(xPosInAtlas, yPosInAtlas);
        entry.UvRect.BottomRight = atlas->ConvertToUvFlipped(xPosInAtlas + width, yPosInAtlas + height);

        if (updateVram) {
            const Bitmap::SharedPointer& bitmap = atlas->GetBitmap();
            if (bitmap == 0) {
                return false;
            }
            const UInt bitsPerPixel = Bitmap::GetBitsPerPixel(bitmap->GetPixelFormat());
            if ((bitsPerPixel % 8) != 0) {
                return false;
            }

            const SizeType bytesPerPixel = static_cast<SizeType>(bitsPerPixel / 8);
            const SizeType clearCount = ((width > height) ? width : height) * padding * bytesPerPixel;
            UInt8* clearedPadding = CANDERA_NEW_ARRAY(UInt8, clearCount);
            if (0 == clearedPadding) {
                return false;
            }

            MemoryPlatform::Set(clearedPadding, 0, clearCount);

            Int xPos = static_cast<Int>(xPosInAtlas);
            Int yPos = static_cast<Int>(yPosInAtlas);
            Int w = static_cast<Int>(width);
            Int h = static_cast<Int>(height);
            bool success = textureImage->Update(xPos + w, yPos, padding, height, clearedPadding);
            success = textureImage->Update(xPos, yPos + h, (width+padding), padding, clearedPadding) && success;
            success = textureImage->Update(xPos, yPos, width, height, glyphData) && success;
            CANDERA_DELETE_ARRAY(clearedPadding);
            return success;
        }

        return true;
    }

    return false;
}


template<typename BitmapImage>
bool GlyphAtlasCommon<BitmapImage>::ClearGlyphInAtlas(BitmapAtlas* atlas, const BitmapImage* textureImage,
    const Entry& entry, UInt padding)
{
    if (0 == atlas) {
        return false;
    }

    const bool updateVram = (0 != textureImage) && (textureImage->IsUploaded());

    UInt left = 0;
    UInt top = 0;
    atlas->ConvertFromFlippedUv(entry.UvRect.TopLeft, left, top);

    UInt right = 0;
    UInt bottom = 0;
    atlas->ConvertFromFlippedUv(entry.UvRect.BottomRight, right, bottom);

    UInt width = (right - left) + padding;
    UInt height = (bottom - top) + padding;

    const Bitmap::SharedPointer& bitmap = atlas->GetBitmap();
    if (bitmap == 0) {
        return false;
    }

    Bitmap::PixelsResource pixelsResource(bitmap->GetPixelsResourceHandle());
    UInt8* bitmapData = pixelsResource.GetMutableData();
    UInt bitsPerPixel = Bitmap::GetBitsPerPixel(bitmap->GetPixelFormat());
    if ((bitsPerPixel % 8) != 0) {
        return false;
    }

    const OffsetType bytesPerPixel = static_cast<OffsetType>(bitsPerPixel / 8);
    const SizeType clearedAreaInBytes = width * height * bytesPerPixel;
    UInt8* clearedRect = CANDERA_NEW_ARRAY(UInt8, clearedAreaInBytes);
    if (0 == clearedRect) {
        return false;
    }

    MemoryPlatform::Set(clearedRect, 0, clearedAreaInBytes);

    bool success = true;
    if (updateVram) {
        success = textureImage->Update(left, top, width, height, clearedRect) && success;
        FEATSTD_DEBUG_ASSERT(success);
    }

    if (0 != bitmapData) {
        const SizeType rectWidthInBytes = static_cast<SizeType>(width)* bytesPerPixel;
        const OffsetType bitmapWidthInBytes = static_cast<OffsetType>(bitmap->GetWidth()) * bytesPerPixel;
        const OffsetType destOffsetInBytes = (static_cast<OffsetType>(left)* bytesPerPixel) +
            (static_cast<OffsetType>(top)* bitmapWidthInBytes);
        UInt8* dstBuffer = FeatStd::Internal::PointerAdd(bitmapData, destOffsetInBytes);
        for (UInt i = 0; i < height; ++i) {
            MemoryPlatform::Set(dstBuffer, 0, rectWidthInBytes);
            dstBuffer = FeatStd::Internal::PointerAdd(dstBuffer, bitmapWidthInBytes);
        }
    }

    CANDERA_DELETE_ARRAY(clearedRect);
    return success;
}

template<typename BitmapImage>
typename GlyphAtlasCommon<BitmapImage>::EntryHandle GlyphAtlasCommon<BitmapImage>::Add(
    const TextRendering::GlyphBitmap& glyphBitmap)
{
    const UInt8* glyphData = glyphBitmap.pixels;
    if (0 == glyphData) {
        return EntryHandle();
    }

    if (glyphBitmap.format != TextRendering::GlyphBitmap::Grayscale) {
        return EntryHandle();
    }

    const TextRendering::GlyphIndex glyphIndex = glyphBitmap.glyphIndex;
    const TextRendering::FaceIdentifier faceIdentifier = glyphBitmap.faceIdentifier;
    GlyphKey key(glyphIndex, faceIdentifier, glyphBitmap.fontHeight, glyphBitmap.fontWidth);
    FEATSTD_DEBUG_ASSERT(0 == m_glyphMap.Find(key)); // The glyph for the font has already been added.
    Entry entry;

    const UInt width = FeatStd::Internal::NumericConversion<UInt>(glyphBitmap.width);
    const UInt height = FeatStd::Internal::NumericConversion<UInt>(glyphBitmap.height);

    const Bitmap::PixelFormat format = Bitmap::AlphaUnsignedBytePixelFormat;

    bool foundPlaceInAtlas = false;
    for (SizeType i = 0; i < m_atlasEntries.Size(); ++i) {
        AtlasEntry& atlasEntry = m_atlasEntries[i];
        if (0 == atlasEntry.Atlas) {
            continue;
        }

        if (atlasEntry.Atlas->GetBitmap()->GetPixelFormat() != format) {
            continue;
        }

        FEATSTD_DEBUG_ASSERT(atlasEntry.Image != 0);
        foundPlaceInAtlas = TryAddingGlyphToAtlas(atlasEntry.Atlas, glyphData,
            atlasEntry.Image.GetPointerToSharedInstance(), width, height, entry, m_padding);
        if (foundPlaceInAtlas) {
            entry.AtlasIndex = i;
            break;
        }
    }

    if (!foundPlaceInAtlas) {
        if (0 == m_atlasWidth) {
            m_atlasWidth = RenderDevice::GetMaxTextureSizeSupportedByDevice();
            if (m_atlasWidth > 2048) {
                m_atlasWidth = 2048;
            }
            FEATSTD_DEBUG_ASSERT(0 < m_atlasWidth);
        }

        if (0 == m_atlasHeight) {
            m_atlasHeight = RenderDevice::GetMaxTextureSizeSupportedByDevice();
            if (m_atlasHeight > 2048) {
                m_atlasHeight = 2048;
            }
            FEATSTD_DEBUG_ASSERT(0 < m_atlasHeight);
        }

        UInt bitmapDataSize = m_atlasWidth * m_atlasHeight * (Bitmap::GetBitsPerPixel(format) / 8);
        UInt8* bitmapData = CANDERA_NEW_ARRAY(UInt8, static_cast<SizeType>(bitmapDataSize));
        if (0 == bitmapData) {
            return EntryHandle();
        }

        MemoryPlatform::Set(bitmapData, 0, bitmapDataSize);

        UInt16 width16 = FeatStd::Internal::NumericConversion<UInt16>(m_atlasWidth);
        UInt16 height16 = FeatStd::Internal::NumericConversion<UInt16>(m_atlasHeight);
        Bitmap::SharedPointer bitmap = Bitmap::Create(width16, height16, format, Candera::Bitmap::PackAlignment1,
#ifndef CANDERA_GLYPHATLAS_DEBUG
            bitmapData, Candera::Bitmap::Disposer::Dispose, bitmapDataSize, false);
#else
            bitmapData, Candera::Bitmap::Disposer::Dispose, bitmapDataSize, true);
#endif

        BitmapAtlas* bitmapAtlas = BitmapAtlas::Create(bitmap, m_padding);
        if (0 == bitmapAtlas) {
            return EntryHandle();
        }

        typename BitmapImage::SharedPointer textureImage = BitmapImage::Create();
        AtlasEntry atlasEntry(bitmapAtlas, textureImage);
        if (!m_atlasEntries.Add(atlasEntry)) {
            return EntryHandle();
        }

        if (!textureImage->SetBitmap(bitmap)) {
            return EntryHandle();
        }

#ifndef CANDERA_GLYPHATLAS_DEBUG
        textureImage->SetDisposedAfterUpload(true);
#endif
        if (!textureImage->Upload(BitmapImage::NoHint)) {
            return EntryHandle();
        }

        foundPlaceInAtlas = TryAddingGlyphToAtlas(bitmapAtlas, glyphData, textureImage.GetPointerToSharedInstance(),
            width, height, entry, m_padding);
        if (foundPlaceInAtlas) {
            entry.AtlasIndex = m_atlasEntries.Size() - 1;
        }
    }

    Entry* entryInMap = m_glyphMap.Insert(key, entry) ? m_glyphMap.Find(key) : 0;

    EntryHandle handle = m_entryHandleFactory.Create(entryInMap);
    if (0 != entryInMap) {
        entryInMap->Handle = handle;
    }
    else {
        FEATSTD_DEBUG_FAIL();
        return EntryHandle();
    }

    return handle;
}

template<typename BitmapImage>
SizeType GlyphAtlasCommon<BitmapImage>::RemoveFont(TextRendering::FaceIdentifier faceId)
{
    SizeType removedGlyphCount = 0;
    Internal::Vector<GlyphKey> glyphKeysToRemove;
    if (!glyphKeysToRemove.Reserve(m_glyphMap.GetKeyValuePairCount())) {
        return 0;
    }

    typename FeatStd::Internal::HashTable<GlyphKey, Entry>::Iterator it(m_glyphMap);
    GlyphKey key;
    for (Entry* value = it.GetFirst(&key); (0 != value); value = it.GetNext(&key)) {
        if (key.FaceIdentifier == faceId) {
            static_cast<void>(glyphKeysToRemove.Add(key));

            FEATSTD_DEBUG_ASSERT(0 != value);

            bool wasSuccessful = m_entryHandleFactory.Remove(value->Handle);
            FEATSTD_DEBUG_ASSERT(wasSuccessful);
            FEATSTD_UNUSED(wasSuccessful);

            const Entry& entry = *value;
            FEATSTD_DEBUG_ASSERT(entry.AtlasIndex < m_atlasEntries.Size());
            BitmapAtlas* bitmapAtlas = m_atlasEntries[entry.AtlasIndex].Atlas;
            FEATSTD_DEBUG_ASSERT(0 != bitmapAtlas);

            UInt x = 0;
            UInt y = 0;
            bitmapAtlas->ConvertFromFlippedUv(entry.UvRect.TopLeft, x, y);
            bool wasRemoved = bitmapAtlas->Remove(x, y);
            FEATSTD_DEBUG_ASSERT(wasRemoved);
            FEATSTD_UNUSED(wasRemoved);
            ++removedGlyphCount;
            ClearGlyphInAtlas(bitmapAtlas, m_atlasEntries[entry.AtlasIndex].Image.GetPointerToSharedInstance(), entry, m_padding);
        }
    }

    for (SizeType i = 0; i < glyphKeysToRemove.Size(); ++i) {
        bool wasRemoved = m_glyphMap.Remove(glyphKeysToRemove[i]);
        FEATSTD_DEBUG_ASSERT(wasRemoved);
        FEATSTD_UNUSED(wasRemoved);
    }

    bool wasActiveAtlasEncountered = false;
    for (SizeType i = (m_atlasEntries.Size() - 1); i < m_atlasEntries.Size(); --i) {
        if (0 != m_atlasEntries[i].Atlas) {
            if (m_atlasEntries[i].Atlas->IsEmpty()) {
                if (!wasActiveAtlasEncountered) {
                    RemoveGlyphAtlas(i);
                }
                else {
                    m_atlasEntries[i].Atlas->Dispose();
                    m_atlasEntries[i].Image->Unload(); // Unload the texture bitmap
                    m_atlasEntries[i] = AtlasEntry(0, typename BitmapImage::SharedPointer(0));
                }
            }
            else {
                wasActiveAtlasEncountered = true;
            }
        }
        else {
            if (!wasActiveAtlasEncountered) {
                RemoveGlyphAtlas(i);
            }
        }
    }

    return removedGlyphCount;
}

template<typename BitmapImage>
void GlyphAtlasCommon<BitmapImage>::Clear()
{
    while (0 < m_atlasEntries.Size()) {
        RemoveGlyphAtlas(m_atlasEntries.Size() - 1);
    }

    typename FeatStd::Internal::HashTable<GlyphKey, Entry>::Iterator it(m_glyphMap);
    GlyphKey key;
    for (Entry* value = it.GetFirst(&key); (0 != value); value = it.GetNext(&key)) {
        FEATSTD_DEBUG_ASSERT(0 != value);
        bool wasSuccessful = m_entryHandleFactory.Remove(value->Handle);
        FEATSTD_DEBUG_ASSERT(wasSuccessful);
        FEATSTD_UNUSED(wasSuccessful);
    }

    m_glyphMap.Clear();
    ++m_clearCount;
}

template<typename BitmapImage>
void GlyphAtlasCommon<BitmapImage>::RemoveGlyphAtlas(SizeType atlasIndex)
{
    FEATSTD_DEBUG_ASSERT(atlasIndex < m_atlasEntries.Size());
    if (0 != m_atlasEntries[atlasIndex].Atlas) {
        m_atlasEntries[atlasIndex].Atlas->Dispose();
        m_atlasEntries[atlasIndex].Atlas = 0;
        m_atlasEntries[atlasIndex].Image->Unload(); // Unload the texture bitmap
    }

    bool wasRemoved = m_atlasEntries.Remove(atlasIndex);
    if (0 == m_atlasEntries.Size()) {
        m_atlasEntries.Free();
    }

    FEATSTD_DEBUG_ASSERT(wasRemoved);
    FEATSTD_UNUSED(wasRemoved);
}

}

#ifdef CANDERA_GLYPHATLAS_DEBUG
#undef CANDERA_GLYPHATLAS_DEBUG
#endif

#endif
