//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_CANDERAOBJECT_H)
#define CANDERA_CANDERAOBJECT_H

#include <Candera/EngineBase/DynamicProperties/DynamicPropertyHost.h>
#include <Candera/System/Rtti/Rtti.h>

namespace Candera {

/** @addtogroup CommonBase
 *  @{
 */

/**
 * @brief Represents the base class for Candera objects related to 3D rendering.
 *          Every CanderaObject can have a name assigned. A name has the semantic of an user ID and is typically
 *          used to find a particular object in a scene.
 */
class CanderaObject : public DynamicProperties::DynamicPropertyHost
{
    friend class Candera::DynamicProperties::DynamicPropertyHost;
    
    FEATSTD_TYPEDEF_BASE(DynamicProperties::DynamicPropertyHost);

    public:
        /**
         *  Constructor. The default value for name is set to null.
         */
        CanderaObject();

        /**
         *  CopyConstructor creates a shallow clone of given CanderaObject.
         *  @param canderaObject The given CanderaObject that is copied.
         */
        CanderaObject(const CanderaObject& canderaObject);

        /**
         *  Destructor. Memory for CanderaObject name is not freed.
         */
        virtual ~CanderaObject() override;

        /**
         *  Set name of this CanderaObject. The name has the semantic of an user id.
         *  @param name   pointer to a null-terminated character array that must not be deleted during runtime
         *                (static memory, string literals, flash memory)
         */
        void SetName(const Char* name);

        /**
         *  Get name (user-ID) of this CanderaObject. The default value of the name is null.
         *  @return Name of this CanderaObject.
         */
        const Char* GetName() const { return m_name; }

        /**
         *  Set id of this CanderaObject.
         *  @param id of the CanderaObject.
         */
        void SetId(Id id) { m_id = id; }

        /**
         *  Get id of this CanderaObject.
         *  @return Id of this CanderaObject.
         */
        Id GetId() const { return m_id; }

        /**
         *  Assignment operator creates a shallow clone of given CanderaObject.
         *  @param canderaObject The given CanderaObject that is cloned.
         *  @return The created CanderaObject.
         */
        CanderaObject& operator=(const CanderaObject& canderaObject);

        FEATSTD_RTTI_DECLARATION();

    protected:
        /**
         *  Is invoked after the name has been changed by CanderaObject::SetName().
         *  The default implementation is empty and does not need to get called.
         */
        virtual void OnNameChanged();

    private:
        // m_name is a pointer to a non-erasable null-terminated string.
        const Char* m_name;
        Id m_id;

        // Required by DynamicPropertyHost.
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1511, Candera::CanderaObject::ParentProvider, "Derived classes also provide ParentProvider functions")
        static const Candera::DynamicProperties::DynamicPropertyHost* ParentProvider(const Candera::DynamicProperties::DynamicPropertyHost* /*host*/) { return 0; }

};

 /** @} */ // end of CommonBase

} // namespace Candera

#endif    // CANDERA_CANDERAOBJECT_H
