//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_BITMAPATLAS_H
#define CANDERA_BITMAPATLAS_H

#include <Candera/EngineBase/Common/Bitmap.h>
#include <Candera/System/Mathematics/Vector2.h>

namespace Candera {
/** @addtogroup CommonBase
 *  @{
 */

/**
 * @brief  The BitmapAtlas class packs several smaller images into the bitmap of the atlas.
 *         It supports padding between packed images so that the user can mipmap the atlas if necessary.
 */

class BitmapAtlas
{
public:
    /**
     *  Creates an instance of this class.
     *  Use Dispose() to delete the instance.
     *  @param bitmap   The bitmap specify the width, height, and pixel format of the atlas.
     *  @param padding  The padding in pixels between packed images in the atlas.
     *  @return  A pointer to the created BitmapAtlas instance, or 0 if the creation failed.
     */
    static BitmapAtlas* Create(Bitmap::SharedPointer bitmap, UInt padding = 0);

    /**
     *  Dispose the bitmap atlas.
     */
    void Dispose();

    /**
     *  Add an image to the bitmap atlas.
     *  The image is specified by the provided width, height, and optionally a pointer to the image data to be
     *  copied into the atlas' bitmap data in system memory. The user is responsible for updating the
     *  BitmapTextureImage or BitmapImage2D in video memory (VRAM) that the Bitmap of the atlas might be
     *  associated with. This can be done by calling:
     *  Bitmap(Texture)Image(2D)::Update(xPositionInAtlas, yPositionInAtlas, width, height, data)
     *  This enables the user to control if the atlas in system memory and/or video memory gets updated.
     *  @param width             Width of the new image to add.
     *  @param height            Height of the new image to add.
     *  @param xPositionInAtlas  Returns the x position of the image in the atlas, if the function returns 'true'.
     *  @param yPositionInAtlas  Returns the y position of the image in the atlas, if the function returns 'true'.
     *  @param data              If the pointer is not 0, the bitmap's data in system memory will be updated with
     *                           the given data. It is assumed that the pixel format of the image is the same as
     *                           the pixel format of the atlas.
     *  @return  True, if the image was successfully added to the bitmap atlas.
     *           False, if the image does not fit in the atlas, or if updating one of the buffers failed.
     */
    bool Add(UInt width, UInt height, UInt& xPositionInAtlas /*out*/, UInt& yPositionInAtlas /*out*/, const UInt8* data = 0);

    /**
     *  Remove the image located at (xPositionInAtlas,yPositionInAtlas) from the bitmap atlas. The origin (0,0) is located top left.
     *  After successful removal the internal structure of the atlas will be cleaned up to avoid fragmentation of free atlas space.
     *  @param xPositionInAtlas  The x (i.e. left) coordinate of the image in the atlas.
     *  @param yPositionInAtlas  The y (i.e. top) coordinate of the image in the atlas.
     *  @return  True, if the image was successfully removed.
     *           False, if no image was added at the given location.
     */
    bool Remove(UInt xPositionInAtlas, UInt yPositionInAtlas);

    /**
     *  Query if the atlas has any entries.
     *  @return  True, if the atlas does not have any entries.
     *           False, if the atlas has at least one entry.
     */
    bool IsEmpty() const;

    /**
     *  Convert the given x,y integer position in the atlas to a uv coordinate.
     *  @param x  The x position to convert to u.
     *  @param y  The y position to convert to v.
     *  @return  The u,v coordinate for given x,y.
     */
    Vector2 ConvertToUv(UInt x, UInt y) const { return Vector2(static_cast<Float>(x) / static_cast<Float>(m_rootEntry.m_rect.width),
        static_cast<Float>(y) / static_cast<Float>(m_rootEntry.m_rect.height));
    }

    /**
     *  Convert the given uv coordinate to a x,y integer position.
     *  @param uv  The u,v coordinate to convert.
     *  @param x   Returns the x position.
     *  @param y   Returns the y position.
     */
    void ConvertFromUv(Vector2 uv, UInt& x, UInt& y) const {
        x = static_cast<UInt>(uv.GetX() * static_cast<Float>(m_rootEntry.m_rect.width) + 0.5F);
        y = static_cast<UInt>(uv.GetY() * static_cast<Float>(m_rootEntry.m_rect.height) + 0.5F);
    }

    /**
     *  Convert the given uv coordinate, where v is flipped according to the GL convention
     *  of having the origin (0,0) bottom left (instead of top left), to a x,y integer position.
     *  @param uv  The u,v coordinate to convert.
     *  @param x   Returns the x position.
     *  @param y   Returns the y position.
     */
    void ConvertFromFlippedUv(Vector2 uv, UInt& x, UInt& y) const {
        x = static_cast<UInt>(uv.GetX() * static_cast<Float>(m_rootEntry.m_rect.width) + 0.5F);
        y = static_cast<UInt>(m_rootEntry.m_rect.height) - static_cast<UInt>(uv.GetY() * static_cast<Float>(m_rootEntry.m_rect.height) + 0.5F);
    }

    /**
     *  Convert the given x,y integer position in the atlas to a uv coordinate, where v is flipped according to the GL convention
     *  of having the origin (0,0) bottom left (instead of top left).
     *  @param x  The x position to convert to u.
     *  @param y  The y position to convert to v.
     *  @return  The u,v coordinate for given x,y.
     */
    Vector2 ConvertToUvFlipped(UInt x, UInt y) const { return Vector2(static_cast<Float>(x) / static_cast<Float>(m_rootEntry.m_rect.width),
        static_cast<Float>(m_rootEntry.m_rect.height - y) / static_cast<Float>(m_rootEntry.m_rect.height));
    }

    /**
     *  Returns the width of the buffer that holds the bitmap atlas image.
     *  @return  The width of the buffer that holds the bitmap atlas image.
     */
    UInt GetWidth() const { return static_cast<UInt>(m_rootEntry.m_rect.width); }

    /**
     *  Returns the height of the buffer that holds the bitmap atlas image.
     *  @return  The height of the buffer that holds the bitmap atlas image.
     */
    UInt GetHeight() const { return static_cast<UInt>(m_rootEntry.m_rect.height); }

    /**
     * Returns the bitmap of the atlas.
     * @return The bitmap of the atlas.
     */
    const Bitmap::SharedPointer& GetBitmap() const { return m_bitmap; }

private:
    FEATSTD_MAKE_CLASS_STATIC(BitmapAtlas);
    FEATSTD_MAKE_CLASS_UNCOPYABLE(BitmapAtlas);

    friend struct FeatStd::MemoryManagement::Internal::Destructor<BitmapAtlas>;
    ~BitmapAtlas();

    typedef UInt32 TextureSizeType;

    struct Rect {
        Rect(TextureSizeType l, TextureSizeType t, TextureSizeType w, TextureSizeType h) : left(l), top(t), width(w), height(h) {}
        TextureSizeType left;
        TextureSizeType top;
        TextureSizeType width;
        TextureSizeType height;
    };

    // Binary tree to manage the atlas.
    struct Entry {
        Entry() : m_entry0(0), m_entry1(0), m_rect(0, 0, 0, 0), m_isOccupied(false) {}
        ~Entry() { CANDERA_DELETE(m_entry0); m_entry0 = 0; CANDERA_DELETE(m_entry1); m_entry1 = 0; }

        bool IsLeaf() const { return ((0 == m_entry0) && (0 == m_entry1)); }
        Entry* Insert(UInt width, UInt height);
        bool Remove(UInt left, UInt top, Entry* parent);
        bool IsSubtreeOccupied() const;
        void DefragSubtree();

        Entry* m_entry0;
        Entry* m_entry1;
        Rect m_rect;
        bool m_isOccupied;
    };

    Bitmap::SharedPointer m_bitmap;
    UInt m_padding;
    Entry m_rootEntry;
    UInt8 m_paddingValue;
};

/** @} */ // end of CommonBase

} // namespace Candera

#endif
