//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "BitmapAtlas.h"

namespace Candera {

BitmapAtlas::BitmapAtlas()
    :
    m_padding(0),
    m_paddingValue(0)
{
}

BitmapAtlas::~BitmapAtlas()
{
}

BitmapAtlas* BitmapAtlas::Create(Bitmap::SharedPointer bitmap, UInt padding)
{
    if (bitmap == 0) {
        return 0;
    }

    if (bitmap->IsCompressed()) {
        return 0;
    }

    BitmapAtlas* bitmapAtlas = CANDERA_NEW(BitmapAtlas);
    if (0 == bitmapAtlas) {
        return 0;
    }

    bitmapAtlas->m_bitmap = bitmap;

    const UInt width = bitmap->GetWidth();
    const UInt height = bitmap->GetHeight();

    bitmapAtlas->m_rootEntry.m_rect = Rect(0, 0, width, height);
    bitmapAtlas->m_padding = padding;
    return bitmapAtlas;
}

void BitmapAtlas::Dispose()
{
    CANDERA_DELETE(this);
}

bool BitmapAtlas::Add(UInt width, UInt height, UInt& xPositionInAtlas, UInt& yPositionInAtlas, const UInt8* data)
{
    Entry* entry = m_rootEntry.Insert(width + m_padding, height + m_padding);
    if (0 != entry) {
        entry->m_isOccupied = true;

        xPositionInAtlas = entry->m_rect.left;
        yPositionInAtlas = entry->m_rect.top;

        Bitmap* bitmap = m_bitmap.GetPointerToSharedInstance();
        if (0 != bitmap) {
            Bitmap::PixelsResource pixelsResource(bitmap->GetPixelsResourceHandle());
            UInt8* bitmapData = pixelsResource.GetMutableData();

            UInt bitsPerPixel = Bitmap::GetBitsPerPixel(bitmap->GetPixelFormat());
            if ((0 != data) && (0 != bitmapData)) {
                if ((bitsPerPixel % 8) != 0) {
                    return false;
                }

                // data and width in bytes
                const OffsetType bytesPerPixel = static_cast<OffsetType>(bitsPerPixel / 8);
                const SizeType paddingInBytes = static_cast<SizeType>(bytesPerPixel * m_padding);
                const OffsetType destWidthInBytes = static_cast<OffsetType>(m_rootEntry.m_rect.width) * bytesPerPixel;
                const OffsetType srcWidthInBytes = static_cast<OffsetType>(width)* bytesPerPixel;
                const UInt8* srcData = data;
                const OffsetType destOffsetInBytes = (static_cast<OffsetType>(xPositionInAtlas) * bytesPerPixel) +
                    (static_cast<OffsetType>(yPositionInAtlas)* destWidthInBytes);
                UInt8* dstBuffer = FeatStd::Internal::PointerAdd(bitmapData, destOffsetInBytes);
                for (UInt i = 0; i < height; ++i) {
                    MemoryPlatform::Copy(dstBuffer, srcData, srcWidthInBytes);
                    if (0 < m_padding) {
                        MemoryPlatform::Set(FeatStd::Internal::PointerAdd(dstBuffer, srcWidthInBytes), m_paddingValue, paddingInBytes);
                    }
                    srcData = FeatStd::Internal::PointerAdd(srcData, srcWidthInBytes);
                    dstBuffer = FeatStd::Internal::PointerAdd(dstBuffer, destWidthInBytes);
                }

                for (UInt i = 0; i < m_padding; ++i) {
                    MemoryPlatform::Set(dstBuffer, m_paddingValue, static_cast<SizeType>(srcWidthInBytes) + paddingInBytes);
                    dstBuffer = FeatStd::Internal::PointerAdd(dstBuffer, destWidthInBytes);
                }
            }
        }

        return true;
    }

    return false;
}

bool BitmapAtlas::Remove(UInt xPositionInAtlas, UInt yPositionInAtlas)
{
    return m_rootEntry.Remove(xPositionInAtlas, yPositionInAtlas, /* parent entry */ 0);
}

bool BitmapAtlas::IsEmpty() const
{
    FEATSTD_DEBUG_ASSERT((((0 == m_rootEntry.m_entry0) && (0 == m_rootEntry.m_entry1) && (!m_rootEntry.m_isOccupied)) ||
        (0 != m_rootEntry.m_entry0)));

    return (0 == m_rootEntry.m_entry0);
}

BitmapAtlas::Entry* BitmapAtlas::Entry::Insert(UInt width, UInt height)
{
    if (!IsLeaf()) {
        Entry* newEntry = m_entry0->Insert(width, height);
        if (0 != newEntry) {
            return newEntry;
        }

        return m_entry1->Insert(width, height);
    }

    if (m_isOccupied) {
        return 0;
    }

    if ((width > m_rect.width) || (height > m_rect.height)) {
        return 0;
    }

    if ((width == m_rect.width) && (height == m_rect.height)) {
        return this;
    }

    FEATSTD_DEBUG_ASSERT(0 == m_entry0);
    FEATSTD_DEBUG_ASSERT(0 == m_entry1);

    m_entry0 = CANDERA_NEW(Entry);
    m_entry1 = CANDERA_NEW(Entry);

    TextureSizeType deltaWidth = m_rect.width - FeatStd::Internal::NumericConversion<TextureSizeType>(width);
    TextureSizeType deltaHeight = m_rect.height - FeatStd::Internal::NumericConversion<TextureSizeType>(height);

    if (deltaWidth > deltaHeight) {
        m_entry0->m_rect = Rect(m_rect.left, m_rect.top, width, m_rect.height);
        m_entry1->m_rect = Rect(m_rect.left + width, m_rect.top, m_rect.width - width, m_rect.height);
    }
    else {
        m_entry0->m_rect = Rect(m_rect.left, m_rect.top, m_rect.width, height);
        m_entry1->m_rect = Rect(m_rect.left, m_rect.top + height, m_rect.width, m_rect.height - height);
    }

    return m_entry0->Insert(width, height);
}

bool BitmapAtlas::Entry::Remove(UInt left, UInt top, Entry* parent)
{
    if ((left == m_rect.left) && (top == m_rect.top) && m_isOccupied) {
        m_isOccupied = false;
        if (!IsSubtreeOccupied()) {
            if (0 != m_entry0) {
                CANDERA_DELETE(m_entry0);
                m_entry0 = 0;
                FEATSTD_DEBUG_ASSERT(0 != m_entry1);
                CANDERA_DELETE(m_entry1);
                m_entry1 = 0;
            }
        }

        if (0 != parent) {
            parent->DefragSubtree();
        }

        return true;
    }

    if ((0 != m_entry0) && (m_entry0->Remove(left, top, this))) {
        if (0 != parent) {
            parent->DefragSubtree();
        }

        return true;
    }

    if ((0 != m_entry1) && (m_entry1->Remove(left, top, this))) {
        if (0 != parent) {
            parent->DefragSubtree();
        }

        return true;
    }

    return false;
}

bool BitmapAtlas::Entry::IsSubtreeOccupied() const
{
    if (m_isOccupied) {
        return true;
    }

    if (0 == m_entry0) {
        FEATSTD_DEBUG_ASSERT(0 == m_entry1);
        return false;
    }

    FEATSTD_DEBUG_ASSERT(0 != m_entry1);

    return ((m_entry0->IsSubtreeOccupied()) || (m_entry1->IsSubtreeOccupied()));
}

void BitmapAtlas::Entry::DefragSubtree()
{
    FEATSTD_DEBUG_ASSERT((0 != m_entry0) && (0 != m_entry1));
    bool canBeCleaned = ((!m_entry0->IsSubtreeOccupied()) && (!m_entry1->IsSubtreeOccupied()));
    if (canBeCleaned) {
        CANDERA_DELETE(m_entry0);
        m_entry0 = 0;
        CANDERA_DELETE(m_entry1);
        m_entry1 = 0;
    }
}

}
