//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_BITMAP_H)
#define CANDERA_BITMAP_H

#include <Candera/EngineBase/Common/BaseStringBufferAppenders.h>
#include <Candera/EngineBase/Common/CanderaObject.h>
#include <Candera/EngineBase/Common/ResourceObject.h>
#include <Candera/System/MemoryManagement/Disposer.h>
#include <FeatStd/Util/Optional.h>
#include <Candera/System/Mathematics/Vector2.h>

namespace Candera {
    /** @addtogroup CommonBase
     *  @{
     */

    /**
     * @brief   A 2-dimensional bitmap that can be used as a texture, background or sprite.
     *          A bitmap pointing to a valid next bitmap not null represents a bitmap chain that
     *          can be used e.g. to initialize a MipMap chain (See TextureImage::SetBitmap).
     */
    class Bitmap : public CanderaObject {
        FEATSTD_TYPEDEF_BASE(CanderaObject);
        
        public:
            /**
            * PixelsResource is a ResourceObject that provides access to the bitmap pixels data.
            *
            * To create a resource handle, call PixelsResource::CreateHandle with the following parameters:
            * - data: pointer to the pixels data
            * - disposer: The function which defines the dispose method for the data.
            *    The function will be called on Bitmap::DisposePixles call or on Bitmap destruction.
            * - size: Size of the data in bytes. Note: With compressed images
            *    it is recommended to set the exact size of the pixels array, otherwise
            *    the VRAM size of an uncompressed image would be allocated.
            *
            * @see ResourceObject
            */
            typedef ResourceObject<const UInt8> PixelsResource;

            typedef MemoryManagement::ArrayDisposer<const UInt8*> Disposer;
            typedef PixelsResource::DisposerFunction DisposerFn;

            FEATSTD_TYPEDEF_SHARED_POINTER(Bitmap);

            enum PixelFormat {
                RgbUnsignedShort565PixelFormat = 0,           ///< 16 bit RGB
                RgbaUnsignedShort4444PixelFormat = 1,         ///< 16 bit RGBA
                RgbaUnsignedShort5551PixelFormat = 2,         ///< 16 bit RGBA
                DepthUnsignedShortPixelFormat = 3,            ///< 16 bit depth
                RgbUnsignedBytePixelFormat = 4,               ///< 32 bit RGB (ES3.0: 24 bit)
                RgbaUnsignedBytePixelFormat = 5,              ///< 32 bit RGBA
                AlphaUnsignedBytePixelFormat = 6,             ///< 32 bit Alpha (ES3.0: 8 bit)
                LuminanceUnsignedBytePixelFormat = 7,         ///< 32 bit luminance (ES3.0: 8 bit)
                LuminanceAlphaUnsignedBytePixelFormat = 8,    ///< 32 bit luminance alpha (ES 3.0 16 bit)
                CoverageUnsignedBytePixelFormat = 9,          ///< Extension: GL_NV_coverage_sample 32 bit
                Etc2CompressedRgbPixelFormat = 10,            ///< 4 bpp compressed RGB
                Etc2CompressedSrgbPixelFormat = 11,           ///< 4 bpp compressed RGB with perceptual weighting
                Etc2Alpha1CompressedRgbaPixelFormat = 12,     ///< 4 bpp compressed RGBA with binary Alpha values
                Etc2Alpha1CompressedSrgbaPixelFormat = 13,    ///< 4 bpp compressed RGBA with perceptual weighting of RGB and binary alpha values
                Etc2EacCompressedRgbaPixelFormat = 14,        ///< 8 bpp compressed RGBA
                Etc2EacCompressedSrgbaPixelFormat = 15,       ///< 8 bpp compressed RGBA with perceptual weighting of RGB
                EacCompressedUnsignedRPixelFormat = 16,       ///< 4 bpp compressed single channel unsigned R format
                EacCompressedSignedRPixelFormat = 17,         ///< 4 bpp compressed single channel signed R format (can preserve 0 exactly)
                EacCompressedUnsignedRGPixelFormat = 18,      ///< 8 bpp compressed two channel unsigned RG format
                EacCompressedSignedRGPixelFormat = 19,        ///< 8 bpp compressed two channel signed RG format (can preserve 0 exactly)
                SrgbUnsignedBytePixelFormat = 20,             ///< 24 bit sRGB
                SrgbaUnsignedBytePixelFormat = 21,            ///< 32 bit sRGBA
                RgbFloatingPointR11G11B10PixelFormat = 22,    ///< 32 bit floating point format with 5 bit exponent and 5/6 bit mantissa
                RgbFloatingPoint9E5PixelFormat = 23,          ///< 32 bit floating point format with shared 5 bit exponent and 9 bit mantissa
                RgbaUnsignedIntRev10A2PixelFormat = 24,       ///< 32 bit RGBA with 10 bit per rgb, 2 bit alpha
                Dxt1CompressedRgbFormat = 25,                 ///< 4 bpp compressed RGB
                Dxt1CompressedRgbaFormat = 26,                ///< 4 bpp compressed RGBA
                Dxt3CompressedRgbaFormat = 27,                ///< 8 bpp compressed RGBA
                Dxt5CompressedRgbaFormat = 28,                ///< 8 bpp compressed RGBA
                PixelFormatCount = 29                         /// Values above this are not supported by default. Please keep this value at the end of this enum.
            };

            enum DeprecatedFormat {
                RgbFormat = 0,
                RgbaFormat = 1,
                AlphaFormat = 2,
                LuminanceFormat = 3,
                LuminanceAlphaFormat = 4,
                DepthFormat = 5,
                CoverageFormat = 6          ///<Extension: GL_NV_coverage_sample.
            };

            typedef DeprecatedFormat Format;

            enum DeprecatedType {
                UnsignedByteType = 0,
                UnsignedShortType,
                UnsignedShort565Type,
                UnsignedShort4444Type,
                UnsignedShort5551Type
            };

            typedef DeprecatedType Type;

            enum PackAlignment {
                PackAlignment1 = 1,
                PackAlignment2 = 2,
                PackAlignment4 = 4,
                PackAlignment8 = 8
            };

            /**
            * The struct NinePatchProperties contains the four borders that define the nine patches as properties.
            */
            struct NinePatchProperties {
                NinePatchProperties():
                    left(0),
                    right(0),
                    top(0),
                    bottom(0) {}

                NinePatchProperties(UInt16 x1, UInt16 x2, UInt16 y1, UInt16 y2):
                    left(x1),
                    right(x2),
                    top(y1),
                    bottom(y2) {}
                
                // Left border
                UInt16 left;
                // Right border
                UInt16 right;
                // Upper border
                UInt16 top;
                // Lower border
                UInt16 bottom;

                bool operator== (const NinePatchProperties& rhs) const {
                    return ((left == rhs.left) && (right == rhs.right) && (top == rhs.top) && (bottom == rhs.bottom));
                }
            };

            /**
             *  Constructs an immutable Bitmap by referencing the pixel data given by \p pixels.
             *  @param  width                   Width of Bitmap.
             *  @param  height                  Height of Bitmap.
             *  @param  format                  PixelFormat of Bitmap.
             *  @param  alignment               Bitmap alignment.
             *  @param  pixels                  The actual pixel data.
             *  @param  disposerFn              Disposer function for the pixel data.
             *  @param  size                    The size of the provided pixels array in bytes.
             *                                  Note: With compressed images it is recommended to set the exact size of the pixels array,
             *                                  otherwise the VRAM size of an uncompressed image would be allocated.
             *  @param  isVerticallyFlipped     true, if the lower left corner of the bitmap is specified by the coordinates 0/0.
             *                                  false, if the upper left corner of the bitmap is specified by the coordinates 0/0.
             */
             static SharedPointer Create(UInt16 width,
                                        UInt16 height,
                                        Bitmap::PixelFormat format,
                                        PackAlignment alignment,
                                        const UInt8* pixels,
                                        DisposerFn disposerFn,
                                        UInt32 size = 0,
                bool isVerticallyFlipped = true);

            /**
             *  Constructs a mutable Bitmap by referencing the pixel data given by \p pixels.
             *  @param  width                   Width of Bitmap.
             *  @param  height                  Height of Bitmap.
             *  @param  format                  PixelFormat of Bitmap.
             *  @param  alignment               Bitmap alignment.
             *  @param  pixels                  The actual pixel data.
             *  @param  disposerFn              Disposer function for the pixel data.
             *  @param  size                    The size of the provided pixels array in bytes.
             *                                  Note: With compressed images it is recommended to set the exact size of the pixels array,
             *                                  otherwise the VRAM size of an uncompressed image would be allocated.
             *  @param  isMutable               In some use-cases it is mandatory to set the bitmap immutable while the pixel pointer is non-const.
             *  @param  isVerticallyFlipped     true, if the lower left corner of the bitmap is specified by the coordinates 0/0.
             *                                  false, if the upper left corner of the bitmap is specified by the coordinates 0/0.
             */
             static SharedPointer Create(UInt16 width,
                                        UInt16 height,
                                        Bitmap::PixelFormat format,
                                        PackAlignment alignment,
                                        UInt8* pixels,
                                        DisposerFn disposerFn,
                                        UInt32 size = 0,
                                        bool isMutable = true,
                                        bool isVerticallyFlipped = true);

            /**
             *  Constructs an immutable Bitmap by referencing the pixel data given by \p pixels and with a custom, platform dependent format.
             *  @param  width                   Width of Bitmap.
             *  @param  height                  Height of Bitmap.
             *  @param  format                  PixelFormat of Bitmap, platform dependent.
             *  @param  alignment               Bitmap alignment.
             *  @param  pixels                  The actual pixel data.
             *  @param  size                    The size of the provided pixels array in bytes.
             *                                  Note: With compressed images it is recommended to set the exact size of the pixels array,
             *                                  otherwise the VRAM size of an uncompressed image would be allocated.
             *  @param  disposerFn              Disposer function for the pixel data.
             *  @param  isVerticallyFlipped     true, if the lower left corner of the bitmap is specified by the coordinates 0/0.
             *                                  false, if the upper left corner of the bitmap is specified by the coordinates 0/0.
             */
            static SharedPointer Create(UInt16 width,
                   UInt16 height,
                   Int format,
                   PackAlignment alignment,
                   const UInt8* pixels,
                   DisposerFn disposerFn,
                   UInt32 size = 0,
                   bool isVerticallyFlipped = true);

            /**
             *  Constructs a mutable Bitmap by referencing the pixel data given by \p pixels and with a custom, platform dependent format.
             *  @param  width                   Width of Bitmap.
             *  @param  height                  Height of Bitmap.
             *  @param  format                  PixelFormat of Bitmap, platform dependent.
             *  @param  alignment               Bitmap alignment.
             *  @param  pixels                  The actual pixel data.
             *  @param  disposerFn              Disposer function for the pixel data.
             *  @param  size                    The size of the provided pixels array in bytes.
             *                                  Note: With compressed images it is recommended to set the exact size of the pixels array,
             *                                  otherwise the VRAM size of an uncompressed image would be allocated.
             *  @param  isMutable               In some use-cases it is mandatory to set the bitmap immutable while the pixel pointer is non-const.
             *  @param  isVerticallyFlipped     true, if the lower left corner of the bitmap is specified by the coordinates 0/0.
             *                                  false, if the upper left corner of the bitmap is specified by the coordinates 0/0.
             */
            static SharedPointer Create(UInt16 width,
                UInt16 height,
                Int format,
                PackAlignment alignment,
                UInt8* pixels,
                DisposerFn disposerFn,
                UInt32 size = 0,
                bool isMutable = true,
                bool isVerticallyFlipped = true);


            /**
            *  Constructs a Bitmap by referencing the pixel data given by \p pixelsResourceHandle.
            *  @param  width                   Width of Bitmap.
            *  @param  height                  Height of Bitmap.
            *  @param  format                  PixelFormat of Bitmap.
            *  @param  alignment               Bitmap alignment.
            *  @param  pixelsResourceHandle    ResourceDataHandle that references pixel data.
            *  @param  isVerticallyFlipped     true, if the lower left corner of the bitmap is specified by the coordinates 0/0.
            *                                  false, if the upper left corner of the bitmap is specified by the coordinates 0/0.
            */
            static SharedPointer Create(UInt16 width,
                                        UInt16 height,
                                        Bitmap::PixelFormat format,
                                        PackAlignment alignment,
                                        const ResourceDataHandle& pixelsResourceHandle,
                                        bool isVerticallyFlipped = true);

            /**
            *  Constructs a mutable Bitmap by referencing the pixel data given by \p pixelsResourceHandle and with a custom, platform dependent format.
            *  @param  width                   Width of Bitmap.
            *  @param  height                  Height of Bitmap.
            *  @param  format                  PixelFormat of Bitmap, platform dependent.
            *  @param  alignment               Bitmap alignment.
            *  @param  pixelsResourceHandle    ResourceDataHandle that references pixel data.
            *  @param  isVerticallyFlipped     true, if the lower left corner of the bitmap is specified by the coordinates 0/0.
            *                                  false, if the upper left corner of the bitmap is specified by the coordinates 0/0.
            */
            static SharedPointer Create(UInt16 width,
                                        UInt16 height,
                                        Int format,
                                        PackAlignment alignment,
                                        const ResourceDataHandle& pixelsResourceHandle,
                                        bool isVerticallyFlipped = true);

            /**
             * Destructs this Bitmap object.
             */
            virtual ~Bitmap() override;

            /**
             * Retrieves if this Bitmap's data can be changed or not.
             * @return true if mutable, false otherwise.
             */
            CANDERA_DEPRECATED_3_0_2("Check if bitmap pixels are mutable: GetPixelsResourceHandle().IsMutable()",
                bool IsMutable() const);

            /**
             *  Retrieves if this bitmap is vertically flipped.
             *  @return     true, if the lower left corner of the bitmap is specified by the coordinates 0/0.
             *              false, if the upper left corner of the bitmap is specified by the coordinates 0/0.
             */
            bool IsVerticallyFlipped() const {
                return m_isVerticallyFlipped;
            }

            /**
             * Retrieves this Bitmap's width.
             * @return The Bitmap's width.
             */
            UInt16 GetWidth() const {
                return m_width;
            }

            /**
             * Retrieves this Bitmap's height.
             * @return The Bitmap's height.
             */
            UInt16 GetHeight() const {
                return m_height;
            }

            /**
             * Checks if the bitmap is using a compressed pixel format.
             * @return true if the bitmap's pixel format is compressed, false otherwise.
             */
            bool IsCompressed() const {
                return c_PixelFormatProperties[GetPixelFormat()].isCompressed;
            }

            /**
             * Checks if the pixel format is a compressed pixel format.
             * @param pixelFormat  The pixel format to check.
             * @return True, if the pixel format is compressed. False, otherwise.
             */
            static bool IsCompressed(Bitmap::PixelFormat pixelFormat) {
                return c_PixelFormatProperties[pixelFormat].isCompressed;
            }

            /**
             * Returns the bits per pixel for the given pixel format.
             * @param pixelFormat  The pixel format to return the bits per pixel for.
             * @return The bits per pixel for the given pixel format.
             */
            static UInt GetBitsPerPixel(Bitmap::PixelFormat pixelFormat) {
                return c_PixelFormatProperties[pixelFormat].BitsPerPixel;
            }

            /**
             * Retrieves this Bitmap's pixel format, as defined by the PixelFormat enumeration.
             * PixelFormat combines OpenGL format and type description in order
             * to only provide valid combinations.
             * @return This Bitmap's format.
             */
            Bitmap::PixelFormat GetPixelFormat() const {
                return static_cast<Bitmap::PixelFormat>(m_pixelFormat);
            }

            /**
            * Checks if current format is a custom pixel format, thus not supported by default.
            * @return <em>true</em> if custom pixel format.
            *         <em>false</em> otherwise.
            */
            bool IsCustomPixelFormat() const { return (m_pixelFormat >= static_cast<Int>(PixelFormatCount)); }

            /**
             * Retrieves this Bitmap's custom pixel format.
             * @return This Bitmap's custom format if not supported by default.
             */
            Int GetCustomPixelFormat() const {
                return m_pixelFormat;
            }

            /**
             * Retrieves this Bitmap's pack alignment.
             * @return Pack Alignment of this Bitmap.
             */
            PackAlignment GetPackAlignment() const {
                return static_cast<PackAlignment>(m_packAlignment);
            }

            /**
             * Retrieves pointer to the pixels if the bitmap mutable.
             * @return Pointer to the pixels if the bitmap is mutable. If the
             *         bitmap is immutable, 0 is returned.
             */
            CANDERA_DEPRECATED_3_0_2("Use PixelsResource(GetPixelsResourceHandle()).GetMutableData() instead",
                UInt8* GetPixels());

            /**
             * @return Pointer to the pixels.
             */
            CANDERA_DEPRECATED_3_0_2("Use PixelsResource(GetPixelsResourceHandle()).GetData() instead",
                const UInt8* GetPixels() const);

            /**
            * @return Constant PixelsResource handle.
            * Create a PixelsResource for accessing the constant pixels data.
            */
            const ResourceDataHandle& GetPixelsResourceHandle() const {
                return m_pixelsResourceHandle;
            }

            /**
            * @return Mutable PixelsResource handle.
            * Create a PixelsResource for accessing the mutable/constant pixels data.
            */
            ResourceDataHandle& GetPixelsResourceHandle() {
                return m_pixelsResourceHandle;
            }

            /**
             * Set pointer to next Bitmap object. A bitmap pointing to a valid next bitmap not null represents a bitmap chain that
             * can be used e.g. to initialize a MipMap chain (See TextureImage::SetBitmap).
             * @param bitmap Next Bitmap to set in Bitmap chain.
             */
            void SetNext(SharedPointer bitmap);

            /**
             * Get pointer to next Bitmap object.
             * @return pointer to next Bitmap of bitmap chain, null otherwise.
             */
            const SharedPointer& GetNext() const;

            /**
             * Checks if pointer to next Bitmap object is null or not.
             * @return true if pointer to next Bitmap object is not null, false otherwise.
             */
            bool HasNext() const;

            /**
             * Set the properties of the Bitmap's NinePatch component.
             * @param ninePatchProperties pointer to an object with NinePatch properties
            */
            void SetNinePatchproperties(const FeatStd::Optional<NinePatchProperties>& ninePatchProperties);

            /**
             * Get pointer to the Bitmap's NinePatch component properties.
             @return pointer to the object with NinePatch properties, null otherwise.
            */
            const FeatStd::Optional<NinePatchProperties>& GetNinePatchProperties() const {
                return m_ninePatchProperties;
            }

            /**
             * Checks if Bitmap instance is a NinePatch.
             * @return true if Bitmap is a NinePatch, false otherwise.
            */
            bool IsNinePatch() const;

            void SetCenter(const FeatStd::Optional<Candera::Vector2>& center);

            const FeatStd::Optional<Candera::Vector2>& GetCenter() const {
                return m_center;
            }

            /**
             * Dispose bitmap pixels.
             */
            void DisposePixels();

            /**
             * Returns the size the pixel array will occupy in system RAM. 
             * @return Returns the size the pixel array will occupy in system RAM.
             */
            UInt32 GetSize() const;

            /**
             * Following conditions are checked for mipmap completeness:
             * Bitmap formats are the same for the entire bitmap chain (i.e. all mipmap levels).
             * Bitmap pack alignment is the same for all mipmap levels.
             * All mipmap levels are defined in having the previous dimensions divided by two.
             * Additionally, it is checked that all levels down to 1x1 are specified.
             * @return True, if the conditions are met. False, otherwise.
             */
            bool IsMipMapChainComplete() const;

            static Int MapDeprecatedEnumsToPixelFormat(Int format, Type type); // deprecated!
            static Int MapPixelFormatToDeprecatedFormat(Int format); // deprecated!
            static Type MapPixelFormatToDeprecatedType(Int format); // deprecated!
           
            FEATSTD_SHARED_POINTER_DECLARATION();

        protected:

            /**
             *  Constructs an immutable Bitmap by referencing the pixel data given by \p pixels.
             *  @param  width                   Width of Bitmap.
             *  @param  height                  Height of Bitmap.
             *  @param  format                  PixelFormat of Bitmap.
             *  @param  alignment               Bitmap alignment.
             *  @param  pixels                  The actual pixel data.
             *  @param  disposerFn              Disposer function for the pixel data.
             *  @param  size                    The size of the provided pixels array in bytes.
             *                                  Note: With compressed images it is recommended to set the exact size of the pixels array,
             *                                  otherwise the VRAM size of an uncompressed image would be allocated.
             *  @param  isVerticallyFlipped     true, if the lower left corner of the bitmap is specified by the coordinates 0/0.
             *                                  false, if the upper left corner of the bitmap is specified by the coordinates 0/0.
             */
            Bitmap(UInt16 width,
                   UInt16 height,
                Bitmap::PixelFormat format,
                   PackAlignment alignment,
                   const UInt8* pixels,
                   DisposerFn disposerFn,
                UInt32 size = 0,
                   bool isVerticallyFlipped = true);

            /**
             *  Constructs a mutable Bitmap by referencing the pixel data given by \p pixels.
             *  @param  width                   Width of Bitmap.
             *  @param  height                  Height of Bitmap.
             *  @param  format                  PixelFormat of Bitmap.
             *  @param  alignment               Bitmap alignment.
             *  @param  pixels                  The actual pixel data.
             *  @param  disposerFn              Disposer function for the pixel data.
             *  @param  size                    The size of the provided pixels array in bytes.
             *                                  Note: With compressed images it is recommended to set the exact size of the pixels array,
             *                                  otherwise the VRAM size of an uncompressed image would be allocated.
             *  @param  isMutable               In some use-cases it is mandatory to set the bitmap immutable while the pixel pointer is non-const.
             *  @param  isVerticallyFlipped     true, if the lower left corner of the bitmap is specified by the coordinates 0/0.
             *                                  false, if the upper left corner of the bitmap is specified by the coordinates 0/0.
             */
            Bitmap(UInt16 width,
                   UInt16 height,
                Bitmap::PixelFormat format,
                   PackAlignment alignment,
                   UInt8* pixels,
                   DisposerFn disposerFn,
                UInt32 size = 0,
                   bool isMutable = true,
                   bool isVerticallyFlipped = true);

            /**
             *  Constructs an immutable Bitmap by referencing the pixel data given by \p pixels and with a custom, platform dependent format.
             *  @param  width                   Width of Bitmap.
             *  @param  height                  Height of Bitmap.
             *  @param  format                  PixelFormat of Bitmap, platform dependent.
             *  @param  alignment               Bitmap alignment.
             *  @param  pixels                  The actual pixel data.
             *  @param  size                    The size of the provided pixels array in bytes.
             *                                  Note: With compressed images it is recommended to set the exact size of the pixels array,
             *                                  otherwise the VRAM size of an uncompressed image would be allocated.
             *  @param  disposerFn              Disposer function for the pixel data.
             *  @param  isVerticallyFlipped     true, if the lower left corner of the bitmap is specified by the coordinates 0/0.
             *                                  false, if the upper left corner of the bitmap is specified by the coordinates 0/0.
             */
            Bitmap(UInt16 width,
                   UInt16 height,
                   Int format,
                   PackAlignment alignment,
                   const UInt8* pixels,
                   DisposerFn disposerFn,
                UInt32 size = 0,
                   bool isVerticallyFlipped = true);

            /**
             *  Constructs a mutable Bitmap by referencing the pixel data given by \p pixels and with a custom, platform dependent format.
             *  @param  width                   Width of Bitmap.
             *  @param  height                  Height of Bitmap.
             *  @param  format                  PixelFormat of Bitmap, platform dependent.
             *  @param  alignment               Bitmap alignment.
             *  @param  pixels                  The actual pixel data.
             *  @param  disposerFn              Disposer function for the pixel data.
             *  @param  size                    The size of the provided pixels array in bytes.
             *                                  Note: With compressed images it is recommended to set the exact size of the pixels array,
             *                                  otherwise the VRAM size of an uncompressed image would be allocated.
             *  @param  isMutable               In some use-cases it is mandatory to set the bitmap immutable while the pixel pointer is non-const.
             *  @param  isVerticallyFlipped     true, if the lower left corner of the bitmap is specified by the coordinates 0/0.
             *                                  false, if the upper left corner of the bitmap is specified by the coordinates 0/0.
             */
            Bitmap(UInt16 width,
                   UInt16 height,
                   Int format,
                   PackAlignment alignment,
                   UInt8* pixels,
                   DisposerFn disposerFn,
                UInt32 size = 0,
                   bool isMutable = true,
                   bool isVerticallyFlipped = true);

           /**
            *  Constructs an immutable Bitmap by referencing the pixel data given by \p pixelsResourceHandle.
            *  @param  width                   Width of Bitmap.
            *  @param  height                  Height of Bitmap.
            *  @param  format                  PixelFormat of Bitmap.
            *  @param  alignment               Bitmap alignment.
            *  @param  pixelsResourceHandle    ResourceDataHandle that references pixel data.
            *  @param  isVerticallyFlipped     true, if the lower left corner of the bitmap is specified by the coordinates 0/0.
            *                                  false, if the upper left corner of the bitmap is specified by the coordinates 0/0.
            */
            Bitmap(UInt16 width,
                   UInt16 height,
                   Bitmap::PixelFormat format,
                   PackAlignment alignment,
                   const ResourceDataHandle& pixelsResourceHandle,
                   bool isVerticallyFlipped = true);

           /**
            *  Constructs an immutable Bitmap by referencing the pixel data given by \p pixelsResourceHandle and with a custom, platform dependent format.
            *  @param  width                   Width of Bitmap.
            *  @param  height                  Height of Bitmap.
            *  @param  format                  PixelFormat of Bitmap, platform dependent.
            *  @param  alignment               Bitmap alignment.
            *  @param  pixelsResourceHandle    ResourceDataHandle that references pixel data.
            *  @param  isVerticallyFlipped     true, if the lower left corner of the bitmap is specified by the coordinates 0/0.
            *                                  false, if the upper left corner of the bitmap is specified by the coordinates 0/0.
            */
            Bitmap(UInt16 width,
                   UInt16 height,
                   Int format,
                   PackAlignment alignment,
                   const ResourceDataHandle& pixelsResourceHandle,
                   bool isVerticallyFlipped = true);

        private:
            struct PixelFormatProperties {
                bool isCompressed;
                UInt8 BitsPerPixel;
            };

            static const PixelFormatProperties c_PixelFormatProperties[];

            bool m_isVerticallyFlipped : 1;
            // UInt for enum. According to MISRA 9-6-2 Bit Fields must be explicitly signed or unsigned. MISRA 9-6-3: Bit-fields shall not have enum type.
            UInt m_packAlignment : 8;
            Int m_pixelFormat;

            UInt16 m_width;
            UInt16 m_height;

            SharedPointer m_next;
            ResourceDataHandle m_pixelsResourceHandle;

            FeatStd::Optional<NinePatchProperties> m_ninePatchProperties;
            FeatStd::Optional<Candera::Vector2> m_center;

            FEATSTD_MAKE_CLASS_UNCOPYABLE(Bitmap);
    };

    /** @} */ // end of CommonBase
} // namespace Candera

#endif    // CANDERA_BITMAP_H
